﻿using Bailun.ServiceFabric.Trace;
using Newtonsoft.Json;
using SkyApm;
using SkyApm.Common;
using SkyApm.Diagnostics;
using SkyApm.Logging;
using SkyApm.Tracing;
using SkyApm.Tracing.Segments;
using System;
using System.Linq;
using System.Net.Http;

namespace Bailun.Diagnostics.HttpClient
{
    public class HttpClientTracingDiagnosticProcessor : ITracingDiagnosticProcessor
    {
        public string ListenerName { get; } = "HttpHandlerDiagnosticListener";

        private readonly ILogger _logger;
        private readonly ILoggerFactory _loggerFactory;
        private readonly ITracingContext _tracingContext;
        private readonly IExitSegmentContextAccessor _contextAccessor;

        public HttpClientTracingDiagnosticProcessor(ITracingContext tracingContext,
            IExitSegmentContextAccessor contextAccessor,
            ILoggerFactory loggerFactory)
        {
            _tracingContext = tracingContext;
            _contextAccessor = contextAccessor;

            _logger = loggerFactory.CreateLogger(typeof(HttpClientTracingDiagnosticProcessor));
            _loggerFactory = loggerFactory;
        }

        [DiagnosticName("System.Net.Http.Request")]
        public void HttpRequest([Property(Name = "Request")] HttpRequestMessage request)
        {
            if (TracePathFilter.IsIgnore(request.RequestUri.ToString()))
            {
                return;
            }
            var context = _tracingContext.CreateExitSegmentContext(request.RequestUri.ToString(),
                $"{request.RequestUri.Host}:{request.RequestUri.Port}",
                new HttpClientICarrierHeaderCollection(request));


            context.Span.SpanLayer = SpanLayer.HTTP;
            context.Span.Component = Components.HTTPCLIENT;
            context.Span.AddTag(Tags.URL, request.RequestUri.ToString());
            context.Span.AddTag(Tags.HTTP_METHOD, request.Method.ToString());
            context.Span.AddTag(TagsExtension.HEADERS, JsonConvert.SerializeObject(request.Headers));


            string requestStr = ReadContentAsString(request.Content, true);
            if (!string.IsNullOrEmpty(requestStr))
            {
                context.Span.AddTag(TagsExtension.REQUEST, requestStr);
            }
        }


        [DiagnosticName("System.Net.Http.Response")]
        public void HttpResponse([Property(Name = "Response")] HttpResponseMessage response)
        {
            var context = _contextAccessor.Context;
            if (context == null)
            {
                return;
            }

            if (response != null)
            {
                var statusCode = (int)response.StatusCode;
                if (statusCode >= 400)
                {
                    context.Span.ErrorOccurred();
                }

                context.Span.AddTag(Tags.STATUS_CODE, statusCode);


                string responseStr = ReadContentAsString(response.Content);
                if (!string.IsNullOrEmpty(responseStr))
                {
                    context.Span.AddTag(TagsExtension.RESPONSE, responseStr);
                }
            }
            _tracingContext.Release(context);

        }

        [DiagnosticName("System.Net.Http.Exception")]
        public void HttpException([Property(Name = "Request")] HttpRequestMessage request,
            [Property(Name = "Exception")] Exception exception)
        {
            _contextAccessor.Context?.Span?.ErrorOccurred(exception);
        }


        private string ReadContentAsString(HttpContent content, bool isRequest = false)
        {
            try
            {
                if (content == null)
                {
                    return string.Empty;
                }
                var includeMediaTypes = new string[] { MediaTypeNames.Application.Json, MediaTypeNames.Application.Xml };
                var mediaType = content.Headers?.ContentType?.MediaType;
                if (includeMediaTypes.Contains(mediaType))
                {
                    return content.ReadAsStringAsync().ConfigureAwait(false).GetAwaiter().GetResult();
                }
                else
                {
                    return string.Empty;
                }
            }
            catch (Exception ex)
            {
                _logger.Error($"获取{(isRequest ? "请求" : "响应")}报文异常", ex);
                return string.Empty;
            }
        }
    }
}