﻿using Bailun.Diagnostics.HttpClient;
using Bailun.ServiceFabric.Trace;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.DependencyInjection.Extensions;
using Serilog;
using SkyApm;
using SkyApm.Agent.GeneralHost;
using System;

namespace Microsoft.Extensions.Hosting
{
    public static class GenericHostBuilderExtensions
    {
        /// <summary>
        /// 日志配置
        /// </summary>
        /// <param name="builder"></param>
        /// <param name="writeToExceptionless">是否写到Exceptionless</param>
        /// <param name="writeToConsole">是否写到控制台</param>
        /// <param name="configureLogger">扩展配置</param>
        /// <returns></returns>
        public static IHostBuilder UseBailunTrace(this IHostBuilder builder, bool writeToExceptionless = false, bool writeToConsole = false, Action<LoggerConfiguration> configureLogger = null)
        {
            return builder.UseBailunSerilog(writeToExceptionless, writeToConsole, configureLogger)
                .AddSkyAPM()
                .ConfigureServices((hostCtx, services) =>
                {
                    services.Replace(ServiceDescriptor.Singleton<ITracingDiagnosticProcessor, HttpClientTracingDiagnosticProcessor>());
                });
        }

        /// <summary>
        /// 日志配置
        /// </summary>
        /// <param name="builder"></param>
        /// <param name="writeToExceptionless">是否写到Exceptionless</param>
        /// <param name="writeToConsole">是否写到控制台</param>
        /// <param name="configureLogger">扩展配置</param>
        /// <returns></returns>
        public static IHostBuilder UseBailunSerilog(this IHostBuilder builder, bool writeToExceptionless = false, bool writeToConsole = false, Action<LoggerConfiguration> configureLogger = null)
        {

            Action<HostBuilderContext, LoggerConfiguration> configure = (hostingContext, loggerConfiguration) =>
            {
                loggerConfiguration
                    .ReadFrom.Configuration(hostingContext.Configuration)
                    .Enrich.FromLogContext();

                if (writeToExceptionless)
                {
                    var exceptionlessSection = hostingContext.Configuration.GetSection("Exceptionless");
                    var apiKey = exceptionlessSection.GetValue<string>("ApiKey");
                    var serverUrl = exceptionlessSection.GetValue<string>("ServerUrl");

                    loggerConfiguration.WriteTo.Exceptionless(apiKey, serverUrl);
                }

                if (hostingContext.HostingEnvironment.IsDevelopment())
                {
                    loggerConfiguration.WriteTo.Debug();
                }

                if (writeToConsole)
                {
                    loggerConfiguration
                        .WriteTo
                        .Console(outputTemplate: "[{Timestamp:HH:mm:ss} {Level:u3}] {TraceId:l} {Message:lj} {NewLine}{Exception}");
                }
                configureLogger?.Invoke(loggerConfiguration);
            };

            return builder.UseSerilog(configure);
        }
    }
}
