﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Net;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Abp.Extensions;
using Abp.Logging;
using Bailun.Core.Extension.Json;
using Bailun.Core.Extension.Utils;
using Bailun.ServiceFabric.Core.ApiWidgets;
using Bailun.ServiceFabric.Extension;
using Newtonsoft.Json;
using Qiniu.Http;
using Qiniu.Storage;
using Qiniu.Util;
using HttpResult = Qiniu.Http.HttpResult;

namespace Bailun.Core.Core
{
    /// <summary>
    /// 七牛云辅助类
    /// </summary>
    public class QiNiuCloudHelper
    {
        public static readonly string AccessKey = ConfigManagerConf.GetValue("QiNiu:AccessKey");
        public static readonly string SecretKey = ConfigManagerConf.GetValue("QiNiu:SecretKey");

        public static readonly string PicServerUrl = ConfigManagerConf.GetValue("QiNiu:CdnUrl");

        public static readonly string ProxyUrl = ConfigManagerConf.GetValue("QiNiu:ProxyUrl");
        /// <summary>
        /// 刊登cdn地址
        /// </summary>
        public static readonly string PublishCdnUrl = ConfigManagerConf.GetValue("QiNiu:PublishCdnUrl");
        /// <summary>
        /// 七牛存储节点
        /// </summary>
        private static readonly string Bucket = "bailun-Publish-img";

        #region 上传文件
        /// <summary>
        /// 上传文件
        /// </summary>
        /// <param name="saveFile"></param>
        /// <param name="content"></param>
        /// <returns></returns>
        public static async Task<HttpResult> UploadStreamAsync(string saveFile, byte[] content)
        {
            Mac mac = new Mac(AccessKey, SecretKey);

            // 设置上传策略，详见：https://developer.qiniu.com/kodo/manual/1206/put-policy
            PutPolicy putPolicy = new PutPolicy();
            putPolicy.Scope = Bucket + ":" + saveFile;  //覆盖式更新

            // putPolicy.SetExpires(3600);
            string token = Auth.CreateUploadToken(mac, putPolicy.ToJsonString());
            Qiniu.Storage.Config config = new Qiniu.Storage.Config();
            // 设置上传区域
            config.Zone = Zone.ZoneCnSouth;
            // 设置 http 或者 https 上传
            //config.UseHttps = true;
            //config.UseCdnDomains = true;
            //config.ChunkSize = ChunkUnit.U512K;
            FormUploader target = new FormUploader(config);
            var result = await target.UploadStream(new MemoryStream(content), saveFile, token, null);
            return result;
        }
        #endregion

        /// <summary>
        /// 是否本系统图片
        /// </summary>
        /// <param name="url">当前url</param>
        /// <returns></returns>
        public static bool IsSystemImage(string url)
        {


            var systemImgDomain = ConfigManagerConf.GetReferenceValue("QiNiu:SystemDomain");
            foreach (var imgDomain in systemImgDomain)
            {
                if (url.Contains(imgDomain))
                {
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// 根据图片路径获得系统支持的格式的所有路径，比如http://img.blsct.com/xxx,jpg得到["http://img.blsct.com/xxx,jpg","http://clound.cn/xxx,jpg"]
        /// </summary>
        /// <param name="imgUrl"></param>
        /// <returns></returns>
        public static List<string> GetMultipleImgPath(string imgUrl)
        {
            var systemImgDomain = ConfigManagerConf.GetReferenceValue("QiNiu:SystemDomain");
            string domain = "";
            foreach (var imgDomain in systemImgDomain)
            {
                if (imgUrl.Contains(imgDomain))
                {
                    domain = imgDomain;
                }
            }
            List<string> imgList = new List<string>();
            foreach (var imgDomain in systemImgDomain)
            {
                imgList.Add(imgUrl.Replace(domain, imgDomain));
            }
            return imgList;
        }

        #region 获取七牛云缩略图，默认为550宽高

        /// <summary>
        /// 获取七牛云等比缩放缩略图，默认为550宽高，可用于放大的配置
        /// </summary>
        /// <param name="url">七牛云图片路径(clouddn.com格式）</param>
        /// <param name="width">宽度</param>
        /// <param name="height">高度</param>
        /// <param name="quantity">图片质量，0-100</param>
        /// <returns></returns>
        public static string GetQiNiuScaleThumbnail(string url, int width = 550, int height = 550, int quantity = 75)
        {
            var imgFunc = $"imageMogr2/thumbnail/!{width}x{height}r/blur/1x0/quality/{quantity}";
            if (url.Contains("clouddn.com") && !url.Contains("imageMogr2") && width > 0 && height > 0)  //代表七牛云图片路径
            {
                return url + "?" + imgFunc;
            }
            return url;
        }
        #endregion

        /// <summary>
        /// 获取缩略图
        /// </summary>
        /// <param name="url">七牛云图片路径(clouddn.com格式）</param>
        /// <param name="width">宽度</param>
        /// <param name="height">高度</param>
        /// <returns></returns>
        public static string GetQiNiuThumbnail(string url, int width = 100, int height = 100)
        {
            var imgFunc = $"imageView2/2/w/{width}/h/{height}/q/75|imageslim";
            if (url.Contains("clouddn.com") && !url.Contains("imageView2") && width > 0 && height > 0)  //代表七牛云图片路径
            {
                return url + "?" + imgFunc;
            }
            return url;
        }

        #region 查找文件路径
        /// <summary>
        /// 查找文件路径
        /// </summary>
        /// <param name="saveFile"></param>
        /// <returns></returns>
        public static string CreatePublishUrl(string saveFile)
        {
            string url = "";
            try
            {
                url = DownloadManager.CreatePublishUrl(PicServerUrl, saveFile);
            }
            catch (Exception ex)
            {
                string error = ex.ToString();
            }
            return url;
        }
        #endregion

        #region 复制文件

        /// <summary>
        /// 复制文件
        /// </summary>
        /// <param name="saveFile"></param>
        /// <param name="descFile"></param>
        /// <returns></returns>
        public static async Task<string> CopyFileAsync(string saveFile, string descFile)
        {
            // 公网可访问的文件链接
            Mac mac = new Mac(AccessKey, SecretKey);
            Config config = new Config
            {
                // 设置上传区域
                Zone = Zone.ZoneCnSouth
            };
            var buckerManager = new BucketManager(mac, config);

            string op = buckerManager.CopyOp(Bucket, saveFile, Bucket, descFile, true);
            List<string> ops = new List<string>
            {
                op
            };
            BatchResult ret = await buckerManager.Batch(ops);
            if (ret.Code / 100 != 2)
            {
            }
            foreach (BatchInfo info in ret.Result)
            {
                if (info.Code != (int)HttpCode.OK)
                {
                }

            }

            return descFile;
        }


        /// <summary>
        /// 复制文件
        /// </summary>
        /// <param name="files"></param>
        /// <returns></returns>
        public static async Task<ApiResult> CopyFileAsync(Dictionary<string, string> files)
        {
            // 公网可访问的文件链接
            Mac mac = new Mac(AccessKey, SecretKey);
            Config config = new Config
            {
                // 设置上传区域
                Zone = Zone.ZoneCnSouth
            };
            var buckerManager = new BucketManager(mac, config);
            List<string> ops = new List<string>();
            foreach (var key in files)
            {
                var fileName = key.Key.Replace(PicServerUrl + "/", "");
                string op = buckerManager.CopyOp(Bucket, fileName, Bucket, key.Value, true);
                ops.Add(op);
            }

            BatchResult ret = await buckerManager.Batch(ops);
            var result = new ApiResult();
            if (ret.Code / 100 != 2)
            {
                return new ApiResult() { StatusCode = ret.Code, Message = "上传文件出错！" + ret.Error };
            }
            var errorMsg = "上传文件错误信息：";
            foreach (BatchInfo info in ret.Result)
            {
                if (info.Code != (int)HttpCode.OK)
                {

                    errorMsg += info.Data.Error + ",";
                    result.StatusCode = info.Code;
                }

            }
            result.Message = errorMsg;

            return result;
        }


        #endregion

        #region 七牛云拉取文件
        /// <summary>
        /// 七牛云拉取文件
        /// </summary>
        /// <param name="url"></param>
        /// <param name="saveFile"></param>
        /// <returns></returns>
        public static async Task<FetchResult> FetchFileAsync(string url, string saveFile)
        {
            // 公网可访问的文件链接
            Mac mac = new Mac(AccessKey, SecretKey);
            Config config = new Config
            {
                // 设置上传区域
                Zone = Zone.ZoneCnSouth
            };
            var buckerManager = new BucketManager(mac, config);
            return await buckerManager.Fetch(url, Bucket, saveFile);
        }

        /// <summary>
        /// 七牛云拉取文件
        /// </summary>
        /// <param name="files"></param>
        /// <returns></returns>
        public static async Task<ApiResult<Dictionary<string, bool>>> FetchFilesAsync(Dictionary<string, string> files)
        {
            var errorFetchResult = new Dictionary<string, bool>();

            // 公网可访问的文件链接
            Mac mac = new Mac(AccessKey, SecretKey);
            Config config = new Config
            {
                // 设置上传区域
                Zone = Zone.ZoneCnSouth
            };
            var buckerManager = new BucketManager(mac, config);
            var result = new ApiResult<Dictionary<string, bool>>();
            var errorMsg = "异步fectch文件错误信息：";
            foreach (var key in files)
            {
                try
                {
                    #region 上传文件

                    var remoteUrl = key.Key.Replace("\\", "/").Replace("httpss", "https"); //
                    if (!key.Key.Contains("http")) //不是正确的路径则跳过
                    {
                        errorFetchResult.Add(key.Key, false);
                        continue;
                    }
                    if (key.Key.Contains("http://i.ebayimg.com"))
                    {
                        remoteUrl = key.Key.Replace("http://i.ebayimg.com", "https://i.ebayimg.com");
                    }
                    HttpResult fetchResult = null;

                    if (remoteUrl.Contains("https"))
                    {

                        var bigImgUrl = ReplaceEbayPicUrl(remoteUrl);

                        fetchResult = await FetchHttpsFileAsync(bigImgUrl, key.Value);

                        if (fetchResult.Code != 200)
                        {
                            fetchResult = await FetchHttpsFileAsync(remoteUrl, key.Value);
                        }
                        else if (fetchResult.Code == 200 && key.Key.Contains("i.ebayimg.com"))
                        {
                            var url = CreatePublishUrl(key.Value);
                            var file = GetQiNiuFileInfo(url);
                            if (file.Width == 80 && file.Height == 80)   //如果宽高=80，则需要重新替换
                            {
                                bigImgUrl = ReplaceEbayPicUrl(remoteUrl, true);

                                fetchResult = await FetchHttpsFileAsync(bigImgUrl, key.Value);
                            }
                        }

                    }
                    else
                    {
                        try
                        {
                            var bigImgUrl = ReplaceEbayPicUrl(remoteUrl);
                            fetchResult = await buckerManager.Fetch(bigImgUrl, Bucket, key.Value);
                            if (fetchResult.Code != 200)
                            {

                                fetchResult = await buckerManager.Fetch(remoteUrl, Bucket, key.Value);
                            }


                        }
                        catch (Exception ex) //如果fetch文件出错！则直接下载
                        {

                            var bigImgUrl = ReplaceEbayPicUrl(remoteUrl);
                            fetchResult = await FetchHttpsFileAsync(bigImgUrl, key.Value);

                            if (fetchResult.Code != 200)
                            {

                                fetchResult = await FetchHttpsFileAsync(remoteUrl, key.Value);
                            }
                            else if (fetchResult.Code == 200 && key.Key.Contains("i.ebayimg.com"))
                            {
                                var url = QiNiuCloudHelper.CreatePublishUrl(key.Value);
                                var file = GetQiNiuFileInfo(url);
                                if (file.Width == 80 && file.Height == 80)   //如果宽高=80，则需要重新替换
                                {
                                    bigImgUrl = ReplaceEbayPicUrl(remoteUrl, true);

                                    fetchResult = await FetchHttpsFileAsync(bigImgUrl, key.Value);
                                }
                            }


                        }
                    }

                    #endregion

                    if (fetchResult.Code != 200)
                    {
                        errorFetchResult.Add(key.Key, false);
                        errorMsg += fetchResult.Text + ",";
                        result.StatusCode = fetchResult.Code;
                    }
                }
                catch (Exception ex)
                {
                    errorFetchResult.Add(key.Key, false);
                    errorMsg += ex.Message + ",";
                    result.StatusCode = 400;
                    LogHelper.LogException(ex);
                }
            }
            result.Message = errorMsg;
            result.Result = errorFetchResult;
            return result;
        }

        /// <summary>
        /// 上传https文件
        /// </summary>
        /// <param name="url"></param>
        /// <param name="saveFile"></param>
        /// <returns></returns>
        private static async Task<HttpResult> FetchHttpsFileAsync(string url, string saveFile)
        {
            try
            {
                var bytes = DownloadHttpsFile(url.Trim());

                return await UploadStreamAsync(saveFile, bytes);

            }
            catch (Exception ex)
            {
                LogHelper.LogException(ex);
                return new HttpResult() { Code = 500, Text = "下载https文件失败!" + url };
            }
        }

        #region 替换小图

        /// <summary>
        /// 替换小图
        /// </summary>
        /// <param name="picUrl"></param>
        /// <param name="isError">是否产生过一次80x80的图片，如果产生过80x80的图片，则按500x500爬取</param>
        /// <returns></returns>
        public static string ReplaceEbayPicUrl(string picUrl, bool isError = false)
        {
            if (!picUrl.IsNullOrEmpty())
            {
                if (picUrl.Contains("i.ebayimg.com/00/"))
                {
                    string regexCode = "https://i.ebayimg.com/00/s/(.*)/z/(.*)/(.*)";
                    var reg = new Regex(regexCode);
                    var mc = reg.Match(picUrl);
                    if (mc.Success)
                    {
                        var str = mc.Groups[2].Value;
                        var newUrl = "https://i.ebayimg.com/images/g/" + str + "/s-l800.jpg";
                        if (isError)
                        {
                            newUrl = "https://i.ebayimg.com/images/g/" + str + "/s-l500.jpg";
                        }
                        return newUrl;
                    }

                    return picUrl;



                }
                else if (picUrl.Contains("ebayimg.com"))
                {
                    if (!isError)
                    {
                        var bigImgUrl = picUrl.Replace("s-l300", "s-l800");
                        bigImgUrl = bigImgUrl.Replace("s-l500", "s-l800");
                        return bigImgUrl;
                    }
                    else
                    {
                        return picUrl.Replace("s-l300", "s-l500");
                    }


                }




            }
            return picUrl;
        }
        #endregion

        #region 获取图片的大小和尺寸
        /// <summary>  
        /// 获取图片的大小和尺寸  
        /// </summary>  
        /// <param name="aPhotoUrl">图片url</param>  
        /// <param name="iWidth">图片宽度</param>  
        /// <param name="iHeight">图片高度</param>  
        private void GetPhotoInfo(string aPhotoUrl, ref int iWidth, ref int iHeight)
        {
            Uri mUri = new Uri(aPhotoUrl);
            HttpWebRequest mRequest = (HttpWebRequest)WebRequest.Create(mUri);
            mRequest.Method = "GET";
            mRequest.Timeout = 200;
            HttpWebResponse mResponse = (HttpWebResponse)mRequest.GetResponse();
            Stream mStream = mResponse.GetResponseStream();
            Image mImage = Image.FromStream(mStream);

            iWidth = mImage.Width;
            iHeight = mImage.Height;
            mStream.Close();

        }
        #endregion

        /// <summary>  
        /// 下载https文件
        /// </summary>  
        /// <param name="posturl">请求的接口链接</param>  
        /// <returns>返回的结果</returns>  
        ///   
        #region 
        public static byte[] DownloadHttpsFile(string posturl)
        {
            try
            {
                WebClient client = new WebClient();
                byte[] bytes = client.DownloadData(posturl);

                return bytes;
            }
            catch (Exception ex)
            {
                try
                {
                    return DownloadByProxy(posturl);
                }
                catch (Exception other)
                {
                    throw other;
                }
            }
        }
        /// <summary>
        /// 通过代理下载图片
        /// </summary>
        /// <param name="postUrl"></param>
        /// <returns></returns>
        public static byte[] DownloadByProxy(string postUrl)
        {

            var downloadUrl = ProxyUrl + "/" + postUrl;
            WebClient client = new WebClient();
            byte[] bytes = client.DownloadData(downloadUrl);

            return bytes;
        }
        #endregion

        #region 七牛云图片信息
        /// <summary>
        /// 七牛云图片信息
        /// </summary>
        /// <param name="url"></param>
        /// <returns></returns>
        public static QiNiuFileInfo GetQiNiuFileInfo(string url)
        {
            if (url.Contains("clouddn.com"))
            {
                var result = HttpClientManager.HttpPost(url + "?imageInfo");
                if (!result.IsNullOrEmpty())
                {
                    return JsonHelper.FromJson<QiNiuFileInfo>(result);
                }
            }
            return new QiNiuFileInfo();
        }
        #endregion

        #endregion
    }
    /// <summary>
    /// 七牛云图片格式
    /// </summary>
    public class QiNiuFileInfo
    {
        [JsonProperty("size")]
        public long Size { get; set; }
        [JsonProperty("format")]
        public string Format { get; set; }
        [JsonProperty("width")]
        public int Width { get; set; }
        [JsonProperty("height")]
        public int Height { get; set; }
        [JsonProperty("colorModel")]
        public string ColorModel { get; set; }
    }
}
