﻿using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Abp.Application.Services;
using Abp.Domain.Entities;
using Bailun.Core.Core;
using Bailun.Core.Extension.Page;
using Microsoft.EntityFrameworkCore;

namespace Bailun.Core.Application
{


    public class BaseApplicationService<TEntity> : ApplicationService, IBaseApplicationService<TEntity> where TEntity : class, IEntity<int>
    {
        private readonly IBaseRepository<TEntity, int> _baseRepository;
        public BaseApplicationService(IBaseRepository<TEntity, int> baseRepository)
        {
            _baseRepository = baseRepository;
            LocalizationSourceName = ProductConsts.ConnectionStringName;
        }

        #region CURD
        /// <summary>
        /// 查询
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public async Task<TEntity> GetAsync(int id)
        {
            return await _baseRepository.Query().AsNoTracking().FirstOrDefaultAsync(x => x.Id == id);
        }

        /// <summary>
        /// 新增
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public async Task<TEntity> InsertAsync(TEntity value)
        {
            return await _baseRepository.InsertAsync(value);
        }

        /// <summary>
        /// 更新
        /// </summary>
        /// <param name="id"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public async Task UpdateAsync(int id, TEntity value)
        {
            value.Id = id;
            await _baseRepository.InsertOrUpdateAsync(value);
        }

        /// <summary>
        /// 删除
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public async Task DeleteAsync(int id)
        {
            await _baseRepository.DeleteAsync(id);
        }

        /// <summary>
        /// 列表
        /// </summary>
        /// <returns></returns>
        public async Task<IList<TEntity>> ListAsync()
        {
            return await _baseRepository.GetAllListAsync();
        }

        /// <summary>
        /// 批量新增
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public async Task BatchInsertAsync(IList<TEntity> values)
        {
            foreach (var value in values)
            {
                await _baseRepository.InsertAsync(value);
            }
        }

        /// <summary>
        /// 批量删除
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public async Task BatchDeleteAsync(int[] ids)
        {
            await _baseRepository.DeleteAsync(x => ids.Contains(x.Id));
        }

        /// <summary>
        /// 分页查询
        /// </summary>
        /// <param name="page"></param>
        /// <returns></returns>
        public async Task<PageResult<List<TEntity>>> PageAsync(PageRequest page)
        {
            var data = await _baseRepository.GetPageListAsync(x => BuildQueryCondition(x), page);
            return new PageResult<List<TEntity>>().ToPageResult(page.PageIndex, page.Total, data);
        }

        /// <summary>
        /// 创建条件
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        private IQueryable<TEntity> BuildQueryCondition(IQueryable<TEntity> query)
        {
            return query = query.Where(x => 1 == 1);
        }

        #endregion
    }
}