﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Threading.Tasks;
using Abp.Domain.Entities;
using Abp.EntityFrameworkCore;
using Abp.EntityFrameworkCore.Repositories;
using Abp.Extensions;
using AutoMapper.QueryableExtensions;
using Bailun.Core.Extension.Page;
using Microsoft.EntityFrameworkCore;
using Z.BulkOperations;

namespace Bailun.Core.Extension.Domain
{
    /// <summary>
    /// 版 本 1.0
    /// Copyright (c) 2016-2021 广州tianzh
    /// 创建人：tianzh
    /// 日 期：2016/12/28 12:00:46
    /// 描 述：
    /// </summary>
    public class RepositoryExtensions<TDbContext, TEntity, TPrimaryKey> : EfCoreRepositoryBase<TDbContext, TEntity, TPrimaryKey>,
        IRepositoryExtension<TEntity, TPrimaryKey>
        where TDbContext : DbContext
        where TEntity : class, IEntity<TPrimaryKey>, new()
    {
        public RepositoryExtensions(IDbContextProvider<TDbContext> dbContextProvider) : base(dbContextProvider)
        {
        }
       
     
        /// <summary>
        /// 分页
        /// </summary>
        /// <typeparam name="TResult"></typeparam>
        /// <param name="filterAction"></param>
        /// <param name="pagination"></param>
        /// <returns></returns>
        public List<TEntity> GetPageList(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, PageRequest pagination)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            if (filterAction != null)
                query = filterAction(query);
            pagination.Total = query.Count();
            var pageIndex = pagination.PageIndex;
            var pageSize = pagination.PageNumber;
            if (pageIndex == 0)
            {
                pageIndex = 1;
            }
            pagination.SortField = pagination.SortField.IsNullOrWhiteSpace() ? "CreationTime" : pagination.SortField;
            query = OrderBy(query, pagination);
            return query.Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ToList();
        }
        /// <summary>
        /// 分页
        /// </summary>
        /// <typeparam name="TResult"></typeparam>
        /// <param name="filterAction"></param>
        /// <param name="pagination"></param>
        /// <returns></returns>
        public async Task<List<TEntity>> GetPageListAsync(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, PageRequest pagination)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");  
            if (filterAction != null)
                query = filterAction(query);
            pagination.Total = await query.CountAsync();
            var pageIndex = pagination.PageIndex;
            var pageSize = pagination.PageNumber;
            if (pageIndex == 0)
            {
                pageIndex = 1;
            }
            pagination.SortField = pagination.SortField.IsNullOrWhiteSpace() ? "Id" : pagination.SortField;
            query = OrderBy(query, pagination);
            return await query.AsNoTracking().Skip((pageIndex - 1) * pageSize).Take(pageSize).ToListAsync();
        }

        /// <summary>
        /// 分页、关联查询指定字段对象
        /// </summary>
        /// <param name="includeExpression"></param>
        /// <param name="filterAction"></param>
        /// <param name="pagination"></param>
        /// <returns></returns>
        public List<TEntity> GetPageListByInclude(Expression<Func<TEntity, object>> includeExpression, Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, PageRequest pagination)
        {
            if (includeExpression == null)
                throw new Exception($"{typeof(TEntity)}对象includeExpression为空！");
            var query = GetAllIncluding(includeExpression);
            if (query == null)
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            if (filterAction != null)
                query = filterAction(query);
            pagination.Total = query.Count();
            var pageIndex = pagination.PageIndex;
            var pageSize = pagination.PageNumber;
            if (pageIndex == 0)
            {
                pageIndex = 1;
            }
            pagination.SortField = pagination.SortField.IsNullOrWhiteSpace() ? "CreationTime" : pagination.SortField;
            query = OrderBy(query, pagination);
            return query.Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ToList();
        }

        /// <summary>
        /// 分页、关联查询指定字段对象
        /// </summary>
        /// <param name="filterAction"></param>
        /// <param name="pagination"></param>
        /// <param name="includeExpression"></param>
        /// <returns></returns>
        public async Task<List<TEntity>> GetPageListByIncludeAsync(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, PageRequest pagination, Expression<Func<TEntity, object>>[] includeExpression)
        {
            if (includeExpression == null)
                throw new Exception($"{typeof(TEntity)}对象includeExpression为空！");
            var query = GetAllIncluding(includeExpression);
            if (query == null)
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            if (filterAction != null)
                query = filterAction(query);
            pagination.Total = await query.CountAsync();
            var pageIndex = pagination.PageIndex;
            var pageSize = pagination.PageNumber;
            if (pageIndex == 0)
            {
                pageIndex = 1;
            }
            pagination.SortField = pagination.SortField.IsNullOrWhiteSpace() ? "CreationTime" : pagination.SortField;
            query = OrderBy(query, pagination);

            return await query.Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ToListAsync();
        }

        private IQueryable<TEntity> OrderBy(IQueryable<TEntity> source, PageRequest pagination)
        {
            if (!pagination.SortField.IsNullOrWhiteSpace())
            {
                var param = Expression.Parameter(typeof(TEntity), "p");
                var property = typeof(TEntity).GetProperty(pagination.SortField);
                var propertyAccessExpression = Expression.MakeMemberAccess(param, property);
                var le = Expression.Lambda(propertyAccessExpression, param);
                var type = typeof(TEntity);
                var orderByStr = pagination.Sort == "asc" ? "OrderBy" : "OrderByDescending";
                var resultExp = Expression.Call(typeof(Queryable), orderByStr, new Type[] { type, property.PropertyType }, source.Expression, Expression.Quote(le));
                return source.Provider.CreateQuery<TEntity>(resultExp);
            }
            return source;
        }
     


        public List<TResult> FindList<TResult>(Expression<Func<TEntity, bool>> condition = null)
        {
            if (condition == null)
            {
                return GetDbContext().Set<TEntity>().ProjectTo<TResult>().ToList();
            }
            return GetDbContext().Set<TEntity>().Where(condition).ProjectTo<TResult>().ToList();
        }

        public async Task<List<TResult>> FindListAsync<TResult>(Expression<Func<TEntity, bool>> condition = null)
        {
            if (condition == null)
            {
                return await GetDbContext().Set<TEntity>().ProjectTo<TResult>().ToListAsync();
            }
            return await GetDbContext().Set<TEntity>().Where(condition).ProjectTo<TResult>().ToListAsync();
        }
        public  List<TResult> FindList<TResult>(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            if (filterAction != null)
                query = filterAction(query);
            return query.AsNoTracking().ProjectTo<TResult>().ToList();
        }
        public async Task<List<TResult>> FindListAsync<TResult>(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            if (filterAction != null)
                query = filterAction(query);
            return await query.AsNoTracking().ProjectTo<TResult>().ToListAsync();
        }

        /// <summary>
        /// 查询单个指定字段的对象
        /// </summary>
        /// <typeparam name="TResult"></typeparam>
        /// <param name="condition"></param>
        /// <returns></returns>
        public TResult Find<TResult>(Expression<Func<TEntity, bool>> condition)
        {
            return GetDbContext().Set<TEntity>().Where(condition).ProjectTo<TResult>().ToList().FirstOrDefault();
        }

        /// <summary>
        /// 查询单个指定字段的对象
        /// </summary>
        /// <typeparam name="TResult"></typeparam>
        /// <param name="condition"></param>
        /// <returns></returns>
        public async Task<TResult> FindAsync<TResult>(Expression<Func<TEntity, bool>> condition)
        {
            return (await GetDbContext().Set<TEntity>().Where(condition).ProjectTo<TResult>().ToListAsync()).FirstOrDefault();
        }
        /// <summary>
        /// 新增实体列表
        /// </summary>
        /// <param name="entities">新增实体列表</param>
        /// <param name="isCommit">是否提交保存到数据库</param>
        public bool Add(TEntity[] entities, bool isCommit = false)
        {
            entities.ToList().ForEach(m =>
            {
                var obj = Insert(m);
            });
            if (isCommit)
            {
                return Context.SaveChanges() > 0;
            }
            return true;
            //var result =  GetDbContext().SaveChanges();
            //return result > 0;
        }

        /// <summary>
        /// 批量新增实体列表
        /// </summary>
        /// <param name="entities">新增实体列表</param>
        /// <param name="isCommit">是否提交保存到数据库</param>
        public async Task<bool> AddAsync(TEntity[] entities, bool isCommit = false)
        {
            entities.ToList().ForEach(m =>
            {
                var obj = Insert(m);
            });
            if (isCommit)
            {
                return (await Context.SaveChangesAsync()) > 0;
            }
            return true;
            //var result =  GetDbContext().SaveChanges();
            //return result > 0;
        }

        /// <summary>
        /// 新增实体
        /// </summary>
        /// <param name="entities"></param>
        /// <returns></returns>
        /// <param name="isCommit">是否提交保存到数据库</param>
        public  bool Add(TEntity entities, bool isCommit = false)
        {
            Insert(entities);
            if (isCommit)
            {
                return Context.SaveChanges() > 0;
            }
            return true;
        }

        /// <summary>
        /// 新增实体
        /// </summary>
        /// <param name="entities"></param>
        /// <returns></returns>
        /// <param name="isCommit">是否提交保存到数据库</param>
        public async Task<bool> AddAsync(TEntity entities, bool isCommit = false)
        {
            Insert(entities);
            if (isCommit)
            {
                return (await Context.SaveChangesAsync()) > 0;
            }
            return true;
        }

        #region 查询并更新
        /// <summary>
        /// 查询并更新
        /// </summary>
        /// <param name="condition"></param>
        /// <param name="updateExpression"></param>
        public void UpdateFromQuery(Expression<Func<TEntity, bool>> condition, Expression<Func<TEntity, TEntity>> updateExpression)
        {
            var context = GetDbContext();
            context.Set<TEntity>().Where(condition).UpdateFromQuery(updateExpression);
        }

        /// <summary>
        /// 异步 查询并更新
        /// </summary>
        /// <param name="condition"></param>
        /// <param name="updateExpression"></param>
        /// <returns></returns>
        public async Task UpdateFormQueryAsync(Expression<Func<TEntity, bool>> condition,
            Expression<Func<TEntity, TEntity>> updateExpression)
        {
            var context = GetDbContext();
           await context.Set<TEntity>().Where(condition).UpdateFromQueryAsync(updateExpression);
        }
        #endregion

        /// <summary>
        /// BulkInsert方式批量写入实体列表(自动提交事务)
        /// </summary>
        /// <param name="entities">新增实体列表</param>
        public void BulkInsert(TEntity[] entities)
        {
            var context = GetDbContext();
            context.BulkInsert(entities);
        }
        
        /// <summary>
        /// BulkInsert方式批量写入实体列表(自动提交事务)
        /// </summary>
        /// <param name="entities">新增实体列表</param>
        public async Task BulkInsertAsync(TEntity[] entities)
        {
            var context = GetDbContext();
            await context.BulkInsertAsync(entities);
        }

        /// <summary>
        /// BulkInsert方式批量写入实体列表(自动提交事务)
        /// </summary>
        /// <param name="entities">新增实体列表</param>
        public async Task BulkInsertIncludeGraphAsync(TEntity[] entities)
        {
            var context = GetDbContext();
            await context.BulkInsertAsync(entities, options => options.IncludeGraph = true);
        }

        /// <summary>
        /// BulkUpdate方式批量更新实体列表(自动提交事务)
        /// </summary>
        /// <param name="entities">实体列表</param>
        public void BulkUpdate(TEntity[] entities, Action<BulkOperation<TEntity>> bulkOperationFactory)
        {
            var context = GetDbContext();
            context.BulkUpdate(entities, bulkOperationFactory);
        }
        /// <summary>
        /// 根据表达式直接删除
        /// </summary>
        /// <typeparam name="TEntity"></typeparam>
        /// <param name="condition"></param>
        public void RemoveDirect(Expression<Func<TEntity, bool>> condition)
        {
            var context = GetDbContext();
            context.Set<TEntity>().Where(condition).DeleteFromQuery();
        }
        /// <summary>
        /// 根据表达式直接删除
        /// </summary>
        /// <typeparam name="TEntity"></typeparam>
        /// <param name="condition"></param>
        public async Task RemoveDirectAsync(Expression<Func<TEntity, bool>> condition)
        {
            var context = GetDbContext();
           await context.Set<TEntity>().Where(condition).DeleteFromQueryAsync();
        }

        /// <summary>
        /// 直接更新，不先从数据库查询出实体对象
        /// </summary>
        /// <param name="entity"></param>
        /// <param name="updatedProperties">表达式：m => new { m.SellerName, m.SellerId }</param>
        public void UpdateDirect(TEntity entity, Expression<Func<TEntity, object>> updatedProperties = null, bool isCommit = false)
        {
            //AttachIfNot(entity);
            if (!Table.Local.Contains(entity))
            {
                Table.Attach(entity);
            }
            if (updatedProperties != null)
            {
                var propertyInfoList = updatedProperties.Compile()(entity).GetType().GetProperties();
                //if (propertyInfoList.Any())
                foreach (var propertyInfo in propertyInfoList)
                {
                    Context.Entry(entity).Property(propertyInfo.Name).IsModified = true;
                }   
            }
            else
            {
                //var dbEntityEntry = Context.Entry(entity);
                //foreach (var property in dbEntityEntry.OriginalValues.PropertyNames)
                //{
                //    var original = dbEntityEntry.OriginalValues.GetValue<object>(property);
                //    var current = dbEntityEntry.CurrentValues.GetValue<object>(property);
                //    if (original != null && !original.Equals(current))
                //    {
                //        dbEntityEntry.Property(property).IsModified = true;
                //    }
                //}
                Context.Entry(entity).State = EntityState.Modified;
            }
            if (isCommit)
            {
                Context.SaveChanges();
            }
        }
        /// <summary>
        /// 直接更新，不先从数据库查询出实体对象
        /// </summary>
        /// <param name="entity"></param>
        /// <param name="updatedProperties">表达式：m => new { m.SellerName, m.SellerId }</param>
        public async Task UpdateDirectAsync(TEntity entity, Expression<Func<TEntity, object>> updatedProperties = null, bool isCommit = false)
        {
            //AttachIfNot(entity);
            if (!Table.Local.Contains(entity))
            {
                Table.Attach(entity);
            }
            if (updatedProperties != null)
            {
                var propertyInfoList = updatedProperties.Compile()(entity).GetType().GetProperties();
                //if (propertyInfoList.Any())
                foreach (var propertyInfo in propertyInfoList)
                {
                    Context.Entry(entity).Property(propertyInfo.Name).IsModified = true;
                }
            }
            else
            {
                Context.Entry(entity).State = EntityState.Modified;
            }
            if (isCommit)
            {
               await Context.SaveChangesAsync();
            }
        }

        /// <summary>
        /// 直接批量更新，不先从数据库查询出实体对象
        /// </summary>
        /// <param name="entitys"></param>
        /// <param name="updatedProperties">表达式：m => new { m.SellerName, m.SellerId }</param>
        public void UpdateDirect(TEntity[] entitys, Expression<Func<TEntity, object>> updatedProperties = null, bool isCommit = false)
        {
            foreach (var entity in entitys)
            {
                if (entity != null)
                    UpdateDirect(entity, updatedProperties, isCommit);
            }
        }

        /// <summary>
        /// 直接批量更新，不先从数据库查询出实体对象
        /// </summary>
        /// <param name="entitys"></param>
        /// <param name="updatedProperties">表达式：m => new { m.SellerName, m.SellerId }</param>
        public async Task UpdateDirectAsync(TEntity[] entitys, Expression<Func<TEntity, object>> updatedProperties = null, bool isCommit = false)
        {
            foreach (var entity in entitys)
            {
                if (entity != null)
                   await UpdateDirectAsync(entity, updatedProperties, isCommit);
            }
        }

        #region 旧版本程序兼容

        /// <summary>
        /// 兼容旧版本程序
        /// </summary>
        /// <returns></returns>
        public IQueryable<TEntity> Query()
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            return query;
        } 

        #region 分页查询方法

        public List<TEntity> GetPageList(int pageSize, int pageIndex, out long total,
         Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            query = filterAction(query);
            total = query.Count();
            if (pageIndex == 0)
            {
                pageIndex = 1;
                
            }


            return query.Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ToList();
        }
    

        public List<TEntity> GetPageList<TOrderKey>(int pageSize, int pageIndex, out long total,
            Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, bool isAsc, Expression<Func<TEntity, TOrderKey>> orderByLambda)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            query = filterAction(query);
            total = query.Count();
            if (pageIndex == 0)
            {
                pageIndex = 1;
            }
            if (isAsc)
            {
                return query.OrderBy(orderByLambda).Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ToList();
            }
            return query.OrderByDescending(orderByLambda).Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ToList();
        }
        public List<TResult> GetPageList<TOrderKey, TResult>(int pageSize, int pageIndex, out long total,
         Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, bool isAsc, Expression<Func<TEntity, TOrderKey>> orderByLambda)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            query = filterAction(query);
            total = query.Count();
            if (pageIndex == 0)
            {
                pageIndex = 1;
            }
            if (isAsc)
            {
                return query.OrderBy(orderByLambda).Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ProjectTo<TResult>().ToList();
            }
            return query.OrderByDescending(orderByLambda).Skip((pageIndex - 1) * pageSize).Take(pageSize).AsNoTracking().ProjectTo<TResult>().ToList();
        }

      
        /// <summary>
        /// 多条件查询，不分页
        /// </summary>
        /// <typeparam name="TOrderKey"></typeparam>
        /// <param name="filterAction"></param>
        /// <param name="isAsc"></param>
        /// <param name="orderByLambda"></param>
        /// <returns></returns>
        public List<TEntity> GetPageListNoPage<TOrderKey>(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, bool isAsc, Expression<Func<TEntity, TOrderKey>> orderByLambda)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            query = filterAction(query);
            if (isAsc)
            {
                return query.OrderBy(orderByLambda).ToList();
            }
            return query.OrderByDescending(orderByLambda).ToList();
        }
        /// <summary>
        /// 多条件查询，不分页
        /// </summary>
        /// <typeparam name="TOrderKey"></typeparam>
        /// <param name="filterAction"></param>
        /// <param name="isAsc"></param>
        /// <param name="orderByLambda"></param>
        /// <returns></returns>
        public async Task<List<TEntity>> GetPageListNoPageAsync<TOrderKey>(Func<IQueryable<TEntity>, IQueryable<TEntity>> filterAction, bool isAsc, Expression<Func<TEntity, TOrderKey>> orderByLambda)
        {
            var context = GetDbContext();
            if (!(context.Set<TEntity>() is IQueryable<TEntity> query))
                throw new Exception($"{typeof(TEntity)}对象query为空！");
            query = filterAction(query);
            if (isAsc)
            {
                return await query.OrderBy(orderByLambda).ToListAsync();
            }
            return await query.OrderByDescending(orderByLambda).ToListAsync();
        }

        #endregion

        #region 更新

        /// <summary>
        /// 更新实体
        /// </summary>
        /// <param name="entity">实体实例</param>
        /// <param name="isCommit">是否提交更改到数据库</param>
        public virtual void Update(TEntity entity, bool isCommit = false)
        {
            Table.Attach(entity);
            Context.Entry(entity).State = EntityState.Modified;


            if (isCommit)
            {
                Context.SaveChanges();
            }
        }
        /// <summary>
        /// 批量更新
        /// </summary>
        /// <param name="entities"></param>
        /// <param name="isCommit">是否提交更改到数据库</param>
        public virtual void Update(TEntity[] entities, bool isCommit = false)
        {
            Array.ForEach(entities, m => Update(m));
            //await Task.Factory.StartNew(() => _repository.Update(entitys));
            if (isCommit)
            {
                Context.SaveChanges();
            }
        }

        /// <summary>
        /// 异步更新实体
        /// </summary>
        /// <param name="entity">实体实例</param>
        /// <param name="isCommit">是否提交更改到数据库</param>
        public virtual async Task UpdateAsync(TEntity entity, bool isCommit = false)
        {
            Update(entity);
            if (isCommit)
            {
                await Context.SaveChangesAsync();
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <param name="updateAction"></param>
        /// <param name="isCommit">是否提交更改到数据库</param>
        public virtual void Update(TPrimaryKey id, Action<TEntity> updateAction, bool isCommit = false)
        {
            var entity = FirstOrDefault(id);
            updateAction(entity);
            Update(entity);
            if (isCommit)
            {
                Context.SaveChanges();
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <param name="updateAction"></param>
        /// <param name="isCommit">是否提交更改到数据库</param>
        /// <returns></returns>
        public virtual async Task UpdateAsync(TPrimaryKey id, Action<TEntity> updateAction, bool isCommit = false)
        {
            var entity = await FirstOrDefaultAsync(id);
            updateAction(entity);
            await UpdateAsync(entity, isCommit);
        }

        /// <summary>
        /// 批量更新
        /// </summary>
        /// <param name="entities"></param>
        /// <returns></returns>
        /// <param name="isCommit">是否提交更改到数据库</param>
        public virtual async Task UpdateAsync(TEntity[] entities, bool isCommit = false)
        {
            Array.ForEach(entities, m=> Update(m));
            
            //await Task.Factory.StartNew(() => _repository.Update(entitys));
            if (isCommit)
            {
              await  Context.SaveChangesAsync();
            }
        }

        /// <summary>
        /// 批量更新
        /// </summary>
        /// <param name="entitys"></param>
        /// <param name="updateAction"></param>
        /// <param name="isCommit"></param>
        /// <returns></returns>
        public virtual async Task UpdateAsync(TEntity[] entitys, Action<TEntity> updateAction, bool isCommit = false)
        {
          
            Array.ForEach(entitys, m => updateAction(m));
            await Task.Factory.StartNew(() => Update(entitys, isCommit));
        }

        public virtual bool UpdateCommit(TEntity entity, params Expression<Func<TEntity, object>>[] updatedProperties)
        {
            Update(entity, updatedProperties);
            return Context.SaveChanges() > 0;
        }
        public virtual async Task<bool> UpdateCommitAsync(TEntity entity, params Expression<Func<TEntity, object>>[] updatedProperties)
        {
            Update(entity, updatedProperties);
            return (await Context.SaveChangesAsync()) > 0;
        }
        public virtual void Update(TEntity entity, params Expression<Func<TEntity, object>>[] updatedProperties)
        {
            if (!Table.Local.Contains(entity))
            {
                Table.Attach(entity);
            }

            if (updatedProperties.Any())
            {
                var dbEntityEntry = Context.Entry(entity);
                foreach (var property in updatedProperties)
                {
                    dbEntityEntry.Property(property).IsModified = true;
                }
            }
            else
            {
                Context.Entry(entity).State = EntityState.Modified;
            }

        }
        
        public virtual void Update(TEntity[] entities, params Expression<Func<TEntity, object>>[] updatedProperties)
        {
            foreach (var entity in entities)
            {
                Update(entity, updatedProperties);
            }
        }

        public virtual bool UpdateCommit(TEntity[] entities, params Expression<Func<TEntity, object>>[] updatedProperties)
        {
            Update(entities, updatedProperties);
          return  Context.SaveChanges()>0;
        }

        public virtual async Task UpdateCommitAsync(TEntity[] entities, params Expression<Func<TEntity, object>>[] updatedProperties)
        {
            Update(entities, updatedProperties);
            await Context.SaveChangesAsync();
        }

        #endregion


        #endregion
    }
}
