﻿using Domain;
using Domain.Domain.TakeStock;
using Domain.Domain.TakeStock.Repository;
using Hangfire;
using IService.TakeStock;
using MessageQueue;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;
using static Domain.Domain.TakeStock.TakeStockEnum;

namespace Service.TakeStock
{
    public class TakeStockService : ITakeStockService
    {
        private readonly RabbitMQClient mqClient;
        private readonly ITakeStockScheduleRepository scheduleRepository;
        private readonly IRepository<TakeStockOrder> orderRepository;
        private readonly IRepository<TakeStockOrderLog> logRepository;
        private readonly IHttpClientFactory httpClientFactory;
        private readonly int delay = 3;

        public TakeStockService(IHttpClientFactory httpClientFactory, RabbitMQClient mqClient, ITakeStockScheduleRepository scheduleRepository, IRepository<TakeStockOrder> orderRepository, IRepository<TakeStockOrderLog> logRepository)
        {
            this.httpClientFactory = httpClientFactory;
            this.mqClient = mqClient;
            this.scheduleRepository = scheduleRepository;
            this.orderRepository = orderRepository;
            this.logRepository = logRepository;
        }

        /// <summary>
        /// 创建盘点记录
        /// </summary>
        /// <param name="schedule"></param>
        /// <returns></returns>
        public async Task<bool> CreateTakeStockOrderAsync(TakeStockSchedule schedule, List<TakeStockOrder> orders)
        {
            return await Task.Run(() =>
            {
                int scheduleId = scheduleRepository.CreateOrder(schedule, orders);
                if (scheduleId > 0)
                    BackgroundJob.Enqueue(() => ReTry(scheduleId));
                return scheduleId > 0;
            });
        }

        /// <summary>
        /// 根据计划Id添加盘点作业
        /// </summary>
        /// <param name="scheduleId"></param>
        /// <returns></returns>
        [DisplayName("添加盘点作业, 盘点计划Id:{0}")]
        public async Task<bool> ReTry(int scheduleId)
        {
            var orders = orderRepository.GetAllList(o => !o.IsDeleted && o.State == TSOrderState.创建 && o.ScheduleId == scheduleId);
            bool isSuccess = false;
            try
            {
                foreach (var order in orders)
                {
                    BackgroundJob.Enqueue(() => FreezeStockAsync(order.Id));
                }
                isSuccess = true;
            }
            catch (Exception ex)
            {
                isSuccess = false;
                BackgroundJob.Schedule(() => ReTry(scheduleId), DateTimeOffset.UtcNow.AddMinutes(delay));
            }
            return isSuccess;
        }

        private async Task<bool> AddOrUpdateLog(int orderId, TSOrderState state, string content, string jsonData = null)
        {
            var log = await logRepository.FirstOrDefaultAsync(l => l.OrderId == orderId && l.State == state);
            if (log == null)
            {
                log = new TakeStockOrderLog();
                log.OrderId = orderId;
                log.State = state;
            }
            else
                log.LastModificationTime = DateTime.Now;

            log.Content = content;
            if (!string.IsNullOrWhiteSpace(jsonData))
                log.JsonData = jsonData;
            if (log.Id > 0)
                return await logRepository.UpdateAsync(log, "Content", "JsonData", "LastModificationTime") > 0;
            else
                return logRepository.InsertFulfill(log);
        }

        /// <summary>
        /// 冻结库存
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [DisplayName("冻结库存, 盘点单Id:{0}")]
        [Queue("freezestock")]
        public async Task<bool> FreezeStockAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.冻结库存)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                //client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/FreezeStock", content);

                bool isFreeze = false;
                if (response.IsSuccessStatusCode)
                {
                    string responseContent = await response.Content.ReadAsStringAsync();
                    var resultObj = JObject.Parse(responseContent)["data"];
                    isFreeze = resultObj["isFreeze"].ToObject<bool>();
                    Task<bool> logTask = null;
                    if (isFreeze)
                    {
                        order.State = TSOrderState.冻结库存;
                        order.BeforeQuantity = resultObj["quantity"].ToObject<int>();
                        order.LastModificationTime = DateTime.Now;

                        logTask = AddOrUpdateLog(id, TSOrderState.冻结库存, "冻结库存成功。");
                        isSuccess = true;
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        order.LastModificationTime = DateTime.Now;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "冻结库存失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }

                    int row = await orderRepository.UpdateAsync(order, "State", "BeforeQuantity", "LastModificationTime");
                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => RollbackStockAsync(id));
            else
                BackgroundJob.Schedule(() => FreezeStockAsync(id), DateTimeOffset.UtcNow.AddMinutes(delay));

            return isSuccess;
        }

        /// <summary>
        /// 回滚库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        [DisplayName("回滚库存, 盘点单Id:{0}")]
        [Queue("rollbackstock")]
        public async Task<bool> RollbackStockAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.释放库存)
                isSuccess = true;
            else
            {
                var schedule = scheduleRepository.Get(order.ScheduleId);
                var client = httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku, IsAutomation = schedule.IsAutomation } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("TakeStockScheduleService/RollbackStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    var jobj = JObject.Parse(responseContent)["data"];
                    isSuccess = jobj["result"].ToObject<bool>();
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.释放库存;
                        order.LastModificationTime = DateTime.Now;

                        logTask = AddOrUpdateLog(id, TSOrderState.释放库存, "已释放占用库存。");
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        order.LastModificationTime = DateTime.Now;

                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "释放占用库存失败，WMS返回消息：" + jobj["message"], responseContent);
                    }

                    var cancelCodes = jobj["cancelCodes"].ToString();
                    var waitCodes = jobj["waitCodes"].ToString();
                    if (!cancelCodes.Equals("[]"))
                    {
                        order.CancelCodes += "," + cancelCodes.Trim('[', ']');
                        order.CancelCodes = order.CancelCodes.TrimStart(',');
                    }
                    if (!waitCodes.Equals("[]"))
                    {
                        order.State = TSOrderState.冻结库存;
                        order.WaitCodes = waitCodes.Trim('[', ']');
                    }

                    int row = await orderRepository.UpdateAsync(order, "State", "CancelCodes", "WaitCodes", "LastModificationTime");

                    if (!(row > 0 && await logTask)) isSuccess = false;
                }
            }

            if (isSuccess)
            {
                var schedule = scheduleRepository.Get(order.ScheduleId);
                if (schedule.IsAutomation)
                    BackgroundJob.Enqueue(() => SyncAndEnabledAsync(id));
            }
            else
                BackgroundJob.Schedule(() => RollbackStockAsync(id), DateTimeOffset.UtcNow.AddMinutes(delay));

            return isSuccess;
        }

        /// <summary>
        /// 同步和启用库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        [DisplayName("同步并启用库存, 盘点单Id:{0}")]
        [Queue("syncandenabledstock")]
        public async Task<bool> SyncAndEnabledAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.完成)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/SyncAndEnabledStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    var resultObj = JObject.Parse(responseContent)["data"];
                    isSuccess = resultObj["isSuccess"].ToObject<bool>();
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.完成;
                        order.AfterQuantity = resultObj["quantity"].ToObject<int>();
                        logTask = AddOrUpdateLog(id, TSOrderState.完成, "同步并启用库存，盘点完成。", responseContent);
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }
                    int row = await orderRepository.UpdateAsync(order, "State", "AfterQuantity", "LastModificationTime");
                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => CheckComplete(order.ScheduleId));
            else
                BackgroundJob.Schedule(() => SyncAndEnabledAsync(id), DateTimeOffset.UtcNow.AddMinutes(delay));

            return isSuccess;
        }

        /// <summary>
        /// 检查是否完成计划
        /// </summary>
        /// <param name="scheduleId"></param>
        /// <returns></returns>
        [DisplayName("检查计划是否完成, 盘点计划Id:{0}")]
        [Queue("checkcomplete")]
        public async Task CheckComplete(int scheduleId)
        {
            bool isComplete = orderRepository.Count(o => o.ScheduleId == scheduleId && o.State != TSOrderState.完成 && o.State != TSOrderState.取消) <= 0;
            if (isComplete)
            {
                var schedule = scheduleRepository.Get(scheduleId);
                schedule.State = TakeStockEnum.TSScheduleState.完成;
                schedule.LastModificationTime = DateTime.Now;
                int row = await scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime");

                //发起报溢报损
                //BackgroundJob.Enqueue(() => GainLoss(schedule.Id));
            }
        }

        /// <summary>
        /// 发起线下盘点
        /// </summary>
        /// <param name="order"></param>
        /// <returns></returns>
        public async Task<bool> StartTakeStock(int id, int beforeQuantity)
        {
            var order = orderRepository.Get(id);
            order.State = TSOrderState.盘点中;
            order.BeforeQuantity = beforeQuantity;
            order.LastModificationTime = DateTime.Now;
            var record = await orderRepository.UpdateAsync(order, "State", "BeforeQuantity", "LastModificationTime");
            var logTask = await AddOrUpdateLog(id, TSOrderState.盘点中, "发起线下盘点。");
            return record > 0 || logTask;
        }

        /// <summary>
        /// 盘点反馈
        /// </summary>
        /// <param name="order"></param>
        /// <returns></returns>
        public async Task<bool> Feedback(int id, int afterQuantity, string description)
        {
            var order = orderRepository.Get(id);
            order.AfterQuantity = afterQuantity;
            order.LastModificationTime = DateTime.Now;
            order.Description = description;
            var record = await orderRepository.UpdateAsync(order, "AfterQuantity", "LastModificationTime", "Description");
            var logTask = await AddOrUpdateLog(id, TSOrderState.盘点中, "盘点反馈。");
            bool isSuccess = record > 0 && logTask;
            if (isSuccess)
                BackgroundJob.Enqueue(() => ModifAndEnabledAsync(order.Id));

            return isSuccess;
        }

        /// <summary>
        /// 盘点反馈
        /// </summary>
        /// <param name="order"></param>
        /// <returns></returns>
        public async Task<bool> SaveDescription(int id, string description)
        {
            var order = orderRepository.Get(id);
            order.LastModificationTime = DateTime.Now;
            order.Description = description;
            var record = await orderRepository.UpdateAsync(order, "LastModificationTime", "Description");
            var logTask = await AddOrUpdateLog(id, order.State, "保存盘点描述。");
            bool isSuccess = record > 0 && logTask;
            if (isSuccess)
                BackgroundJob.Enqueue(() => ModifAndEnabledAsync(order.Id));

            return isSuccess;
        }

        /// <summary>
        /// 同步和启用库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        [DisplayName("修改并启用库存, 盘点单Id:{0}")]
        [Queue("modifandenabledstock")]
        public async Task<bool> ModifAndEnabledAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.完成)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku, Quantity = order.AfterQuantity } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/ModifAndEnabled", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    isSuccess = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.完成;
                        logTask = AddOrUpdateLog(id, TSOrderState.完成, "更新为盘点数并启用库存，盘点完成。");
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }
                    int row = await orderRepository.UpdateAsync(order, "State", "LastModificationTime");

                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => Audit(order.ScheduleId));
            else
                BackgroundJob.Schedule(() => ModifAndEnabledAsync(id), DateTimeOffset.UtcNow.AddMinutes(delay));

            return isSuccess;
        }

        /// <summary>
        /// 盘点审核
        /// 线下盘点完成
        /// </summary>
        /// <returns></returns>
        public async Task<bool> Audit(int scheduleId)
        {
            var schedule = scheduleRepository.Get(scheduleId);
            bool isSuccess = false;
            if (schedule.State == TSScheduleState.完成 || schedule.State == TSScheduleState.取消)
                isSuccess = true;
            else
            {
                var orders = orderRepository.GetAllList(o => o.ScheduleId == scheduleId);
                int count = orders.Count(o => o.State != TSOrderState.完成 && o.State != TSOrderState.取消);
                if (count <= 0)
                {
                    bool isAudit = orders.Any(o => o.BeforeQuantity != o.AfterQuantity);
                    if (isAudit)
                    {
                        schedule.State = TSScheduleState.审核;
                    }
                    else
                    {
                        schedule.State = TSScheduleState.完成;
                    }

                    schedule.LastModificationTime = DateTime.UtcNow;
                    int record = await scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime");

                    isSuccess = record > 0;
                    if (isAudit)
                        BackgroundJob.Enqueue(() => GainLoss(schedule.Id));
                }
                else
                    isSuccess = true;
            }

            return isSuccess;
        }

        /// <summary>
        /// 报溢报损
        /// </summary>
        /// <param name="scheduleId"></param>
        [DisplayName("报溢报损, 盘点计划Id:{0}")]
        [Queue("modifandenabledstock")]
        public void GainLoss(int scheduleId)
        {
            var schedule = scheduleRepository.Get(scheduleId);
            var orders = orderRepository.GetAllList(o => o.ScheduleId == scheduleId && o.BeforeQuantity != o.AfterQuantity);
            //报损
            var orders2 = orders.Where(o => o.BeforeQuantity > o.AfterQuantity);
            if (orders2?.Count() > 0)
            {
                var data = new
                {
                    Type = 1,
                    ExternalCode = schedule.Code,
                    WarehouseCode = schedule.WarehouseCode,
                    Details = orders.Where(o => o.BeforeQuantity > o.AfterQuantity).Select(o => new
                    {
                        Sku = o.Sku,
                        Quantity = o.BeforeQuantity - o.AfterQuantity,
                        Description = o.Description
                    })
                };
                AddGainLoss(data);
            }
            orders2 = orders.Where(o => o.BeforeQuantity < o.AfterQuantity);
            if (orders2.Count() > 0)
            {
                //报溢
                var data = new
                {
                    Type = 0,
                    ExternalCode = schedule.Code,
                    WarehouseCode = schedule.WarehouseCode,
                    Details = orders.Where(o => o.BeforeQuantity < o.AfterQuantity).Select(o => new
                    {
                        Sku = o.Sku,
                        Quantity = o.AfterQuantity - o.BeforeQuantity,
                        Description = o.Description
                    })
                };
                AddGainLoss(data);
            }
        }

        [DisplayName("远程调用报溢报损，")]
        [Queue("modifandenabledstock")]
        public async Task<string> AddGainLoss(object obj)
        {
            string code = string.Empty;
            var client = httpClientFactory.CreateClient("WMS");
            client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
            string dataStr = JsonConvert.SerializeObject(new { Data = obj });
            HttpContent content = new StringContent(dataStr);
            content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
            var response = await client.PostAsync("GainLossOrderService/AddGainLoss", content);
            if (response.IsSuccessStatusCode)
            {
                var responseContent = await response.Content.ReadAsStringAsync();
                code = JObject.Parse(responseContent)["data"].ToString();
            }

            if (string.IsNullOrWhiteSpace(code))
                BackgroundJob.Schedule(() => AddGainLoss(obj), DateTimeOffset.UtcNow.AddMinutes(delay));

            return code;
        }

        /// <summary>
        /// 取消盘点单
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [DisplayName("取消盘点, 盘点单Id:{0}")]
        [Queue("canceltakestock")]
        public async Task<bool> CancelOrder(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.完成 || order.State == TSOrderState.取消)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string dataStr = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(dataStr);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/EnabledStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    isSuccess = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.取消;
                        logTask = AddOrUpdateLog(id, TSOrderState.取消, "取消盘点。");
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "取消盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }
                    int row = await orderRepository.UpdateAsync(order, "State", "LastModificationTime");

                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => CancelSchedule(order.ScheduleId));

            return isSuccess;
        }

        /// <summary>
        /// 取消盘点计划
        /// </summary>
        /// <param name="scheduleId"></param>
        /// <returns></returns>
        [DisplayName("取消盘点, 盘点计划Id:{0}")]
        [Queue("canceltakestock")]
        public async Task CancelSchedule(int scheduleId)
        {
            var schedule = scheduleRepository.Get(scheduleId);
            bool isSuccess = false;
            if (schedule.State == TSScheduleState.完成 || schedule.State == TSScheduleState.取消)
                isSuccess = true;
            var orders = orderRepository.GetAllList(o => o.ScheduleId == scheduleId);
            if (orders.All(o => o.State == TSOrderState.完成 || o.State == TSOrderState.取消))
            {
                if (orders.Any(o => o.State == TSOrderState.完成))
                    schedule.State = TSScheduleState.完成;
                else
                    schedule.State = TSScheduleState.取消;

                schedule.LastModificationTime = DateTime.UtcNow;
                int record = await scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime");
                isSuccess = record > 0;
            }
            else
                isSuccess = true;

            if (!isSuccess)
                BackgroundJob.Schedule(() => CancelSchedule(scheduleId), DateTimeOffset.UtcNow.AddMinutes(delay));
        }


        public (int total, List<TakeStockSchedule> items) SearchScheduleByPage(int pageIndex, int pageSize, Expression<Func<TakeStockSchedule, bool>> expr)
        {
            int totalCount = 0;
            var data = scheduleRepository.PageList(pageIndex, pageSize, expr, out totalCount);
            return (totalCount, data);
        }

        public (int total, List<TakeStockOrder> items) SearchOrderByPage(int pageIndex, int pageSize, Expression<Func<TakeStockOrder, bool>> expr)
        {
            int totalCount = 0;
            var data = scheduleRepository.PageList(pageIndex, pageSize, expr, out totalCount);
            return (totalCount, data);
        }

        public List<TakeStockOrderLog> SearchOrderLogs(Expression<Func<TakeStockOrderLog, bool>> expr)
        {
            var data = logRepository.GetAllList(expr);
            return data;
        }
    }
}
