﻿using Domain;
using Domain.Domain.TakeStock;
using Hangfire;
using IService.TakeStock;
using MessageQueue;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;

namespace Service.TakeStock
{
    public class TakeStockService : ITakeStockService
    {
        private readonly RabbitMQClient mqClient;
        private readonly IRepository<CheckRecord> checkRecordRepository;
        private readonly IRepository<CheckLog> checkLogRepository;
        private readonly IHttpClientFactory httpClientFactory;

        public TakeStockService(RabbitMQClient mqClient, IRepository<CheckRecord> checkRecordRepository, IRepository<CheckLog> checkLogRepository, IHttpClientFactory httpClientFactory)
        {
            this.mqClient = mqClient;
            this.checkRecordRepository = checkRecordRepository;
            this.checkLogRepository = checkLogRepository;
            this.httpClientFactory = httpClientFactory;
        }

        /// <summary>
        /// 创建盘点记录
        /// </summary>
        /// <param name="record"></param>
        /// <returns></returns>
        public async Task<bool> CreateCheckRecordAsync(CheckRecord record)
        {
            return await Task.Run(() =>
            {
                bool isSuccess = false;
                int id = checkRecordRepository.InsertAndGetId(record);
                if (id > 0)
                {
                    BackgroundJob.Enqueue(() => FreezeStockAsync(id));
                    isSuccess = true;
                }
                return isSuccess;
            });
        }

        /// <summary>
        /// 冻结库存
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public async Task<bool> FreezeStockAsync(int id)
        {
            var resord = checkRecordRepository.Get(id);
            var client = httpClientFactory.CreateClient("WMS");
            client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
            string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = resord.WarehouseCode, Sku = resord.Sku } });
            HttpContent content = new StringContent(data);
            content.Headers.ContentType = new MediaTypeHeaderValue("application/json");

            var response = await client.PostAsync("WMSStockService/FreezeStock", content);

            bool isFreeze = false;
            bool isSuccess = false;
            string dataStr = await response.Content.ReadAsStringAsync();
            isFreeze = Convert.ToBoolean(JObject.Parse(dataStr)["data"]);
            CheckLog log = new CheckLog();
            log.CheckRecordId = id;

            if (isFreeze)
            {
                resord.TheCurrentLink = TakeStockEnum.Link.冻结库存;
                resord.LastModifcationTime = DateTime.Now;

                log.Link = TakeStockEnum.Link.冻结库存;
                log.Description = "冻结库存成功。";
                isSuccess = true;
            }
            else
            {
                resord.TheCurrentLink = TakeStockEnum.Link.异常;
                log.Link = TakeStockEnum.Link.异常;
                log.Description = "冻结库存失败，WMS返回消息：" + JObject.Parse(dataStr)["message"];
            }

            int row = await checkRecordRepository.UpdateAsync(resord, "TheCurrentLink");
            log = await checkLogRepository.InsertAsync(log);
            if (!(row > 0 && log.Id > 0))
                isSuccess = false;
            if (isSuccess)
                BackgroundJob.Enqueue(() => RollbackStockAsync(id));
            //await mqClient.SendAsync("takeStock.", resord).ContinueWith(task =>
            //{
            //    isSuccess = task.Result;
            //});
            else
                BackgroundJob.Schedule(() => FreezeStockAsync(id), DateTimeOffset.UtcNow.AddMinutes(1));
            return isSuccess;
        }

        /// <summary>
        /// 回滚库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        public async Task<bool> RollbackStockAsync(int id)
        {
            var resord = checkRecordRepository.Get(id);
            var client = httpClientFactory.CreateClient("WMS");
            client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
            string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = resord.WarehouseCode, Sku = resord.Sku } });
            HttpContent content = new StringContent(data);
            content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
            var response = await client.PostAsync("ACOrderService/RollbackStock", content);
            bool isSuccess = false;
            if (response.IsSuccessStatusCode)
            {
                var responseContent = await response.Content.ReadAsStringAsync();
                isSuccess = JObject.Parse(responseContent)["data"].ToObject<bool>();
                CheckLog log = new CheckLog();
                log.CheckRecordId = id;
                if (isSuccess)
                {
                    resord.TheCurrentLink = TakeStockEnum.Link.释放库存;

                    log.Link = TakeStockEnum.Link.释放库存;
                    log.Description = $"已释放占用库存。";
                }
                else
                {
                    resord.TheCurrentLink = TakeStockEnum.Link.异常;
                    log.Link = TakeStockEnum.Link.异常;
                    log.Description = "释放占用库存失败，WMS返回消息：" + responseContent;
                }

                int row = await checkRecordRepository.UpdateAsync(resord, "TheCurrentLink");
                log = await checkLogRepository.InsertAsync(log);
                if (!(row > 0 && log.Id > 0)) isSuccess = false;
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => SyncAndEnabledAsync(id));
            else
                BackgroundJob.Schedule(() => RollbackStockAsync(id), DateTimeOffset.UtcNow.AddMinutes(1));

            return isSuccess;
        }

        /// <summary>
        /// 同步和启用库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        public async Task<bool> SyncAndEnabledAsync(int id)
        {
            var resord = checkRecordRepository.Get(id);
            var client = httpClientFactory.CreateClient("WMS");
            client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
            string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = resord.WarehouseCode, Sku = resord.Sku } });
            HttpContent content = new StringContent(data);
            content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
            var response = await client.PostAsync("WMSStockService/SyncAndEnabledStock", content);
            bool isSuccess = false;
            if (response.IsSuccessStatusCode)
            {
                var responseContent = await response.Content.ReadAsStringAsync();
                isSuccess = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                CheckLog log = new CheckLog();
                log.CheckRecordId = id;
                if (isSuccess)
                {
                    resord.TheCurrentLink = TakeStockEnum.Link.完成;
                    log.Link = TakeStockEnum.Link.完成;
                    log.Description = $"同步并启用库存，盘点完成。";
                }
                else
                {
                    resord.TheCurrentLink = TakeStockEnum.Link.异常;
                    log.Link = TakeStockEnum.Link.异常;
                    log.Description = "盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"];
                    BackgroundJob.Schedule(() => SyncAndEnabledAsync(id), DateTimeOffset.UtcNow.AddMinutes(1));
                }
                int row = await checkRecordRepository.UpdateAsync(resord, "TheCurrentLink");
                log = await checkLogRepository.InsertAsync(log);

                if (!(row > 0 && log.Id > 0))
                    isSuccess = false;
            }

            return isSuccess;
        }
    }
}
