﻿using AutoMapper;
using Common.Extensions;
using Common.Helper;
using Domain.Dto;
using Domain.TakeStock;
using IService.TakeStock;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Threading.Tasks;
using TakeStock.Application.TakeStock.Dto;

namespace TakeStock.Application.TakeStock
{
    public class TakeStockAppService
    {
        private readonly ITakeStockService takeStockService;
        private readonly IMapper _mapper;

        public TakeStockAppService(IMapper mapper, ITakeStockService takeStockService)
        {
            this._mapper = mapper;
            this.takeStockService = takeStockService;
        }

        public async Task<ReturnPlanDto> CreateTakeStockSchedule(ScheduleInputDto input)
        {
            var str = JsonConvert.SerializeObject(input);
            TakeStockSchedule schedule = JsonConvert.DeserializeObject<TakeStockSchedule>(str);
            List<TakeStockOrder> orders = JObject.Parse(str)["Orders"].ToObject<List<TakeStockOrder>>();
            schedule.State = TakeStockEnum.TSScheduleState.待审核;
            schedule.SysSerialNumber = GetSysSerialNumber();
            if (input.IsAutomation) // 自动盘点
            {
                orders.ForEach(x => x.AfterQuantity = null);
            }
            else if (orders.Count(x => !x.AfterQuantity.HasValue) > 0)
            {
                return new ReturnPlanDto { Message = "手动盘点sku的实际数量不能为空！" };
            }
            return await takeStockService.CreateTakeStockOrderAsync(schedule, orders);
        }

        private string GetSysSerialNumber()
        {
            string sysSerialNumber = $"SYS{DateTime.Now.ToString("yyyyMMddHHmmssfff")}" + new Random().Next(1000, 9999);
            return sysSerialNumber;
        }

        public async Task<bool> ReTry(int scheduleId)
        {
            return await takeStockService.ReTry(scheduleId);
        }

        public async Task<bool> CanceledInform(int id, string acOrderCode)
        {
            await takeStockService.CanceledInform(id, acOrderCode);
            return true;
        }

        public async Task<bool> DeliveryInform(int id, string acOrderCode)
        {
            await takeStockService.DeliveryInform(id, acOrderCode);
            return true;
        }

        public async Task<bool> StartTakeStock(StartTakeStockInputDto input)
        {
            return await takeStockService.StartTakeStock(input.Id, input.BeforeQuantity);
        }

        public async Task<bool> Feedback(FeedbackInputDto input)
        {
            return await takeStockService.Feedback(input.Id, input.AfterQuantity, input.Description);
        }

        public async Task<bool> SaveDescription(FeedbackInputDto input)
        {
            return await takeStockService.SaveDescription(input.Id, input.Description);
        }

        public async Task<bool> CancelOrder(int id)
        {
            return await takeStockService.CancelOrder(id);
        }

        private Expression<Func<TakeStockSchedule, bool>> ScheduleQuery(SearchScheduleInputDto search)
        {
            Expression<Func<TakeStockSchedule, bool>> expr = ExpressionExtensions.True<TakeStockSchedule>();
            if (!string.IsNullOrWhiteSpace(search.Code))
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.Code.Equals(search.Code);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.Code.Equals(search.Code));
            }
            if (search.State != null)
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.State.Equals(search.State);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }
            if (!string.IsNullOrWhiteSpace(search.WarehouseCode))
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.WarehouseCode.Equals(search.WarehouseCode);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }

            if (search.IsAutomation != null)
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.IsAutomation.Equals(search.IsAutomation);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }
            //if (!string.IsNullOrWhiteSpace(search.Skus))
            //{
            //    string[] skus = search.Skus.Split(';');
            //    Expression<Func<TakeStockSchedule, bool>> temp = o => skus.Contains(o.Sku);
            //    expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);

            //}

            return expr;
        }

        public async Task<SearchScheduleByPageOutputDto> SearchScheduleByPage(SearchScheduleByPageInputDto input)
        {
            int pageIndex = input.Page.Page;
            int pageSize = input.Page.Rows;
            var expr = ScheduleQuery(input.Search);
            var data = takeStockService.SearchScheduleByPage(pageIndex, pageSize, expr);
            var total = data.total;
            var items = data.items;
            return new SearchScheduleByPageOutputDto
            {
                Items = _mapper.Map<List<ScheduleOutputDto>>(items),
                PageItem = new Page.PageOutputDto
                {
                    Total = total,
                    CurrentPage = pageIndex,
                    PageSize = pageSize
                }
            };
        }

        /// <summary>
        /// 导出盘点数据
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        public MemoryStream ExportTakeStockData(SearchScheduleByPageInputDto input)
        {
            var dataList = QueryExportTakeStockData(input.Search);

            var titles = new Dictionary<string, string>()
            {
                { "Sku","Sku" },
                { "BeforeQuantity","盘点前数量" },
                { "AfterQuantity","盘点后数量" },
                { "Code","计划单号" },
                { "StateStr","盘点状态" },
                { "TakeStockTypeStr","盘点类型" },
                { "InventoryLossTypeStr","盘亏类型" },
                { "CredentialsImgUrl","附件" },
                { "AuditUserName","审核人" },
                { "AuditDateTime","审核时间" },
                { "AuditExplain","审核说明" },
                { "WarehouseName","仓库名称" },
                { "SysSerialNumber","交易流水号" },
                { "Description","盘点说明" },
                { "CreatorUserName","盘点人" },
                { "CreationTime","盘点时间" }
            };
            var list = dataList.Select(x => new { 
                x.Sku,x.BeforeQuantity,x.AfterQuantity,x.Code,x.StateStr,x.TakeStockTypeStr,x.InventoryLossTypeStr,
                x.AuditUserName,x.AuditDateTime,x.AuditExplain,x.WarehouseName,x.SysSerialNumber,x.Description,
                x.CreatorUserName,x.CreationTime,x.CredentialsImgUrl
            }).ToList();
            var ms = EPPlusUtils.ExportStream(list, titles);
            return ms;
        }

        /// <summary>
        /// 查询
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        private List<ExportTakeStockOutputDto> QueryExportTakeStockData(SearchScheduleInputDto input)
        {
            System.Text.StringBuilder sqlStr = new System.Text.StringBuilder();
            sqlStr.Append(@"
                         SELECT t1.Sku,t1.AfterQuantity,t1.BeforeQuantity,t2.`Code`,t2.AuditUserName,
                         t2.AuditDateTime,t2.AuditExplain,t3.`Name` AS WarehouseName,t2.Description,t2.TakeStockType,
                         t2.CreationTime,t2.CredentialsImgUrl,t2.State,t2.CreatorUserName,t2.SysSerialNumber,t2.InventoryLossType
                          FROM stock_takestockorder AS t1
                         INNER JOIN stock_takestockschedule AS t2 ON t2.Id = t1.ScheduleId
                         LEFT JOIN warehouse_warehouse AS t3 ON t3.`Code` = t2.WarehouseCode
                         WHERE t1.IsDeleted =0 AND t2.IsDeleted =0 ");
            if (!string.IsNullOrWhiteSpace(input.Code))
            {
                sqlStr.Append(" AND t2.`Code` = @Code");
            }
            if (!string.IsNullOrWhiteSpace(input.WarehouseCode))
            {
                sqlStr.Append(" AND t2.`WarehouseCode` = @WarehouseCode");
            }
            if (input.CreDateTime != null && !string.IsNullOrWhiteSpace(input.CreDateTime[0]))
            {
                sqlStr.Append(" AND DATE(t2.CreationTime)>=@StartTime AND DATE(t2.CreationTime)<=@EndTime ");
            }
            sqlStr.Append(" ORDER BY t2.CreationTime DESC ");
            var dataList = takeStockService.ExportTakeStockData<ExportTakeStockOutputDto>(sqlStr.ToString(), new { Code = input.Code, WarehouseCode = input.WarehouseCode, StartTime = input.CreDateTime[0], EndTime = input.CreDateTime[1] });
            return dataList;
        }

        private Expression<Func<TakeStockOrder, bool>> OrderQuery(SearchOrderInputDto search)
        {
            Expression<Func<TakeStockOrder, bool>> expr = s => s.ScheduleId == search.ScheduleId;
            if (!string.IsNullOrWhiteSpace(search.Code))
            {
                Expression<Func<TakeStockOrder, bool>> temp = o => o.Code.Equals(search.Code);
                expr = Expression.Lambda<Func<TakeStockOrder, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.Code.Equals(search.Code));
            }
            if (search.State != null)
            {
                Expression<Func<TakeStockOrder, bool>> temp = o => o.State.Equals(search.State);
                expr = Expression.Lambda<Func<TakeStockOrder, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }
            if (!string.IsNullOrWhiteSpace(search.Skus))
            {
                string[] skus = search.Skus.Split(';');
                Expression<Func<TakeStockOrder, bool>> temp = o => skus.Contains(o.Sku);
                expr = Expression.Lambda<Func<TakeStockOrder, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);

            }

            return expr;
        }

        public async Task<SearchOrderByPageOutputDto> SearchOrderByPage(SearchOrderByPageInputDto input)
        {
            int pageIndex = input.Page.Page;
            int pageSize = input.Page.Rows;
            var expr = OrderQuery(input.Search);
            var data = takeStockService.SearchOrderByPage(pageIndex, pageSize, expr);
            var total = data.total;
            var items = data.items;
            return new SearchOrderByPageOutputDto
            {
                Items = _mapper.Map<List<OrderOutputDto>>(items),
                PageItem = new Page.PageOutputDto
                {
                    Total = total,
                    CurrentPage = pageIndex,
                    PageSize = pageSize
                }
            };
        }

        public async Task<List<LogOutputDto>> SearchOrderLogs(int orderId)
        {
            Expression<Func<TakeStockOrderLog, bool>> expr = l => l.OrderId == orderId;

            var data = takeStockService.SearchOrderLogs(expr);
            return _mapper.Map<List<LogOutputDto>>(data);
        }

        public async Task PushTakeStockMsg()
        {
            //await takeStockService.PushTakeStockMsg(100);
        }

        /// <summary>
        /// 审核计划单
        /// </summary>
        /// <param name="inputDto"></param>
        /// <returns></returns>
        public async Task<ReturnPlanDto> AuditTakeStockSchedule(AuditTakeStockScheduleInputDto inputDto)
        {
            return await takeStockService.AuditTakeStockSchedule(inputDto);
        }

        /// <summary>
        /// 取消盘点计划单
        /// </summary>
        /// <param name="scheduleIds"></param>
        /// <returns></returns>
        public async Task<ReturnPlanDto> CancelTakeStockSchedule(CancelTakeStockScheduleInPutDto input)
        {
            return await takeStockService.CancelTakeStockSchedule(input);
        }
    }
}
