﻿using Domain;
using Domain.Dto;
using Domain.TakeStock;
using Domain.TakeStock.Repository;
using EasyNetQ;
using EasyNetQ.Topology;
using Hangfire;
using IService.TakeStock;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Linq.Expressions;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;
using static Domain.TakeStock.TakeStockEnum;
using QueueAttribute = Hangfire.QueueAttribute;

namespace Service.TakeStock
{
    public class TakeStockService : ITakeStockService
    {
        //private readonly RabbitMQClient mqClient;
        private readonly ITakeStockScheduleRepository _scheduleRepository;
        private readonly IRepository<TakeStockOrder> _orderRepository;
        private readonly IRepository<TakeStockOrderLog> _logRepository;
        private readonly IHttpClientFactory _httpClientFactory;
        private readonly int _delay = 3;
        private readonly IBus _bus;

        public TakeStockService(IHttpClientFactory httpClientFactory, ITakeStockScheduleRepository scheduleRepository, IRepository<TakeStockOrder> orderRepository, IRepository<TakeStockOrderLog> logRepository, IBus bus)
        {
            this._httpClientFactory = httpClientFactory;
            //this.mqClient = mqClient;
            this._scheduleRepository = scheduleRepository;
            this._orderRepository = orderRepository;
            this._logRepository = logRepository;
            this._bus = bus;
        }

        /// <summary>
        /// 创建盘点记录
        /// </summary>
        /// <param name="schedule"></param>
        /// <returns></returns>
        public async Task<ReturnPlanDto> CreateTakeStockOrderAsync(TakeStockSchedule schedule, List<TakeStockOrder> orders)
        {
            string[] skus = orders.Select(o => o.Sku).ToArray();

            Expression<Func<TakeStockOrder, bool>> predicate = o => o.WarehouseCode.Equals(schedule.WarehouseCode) && skus.Contains(o.Sku) && o.State != TSOrderState.取消 && o.State != TSOrderState.完成;
            int count = _orderRepository.Count(predicate);   //&& o.CreationTime > DateTime.Now.AddDays(-1)
            if (count > 0)
            {
                // 如果发现有存在的记录再查询这些记录返回单号
                var existIds = _orderRepository.GetAllList(predicate).Select(t => t.ScheduleId).ToArray();
                var exisPlanCodes = _scheduleRepository.GetAllList(t => existIds.Contains(t.Id)).Select(t => t.Code).ToArray();

                return new ReturnPlanDto() { IsSuccess = false, Message = $"存在盘点计划单号未处理：{string.Join(',', exisPlanCodes)}" };
            }
            var requestData = new RequestDto<List<FreezeStockInputDot>> { Data = orders.Select(x => new FreezeStockInputDot { sku = x.Sku, ExternalCode = schedule.Code, warehouseCode = schedule.WarehouseCode }).ToList() };

            var result = await PostData<WMSResponseDto<List<TakeStockFreezeOutputDto>>, RequestDto<List<FreezeStockInputDot>>>(requestData, "WMSStockService/BatchFreezeStock");
            if (!result.IsSuccess)
            {
                return new ReturnPlanDto() { IsSuccess = true, Message = "创建计划单失败！" };
            }
            foreach (var item in orders)
            {
                item.BeforeQuantity = result.Data.FirstOrDefault(x => x.Sku.Equals(item.Sku))?.Quantity;
                item.State = TSOrderState.冻结库存;
                item.LastModificationTime = DateTime.Now;
            }
            int scheduleId = _scheduleRepository.CreateOrderAndFreezeStock(schedule, orders);
            if (scheduleId <= 0)
            {
                await PostData<WMSResponseDto<bool>, RequestDto<List<FreezeStockInputDot>>>(requestData, "TakeStockScheduleService/BatchEnabledStock");
                return new ReturnPlanDto() { IsSuccess = true, Message = "创建计划单失败！" };
            }

            return new ReturnPlanDto() { IsSuccess = true, Message = "" };
        }

        /// <summary>
        /// 根据计划Id添加盘点作业
        /// </summary>
        /// <param name="scheduleId"></param>
        /// <returns></returns>
        [DisplayName("添加盘点作业, 盘点计划Id:{0}")]
        public async Task<bool> ReTry(int scheduleId)
        {
            var orders = _orderRepository.GetAllList(o => !o.IsDeleted && o.State != TSOrderState.取消 && o.State != TSOrderState.完成 && o.ScheduleId == scheduleId);
            bool isSuccess = false;
            try
            {
                foreach (var order in orders)
                {
                    order.State = TSOrderState.创建;
                    await _orderRepository.UpdateAsync(order, "State", "LastModificationTime");
                    BackgroundJob.Enqueue(() => FreezeStockAsync(order.Id));
                }
                isSuccess = true;
            }
            catch (Exception ex)
            {
                isSuccess = false;
            }
            return isSuccess;
        }

        private async Task<bool> AddOrUpdateLog(int orderId, TSOrderState state, string content, string jsonData = null)
        {
            var log = await _logRepository.FirstOrDefaultAsync(l => l.OrderId == orderId && l.State == state);
            if (log == null)
            {
                log = new TakeStockOrderLog();
                log.OrderId = orderId;
                log.State = state;
            }
            else
                log.LastModificationTime = DateTime.Now;

            log.Content = content;
            if (!string.IsNullOrWhiteSpace(jsonData))
                log.JsonData = jsonData;
            if (log.Id > 0)
                return await _logRepository.UpdateAsync(log, "Content", "JsonData", "LastModificationTime") > 0;
            else
                return _logRepository.InsertFulfill(log);
        }

        /// <summary>
        /// 冻结库存
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [DisplayName("冻结库存, 盘点单Id:{0}")]
        [Queue("freezestock")]
        public async Task<bool> FreezeStockAsync(int id)
        {
            var order = _orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.冻结库存)
                isSuccess = true;
            else
            {
                var schedule = _scheduleRepository.Get(order.ScheduleId);
                var client = _httpClientFactory.CreateClient("WMS");
                //client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku, ExternalCode = schedule.Code } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/FreezeStock", content);

                bool isFreeze = false;
                if (response.IsSuccessStatusCode)
                {
                    string responseContent = await response.Content.ReadAsStringAsync();
                    var resultObj = JObject.Parse(responseContent)["data"];
                    isFreeze = resultObj["isFreeze"].ToObject<bool>();
                    Task<bool> logTask = null;
                    if (isFreeze)
                    {
                        order.State = TSOrderState.冻结库存;
                        order.BeforeQuantity = resultObj["quantity"].ToObject<int>();
                        order.LastModificationTime = DateTime.Now;

                        logTask = AddOrUpdateLog(id, TSOrderState.冻结库存, "冻结库存成功。");
                        isSuccess = true;
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        order.LastModificationTime = DateTime.Now;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "冻结库存失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }

                    int row = await _orderRepository.UpdateAsync(order, "State", "BeforeQuantity", "LastModificationTime");
                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => RollbackStockAsync(id));
            else
                BackgroundJob.Schedule(() => FreezeStockAsync(id), DateTime.Now.AddMinutes(_delay));

            return isSuccess;
        }

        /// <summary>
        /// 回滚库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        [DisplayName("回滚库存, 盘点单Id:{0}")]
        [Queue("rollbackstock")]
        public async Task<bool> RollbackStockAsync(int id)
        {
            var order = _orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            var schedule = _scheduleRepository.Get(order.ScheduleId);
            var client = _httpClientFactory.CreateClient("WMS");
            client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
            string data = JsonConvert.SerializeObject(new { Data = new { TakeId = id, WarehouseCode = order.WarehouseCode, Sku = order.Sku, IsAutomation = schedule.IsAutomation } });
            HttpContent content = new StringContent(data);
            content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
            var response = await client.PostAsync("TakeStockScheduleService/RollbackStock", content);
            if (response.IsSuccessStatusCode)
            {
                var responseContent = await response.Content.ReadAsStringAsync();
                var jobj = JObject.Parse(responseContent)["data"];
                isSuccess = jobj["result"].ToObject<bool>();
                if (isSuccess)
                {
                    var toCancelCodes = jobj["toCancelCodes"].ToObject<string[]>();
                    var waitCodes = jobj["waitCodes"].ToObject<string[]>();
                    if (toCancelCodes?.Length > 0)
                    {
                        toCancelCodes = toCancelCodes.Distinct().ToArray();
                        order.ToCancelCodes = string.Join(',', toCancelCodes);

                        await AddOrUpdateLog(id, TSOrderState.异常, $"存在待取消的单号：{order.ToCancelCodes}");
                    }
                    if (waitCodes?.Length > 0)
                        order.WaitCodes = string.Join(',', waitCodes);
                }
                else
                {
                    order.State = TSOrderState.异常;
                    order.LastModificationTime = DateTime.Now;

                    await AddOrUpdateLog(id, TSOrderState.异常, "请求释放占用库存失败，WMS返回消息：" + jobj["message"], responseContent);
                }

                await _orderRepository.UpdateAsync(order, "State", "ToCancelCodes", "WaitCodes", "LastModificationTime");
            }

            if (isSuccess)
            {
                if ((schedule.IsAutomation && string.IsNullOrWhiteSpace(order.ToCancelCodes)) || (!schedule.IsAutomation && string.IsNullOrWhiteSpace(order.WaitCodes)))
                    BackgroundJob.Enqueue(() => Completed(id));
            }
            else
                BackgroundJob.Schedule(() => RollbackStockAsync(id), DateTime.Now.AddMinutes(_delay));

            return isSuccess;
        }

        [DisplayName("开始盘点, 开始盘点id:{0}")]
        public void StartTakeStock(int id)
        {
            var orders = _orderRepository.GetAllList(o => !o.IsDeleted && o.State != TSOrderState.取消 && o.State != TSOrderState.完成 && o.ScheduleId == id);
            if (orders.Count <= 0) return;
            foreach (var item in orders)
            {
                BackgroundJob.Enqueue(() => RollbackStockAsync(item.Id));
            }
        }

        /// <summary>
        /// Post请求
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <typeparam name="Request"></typeparam>
        /// <param name="request"></param>
        /// <param name="requestUri"></param>
        /// <returns></returns>
        private async Task<T> PostData<T, Request>(Request request, string requestUri)
        {
            try
            {
                var client = _httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(request);
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync(requestUri, content);
                if (!response.IsSuccessStatusCode) return default(T);
                var responseContent = await response.Content.ReadAsStringAsync();
                return JsonConvert.DeserializeObject<T>(responseContent);
            }
            catch (Exception ex)
            {
                return default(T);
            }
        }

        /// <summary>
        /// WMS取消配货单成功后异步通知
        /// </summary>
        /// <param name="id"></param>
        /// <param name="acOrderCode"></param>
        /// <returns></returns>
        public async Task CanceledInform(int id, string acOrderCode)
        {
            bool isSuccess = false;
            var order = _orderRepository.Get(id);

            if (string.IsNullOrWhiteSpace(order.ToCancelCodes))
                isSuccess = true;
            else
            {
                var toCancelCodes = order.ToCancelCodes.Split(',');
                order.ToCancelCodes = string.Join(',', toCancelCodes.Where(c => !c.Equals(acOrderCode)).ToArray());
                var canceledCodes = order.CancelCodes?.Split(',').ToList();
                if (canceledCodes == null) canceledCodes = new List<string>();
                canceledCodes.Add(acOrderCode);
                order.CancelCodes = string.Join(',', canceledCodes);
                await _orderRepository.UpdateAsync(order, "ToCancelCodes", "CancelCodes", "LastModificationTime");

                if (string.IsNullOrWhiteSpace(order.ToCancelCodes))
                    isSuccess = true;
            }

            if (string.IsNullOrWhiteSpace(order.ToCancelCodes) && string.IsNullOrWhiteSpace(order.WaitCodes))
                BackgroundJob.Enqueue(() => Completed(id));
        }

        /// <summary>
        /// WMS取消配货单成功后异步通知
        /// </summary>
        /// <param name="id"></param>
        /// <param name="acOrderCode"></param>
        /// <returns></returns>
        public async Task DeliveryInform(int id, string acOrderCode)
        {
            bool isSuccess = false;
            var order = _orderRepository.Get(id);

            if (string.IsNullOrWhiteSpace(order.WaitCodes))
                isSuccess = true;
            else
            {
                var waitCodes = order.WaitCodes.Split(',').ToList();
                order.WaitCodes = string.Join(',', waitCodes.Where(c => !c.Equals(acOrderCode)).ToArray());
                await _orderRepository.UpdateAsync(order, "WaitCodes", "LastModificationTime");
                if (string.IsNullOrWhiteSpace(order.WaitCodes))
                    isSuccess = true;
            }

            if (string.IsNullOrWhiteSpace(order.ToCancelCodes) && string.IsNullOrWhiteSpace(order.WaitCodes))
                BackgroundJob.Enqueue(() => Completed(id));
        }

        /// <summary>
        /// 完成取消配货单
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public async Task Completed(int id)
        {
            var order = _orderRepository.Get(id);
            order.State = TSOrderState.释放库存;
            order.LastModificationTime = DateTime.Now;
            await _orderRepository.UpdateAsync(order, "State", "LastModificationTime");
            await AddOrUpdateLog(id, TSOrderState.释放库存, "已释放占用库存。");

            var schedule = _scheduleRepository.Get(order.ScheduleId);
            if (schedule.IsAutomation)
                BackgroundJob.Enqueue(() => SyncAndEnabledAsync(id));
        }

        /// <summary>
        /// 同步和启用库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        [DisplayName("同步并启用库存, 盘点单Id:{0}")]
        [Queue("syncandenabledstock")]
        public async Task<bool> SyncAndEnabledAsync(int id)
        {
            var order = _orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.完成)
                isSuccess = true;
            else
            {
                var schedule = _scheduleRepository.Get(order.ScheduleId);
                var client = _httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku, ExternalCode = schedule.Code, CheduleCreationTime = schedule.CreationTime, SysSerialNumber = schedule.SysSerialNumber } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/SyncAndEnabledStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    var resultObj = JObject.Parse(responseContent)["data"];
                    isSuccess = resultObj["isSuccess"].ToObject<bool>();
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.完成;
                        order.AfterQuantity = resultObj["quantity"].ToObject<int>();
                        logTask = AddOrUpdateLog(id, TSOrderState.完成, "同步并启用库存，盘点完成。", responseContent);
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "盘点失败，WMS返回消息：" + resultObj["message"], responseContent);
                    }
                    int row = await _orderRepository.UpdateAsync(order, "State", "AfterQuantity", "LastModificationTime");
                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
            {
                BackgroundJob.Enqueue(() => CheckComplete(order.ScheduleId));
                //if (order.State == TSOrderState.完成)
                //    PushTakeStockMsg(order);
            }
            else
                BackgroundJob.Schedule(() => SyncAndEnabledAsync(id), DateTime.Now.AddMinutes(15));

            return isSuccess;
        }

        /// <summary>
        /// 检查是否完成计划
        /// </summary>
        /// <param name="scheduleId"></param>
        /// <returns></returns>
        [DisplayName("检查计划是否完成, 盘点计划Id:{0}")]
        [Queue("checkcomplete")]
        public async Task CheckComplete(int scheduleId)
        {
            bool isComplete = _orderRepository.Count(o => o.ScheduleId == scheduleId && o.State != TSOrderState.完成 && o.State != TSOrderState.取消) <= 0;
            if (isComplete)
            {
                var schedule = _scheduleRepository.Get(scheduleId);
                schedule.State = TakeStockEnum.TSScheduleState.完成;
                schedule.LastModificationTime = DateTime.Now;
                int row = await _scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime");

                //发起报溢报损
                //BackgroundJob.Enqueue(() => GainLoss(schedule.Id));
            }
        }

        public async Task PushTakeStockMsg(TakeStockOrder order)
        {
            if (order.AfterQuantity > 0)
            {
                string message = JsonConvert.SerializeObject(new { Code = order.Code, WarehouseCode = order.WarehouseCode, Sku = order.Sku, Quantity = order.AfterQuantity });
                _bus.Send("skuputin_queue", message);
                _bus.Dispose();
            }

            //TakeStockMessage msg = new TakeStockMessage
            //{
            //    Code = order.Code,
            //    WarehouseCode = order.WarehouseCode,
            //    Sku = order.Sku,
            //    Quantity = Convert.ToInt16(order.AfterQuantity)
            //};

            //var advancedBus = _bus.Advanced;
            //var queue = advancedBus.QueueDeclare("sku_quenen");
            //var exchange = advancedBus.ExchangeDeclare("sku_quenen", ExchangeType.Direct);
            ////var binding = advancedBus.Bind(exchange, queue, null);
            //var message = new Message<TakeStockMessage>(msg);
            //message.Properties.Type = "System.String, System.Private.CoreLib";
            //await advancedBus.PublishAsync(exchange, "sku_quenen", false, message);

            //TakeStock.Message.CreateTakeStockMessage msg = new TakeStock.Message.CreateTakeStockMessage
            //{
            //    WarehouseCode = "BLTEST",
            //    Sku = "TEST0001"
            //};

            //_bus.Publish(msg);


            //await _bus.SendAsync("sku_quenen", msg);
        }

        /// <summary>
        /// 发起线下盘点
        /// </summary>
        /// <param name="order"></param>
        /// <returns></returns>
        public async Task<bool> StartTakeStock(int id, int beforeQuantity)
        {
            var order = _orderRepository.Get(id);
            order.State = TSOrderState.盘点中;
            order.BeforeQuantity = beforeQuantity;
            order.LastModificationTime = DateTime.Now;
            var record = await _orderRepository.UpdateAsync(order, "State", "BeforeQuantity", "LastModificationTime");
            var logTask = await AddOrUpdateLog(id, TSOrderState.盘点中, "发起线下盘点。");
            return record > 0 || logTask;
        }

        /// <summary>
        /// 盘点反馈
        /// </summary>
        /// <param name="order"></param>
        /// <returns></returns>
        public async Task<bool> Feedback(int id, int afterQuantity, string description)
        {
            var order = _orderRepository.Get(id);
            order.AfterQuantity = afterQuantity;
            order.LastModificationTime = DateTime.Now;
            order.Description = description;
            var record = await _orderRepository.UpdateAsync(order, "AfterQuantity", "LastModificationTime", "Description");
            var logTask = await AddOrUpdateLog(id, TSOrderState.盘点中, "盘点反馈。");
            bool isSuccess = record > 0 && logTask;
            if (isSuccess)
                BackgroundJob.Enqueue(() => ModifAndEnabledAsync(order.Id));

            return isSuccess;
        }

        /// <summary>
        /// 盘点反馈
        /// </summary>
        /// <param name="order"></param>
        /// <returns></returns>
        public async Task<bool> SaveDescription(int id, string description)
        {
            var order = _orderRepository.Get(id);
            order.LastModificationTime = DateTime.Now;
            order.Description = description;
            var record = await _orderRepository.UpdateAsync(order, "LastModificationTime", "Description");
            var logTask = await AddOrUpdateLog(id, order.State, "保存盘点描述。");
            bool isSuccess = record > 0 && logTask;
            if (isSuccess)
                BackgroundJob.Enqueue(() => ModifAndEnabledAsync(order.Id));

            return isSuccess;
        }

        /// <summary>
        /// 同步和启用库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        [DisplayName("修改并启用库存, 盘点单Id:{0}")]
        [Queue("modifandenabledstock")]
        public async Task<bool> ModifAndEnabledAsync(int id)
        {
            var order = _orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.取消)
                return true;
            if (order.State == TSOrderState.完成)
                isSuccess = true;
            else
            {
                var client = _httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku, Quantity = order.AfterQuantity } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/ModifAndEnabled", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    isSuccess = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.完成;
                        logTask = AddOrUpdateLog(id, TSOrderState.完成, "更新为盘点数并启用库存，盘点完成。");
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }
                    int row = await _orderRepository.UpdateAsync(order, "State", "LastModificationTime");

                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
            {
                BackgroundJob.Enqueue(() => Audit(order.ScheduleId));
                if (order.State == TSOrderState.完成)
                    PushTakeStockMsg(order);
            }
            else
                BackgroundJob.Schedule(() => ModifAndEnabledAsync(id), DateTime.Now.AddMinutes(_delay));

            return isSuccess;
        }

        /// <summary>
        /// 盘点审核
        /// 线下盘点完成
        /// </summary>
        /// <returns></returns>
        public async Task<bool> Audit(int scheduleId)
        {
            var schedule = _scheduleRepository.Get(scheduleId);
            bool isSuccess = false;
            if (schedule.State == TSScheduleState.完成 || schedule.State == TSScheduleState.取消)
                isSuccess = true;
            else
            {
                var orders = _orderRepository.GetAllList(o => o.ScheduleId == scheduleId);
                int count = orders.Count(o => o.State != TSOrderState.完成 && o.State != TSOrderState.取消);
                if (count <= 0)
                {
                    bool isAudit = orders.Any(o => o.BeforeQuantity != o.AfterQuantity);
                    if (isAudit)
                    {
                        schedule.State = TSScheduleState.待审核;
                    }
                    else
                    {
                        schedule.State = TSScheduleState.完成;
                    }

                    schedule.LastModificationTime = DateTime.Now;
                    int record = await _scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime");

                    isSuccess = record > 0;
                    if (isAudit)
                        BackgroundJob.Enqueue(() => GainLoss(schedule.Id));
                }
                else
                    isSuccess = true;
            }

            return isSuccess;
        }

        /// <summary>
        /// 报溢报损
        /// </summary>
        /// <param name="scheduleId"></param>
        [DisplayName("报溢报损, 盘点计划Id:{0}")]
        [Queue("modifandenabledstock")]
        public void GainLoss(int scheduleId)
        {
            var schedule = _scheduleRepository.Get(scheduleId);
            var orders = _orderRepository.GetAllList(o => o.ScheduleId == scheduleId && o.BeforeQuantity != o.AfterQuantity);
            //报损
            var orders2 = orders.Where(o => o.BeforeQuantity > o.AfterQuantity);
            if (orders2?.Count() > 0)
            {
                var data = new
                {
                    Type = 1,
                    ExternalCode = schedule.Code,
                    WarehouseCode = schedule.WarehouseCode,
                    Details = orders.Where(o => o.BeforeQuantity > o.AfterQuantity).Select(o => new
                    {
                        Sku = o.Sku,
                        Quantity = o.BeforeQuantity - o.AfterQuantity,
                        Description = o.Description
                    })
                };
                AddGainLoss(data);
            }
            orders2 = orders.Where(o => o.BeforeQuantity < o.AfterQuantity);
            if (orders2.Count() > 0)
            {
                //报溢
                var data = new
                {
                    Type = 0,
                    ExternalCode = schedule.Code,
                    WarehouseCode = schedule.WarehouseCode,
                    Details = orders.Where(o => o.BeforeQuantity < o.AfterQuantity).Select(o => new
                    {
                        Sku = o.Sku,
                        Quantity = o.AfterQuantity - o.BeforeQuantity,
                        Description = o.Description
                    })
                };
                AddGainLoss(data);
            }
        }

        [DisplayName("远程调用报溢报损，")]
        [Queue("modifandenabledstock")]
        public async Task<string> AddGainLoss(object obj)
        {
            string code = string.Empty;
            var client = _httpClientFactory.CreateClient("WMS");
            client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
            string dataStr = JsonConvert.SerializeObject(new { Data = obj });
            HttpContent content = new StringContent(dataStr);
            content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
            var response = await client.PostAsync("GainLossOrderService/AddGainLoss", content);
            if (response.IsSuccessStatusCode)
            {
                var responseContent = await response.Content.ReadAsStringAsync();
                code = JObject.Parse(responseContent)["data"].ToString();
            }

            if (string.IsNullOrWhiteSpace(code))
                BackgroundJob.Schedule(() => AddGainLoss(obj), DateTime.Now.AddMinutes(_delay));

            return code;
        }

        /// <summary>
        /// 取消盘点单
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [DisplayName("取消盘点, 盘点单Id:{0}")]
        [Queue("canceltakestock")]
        public async Task<bool> CancelOrder(int id)
        {
            var order = _orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TSOrderState.完成 || order.State == TSOrderState.取消)
                isSuccess = true;
            else
            {
                var client = _httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string dataStr = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(dataStr);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/EnabledStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    isSuccess = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                    Task<bool> logTask = null;
                    if (isSuccess)
                    {
                        order.State = TSOrderState.取消;
                        logTask = AddOrUpdateLog(id, TSOrderState.取消, "取消盘点。");
                    }
                    else
                    {
                        order.State = TSOrderState.异常;
                        logTask = AddOrUpdateLog(id, TSOrderState.异常, "取消盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"], responseContent);
                    }
                    int row = await _orderRepository.UpdateAsync(order, "State", "LastModificationTime");

                    if (!(row > 0 && await logTask))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => CancelSchedule(order.ScheduleId));

            return isSuccess;
        }

        /// <summary>
        /// 取消盘点计划
        /// </summary>
        /// <param name="scheduleId"></param>
        /// <returns></returns>
        [DisplayName("取消盘点, 盘点计划Id:{0}")]
        [Queue("canceltakestock")]
        public async Task CancelSchedule(int scheduleId)
        {
            var schedule = _scheduleRepository.Get(scheduleId);
            bool isSuccess = false;
            if (schedule.State == TSScheduleState.完成 || schedule.State == TSScheduleState.取消)
                isSuccess = true;
            var orders = _orderRepository.GetAllList(o => o.ScheduleId == scheduleId);
            if (orders.All(o => o.State == TSOrderState.完成 || o.State == TSOrderState.取消))
            {
                if (orders.Any(o => o.State == TSOrderState.完成))
                    schedule.State = TSScheduleState.完成;
                else
                {
                    schedule.State = TSScheduleState.取消;
                    schedule.SysSerialNumber = string.Empty; // 清空流水号
                }

                schedule.LastModificationTime = DateTime.Now;
                int record = await _scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime", "SysSerialNumber");
                isSuccess = record > 0;
            }
            else
                isSuccess = true;

            if (!isSuccess)
                BackgroundJob.Schedule(() => CancelSchedule(scheduleId), DateTime.Now.AddMinutes(_delay));
        }


        public (int total, List<TakeStockSchedule> items) SearchScheduleByPage(int pageIndex, int pageSize, Expression<Func<TakeStockSchedule, bool>> expr)
        {
            int totalCount = 0;
            var data = _scheduleRepository.PageList(pageIndex, pageSize, expr, out totalCount);
            return (totalCount, data);
        }

        public (int total, List<TakeStockOrder> items) SearchOrderByPage(int pageIndex, int pageSize, Expression<Func<TakeStockOrder, bool>> expr)
        {
            int totalCount = 0;
            var data = _scheduleRepository.PageList(pageIndex, pageSize, expr, out totalCount);
            return (totalCount, data);
        }

        public List<TakeStockOrderLog> SearchOrderLogs(Expression<Func<TakeStockOrderLog, bool>> expr)
        {
            var data = _logRepository.GetAllList(expr);
            return data;
        }

        /// <summary>
        /// 审核计划单
        /// </summary>
        /// <param name="inputDto"></param>
        /// <returns></returns>
        public async Task<ReturnPlanDto> AuditTakeStockSchedule(AuditTakeStockScheduleInputDto inputDto)
        {
            var result = new ReturnPlanDto();
            var findSchedules = _scheduleRepository.GetAllList(x => inputDto.ScheduleIds.Contains(x.Id));
            if (findSchedules == null || findSchedules.Count <= 0)
            {
                result.Message = "盘点计划单不存在！";
                return result;
            }
            var noAuditCount = findSchedules.Count(x => x.State != TSScheduleState.待审核) > 0;
            if (noAuditCount)
            {
                result.Message = "选中的计划单存在不符合需要审核的数据！";
                return result;
            }
            var voidgoodsCount = findSchedules.Count(x => x.TakeStockType == TakeStockType.Voidgoods);
            var auditUserNames = "赵鹏,姚晓婷";
            if (voidgoodsCount > 0 && !auditUserNames.Contains(inputDto.AuditUserName))
            {
                result.Message = $"作废单请找{auditUserNames}审核！";
                return result;
            }
            if (inputDto.State == TSScheduleState.审核不通过)
            {
                foreach (var item in findSchedules)
                {
                    var orders = _orderRepository.GetAllList(o => !o.IsDeleted && o.State == TSOrderState.创建 && o.ScheduleId == item.Id);
                    if (orders.Count <= 0) continue;
                    var requestData = new RequestDto<List<FreezeStockInputDot>> { Data = orders.Select(x => new FreezeStockInputDot { ExternalCode = item.Code, sku = x.Sku, warehouseCode = item.WarehouseCode }).ToList() };
                    var response = await PostData<WMSResponseDto<bool>, RequestDto<List<FreezeStockInputDot>>>(requestData, "WMSStockService/BatchEnabledStock");
                    if (response.IsSuccess)
                    {
                        item.State = inputDto.State;
                        item.AuditDateTime = DateTime.Now;
                        item.AuditExplain = inputDto.AuditExplain;
                        item.AuditUserId = inputDto.AuditUserId;
                        item.AuditUserName = inputDto.AuditUserName;
                        item.SysSerialNumber = string.Empty;
                        await _scheduleRepository.UpdateAsync(item, "State", "AuditDateTime", "AuditExplain", "AuditUserId", "AuditUserName", "SysSerialNumber");
                    }
                }
                return result;
            }
            foreach (var findSchedule in findSchedules)
            {
                findSchedule.State = inputDto.State;
                findSchedule.AuditDateTime = DateTime.Now;
                findSchedule.AuditExplain = inputDto.AuditExplain;
                findSchedule.AuditUserId = inputDto.AuditUserId;
                findSchedule.AuditUserName = inputDto.AuditUserName;
                if (findSchedule.IsAutomation)
                {
                    result.IsSuccess = await _scheduleRepository.UpdateAsync(findSchedule, "State", "AuditDateTime", "AuditExplain", "AuditUserId", "AuditUserName", "IsAutomation") > 0;
                    if (result.IsSuccess)
                    {
                        BackgroundJob.Enqueue(() => StartTakeStock(findSchedule.Id));
                    }
                }
                else
                {
                    var orders = _orderRepository.GetAllList(o => !o.IsDeleted && o.State != TSOrderState.取消 && o.State != TSOrderState.完成 && o.ScheduleId == findSchedule.Id);
                    var requestData = new RequestDto<List<ModifStockInputDto>> { Data = orders.Select(x => new ModifStockInputDto { Sku = x.Sku, Quantity = x.AfterQuantity.Value, WarehouseCode = findSchedule.WarehouseCode, ExternalCode = findSchedule.Code, CheduleCreationTime = findSchedule.CreationTime, SysSerialNumber = findSchedule.SysSerialNumber }).ToList() };
                    var response = await PostData<WMSResponseDto<string>, RequestDto<List<ModifStockInputDto>>>(requestData, "WMSStockService/BatchModifAndEnabled");
                    if (response.IsSuccess)
                    {
                        result.IsSuccess = await _scheduleRepository.UpdateAsync(findSchedule, "State", "AuditDateTime", "AuditExplain", "AuditUserId", "AuditUserName", "IsAutomation") > 0;
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// 取消盘点计划单
        /// </summary>
        /// <param name="scheduleIds"></param>
        /// <returns></returns>
        public async Task<ReturnPlanDto> CancelTakeStockSchedule(CancelTakeStockScheduleInPutDto input)
        {
            var result = new ReturnPlanDto();
            var findSchedules = _scheduleRepository.GetAllList(x => input.ScheduleIds.Contains(x.Id));
            if (findSchedules == null || findSchedules.Count <= 0)
            {
                result.Message = "盘点计划单不存在！";
                return result;
            }
            foreach (var item in findSchedules)
            {
                if (item.State != TSScheduleState.待审核) continue;
                var orders = _orderRepository.GetAllList(o => !o.IsDeleted && o.ScheduleId == item.Id);
                if (orders.Count <= 0) continue;
                var requestData = new RequestDto<List<FreezeStockInputDot>> { Data = orders.Select(x => new FreezeStockInputDot { ExternalCode = item.Code, sku = x.Sku, warehouseCode = item.WarehouseCode }).ToList() };
                var response = await PostData<WMSResponseDto<bool>, RequestDto<List<FreezeStockInputDot>>>(requestData, "WMSStockService/BatchEnabledStock");
                if (response.IsSuccess)
                {
                    item.State = TSScheduleState.取消;
                    item.LastModificationTime = DateTime.Now;
                    item.LastModifierUserName = input.LastModifierUserName;
                    item.LastModifierUserId = input.LastModifierUserId;
                    item.SysSerialNumber = string.Empty;
                    await _scheduleRepository.UpdateAsync(item, "State", "LastModificationTime", "LastModifierUserName", "LastModifierUserId", "SysSerialNumber");
                    result.IsSuccess = true;
                    foreach (var order in orders)
                    {
                        order.State = TSOrderState.取消;
                        order.LastModificationTime = DateTime.Now;
                        order.LastModifierUserId = input.LastModifierUserId;
                        await _orderRepository.UpdateAsync(order, "State", "LastModificationTime", "LastModifierUserId");
                    }
                }
                else
                {
                    result.IsSuccess = false;
                }
            }
            return result;
        }
    }
}
