﻿using AutoMapper;
using Common.Extensions;
using Domain.TakeStock;
using IService.TakeStock;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Threading.Tasks;
using TakeStock.Application.TakeStock.Dto;

namespace TakeStock.Application.TakeStock
{
    public class TakeStockAppService
    {
        private readonly ITakeStockService takeStockService;
        private readonly IMapper _mapper;

        public TakeStockAppService(IMapper mapper, ITakeStockService takeStockService)
        {
            this._mapper = mapper;
            this.takeStockService = takeStockService;
        }

        public async Task<ReturnPlanDto> CreateTakeStockSchedule(ScheduleInputDto input)
        {
            var str = JsonConvert.SerializeObject(input);
            TakeStockSchedule schedule = JsonConvert.DeserializeObject<TakeStockSchedule>(str);
            List<TakeStockOrder> orders = JObject.Parse(str)["Orders"].ToObject<List<TakeStockOrder>>();
            return await takeStockService.CreateTakeStockOrderAsync(schedule, orders);
        }

        public async Task<bool> ReTry(int scheduleId)
        {
            return await takeStockService.ReTry(scheduleId);
        }

        public async Task<bool> CanceledInform(int id, string acOrderCode)
        {
            await takeStockService.CanceledInform(id, acOrderCode);
            return true;
        }

        public async Task<bool> DeliveryInform(int id, string acOrderCode)
        {
            await takeStockService.DeliveryInform(id, acOrderCode);
            return true;
        }

        public async Task<bool> StartTakeStock(StartTakeStockInputDto input)
        {
            return await takeStockService.StartTakeStock(input.Id, input.BeforeQuantity);
        }

        public async Task<bool> Feedback(FeedbackInputDto input)
        {
            return await takeStockService.Feedback(input.Id, input.AfterQuantity, input.Description);
        }

        public async Task<bool> SaveDescription(FeedbackInputDto input)
        {
            return await takeStockService.SaveDescription(input.Id, input.Description);
        }

        public async Task<bool> CancelOrder(int id)
        {
            return await takeStockService.CancelOrder(id);
        }

        private Expression<Func<TakeStockSchedule, bool>> ScheduleQuery(SearchScheduleInputDto search)
        {
            Expression<Func<TakeStockSchedule, bool>> expr = ExpressionExtensions.True<TakeStockSchedule>();
            if (!string.IsNullOrWhiteSpace(search.Code))
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.Code.Equals(search.Code);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.Code.Equals(search.Code));
            }
            if (search.State != null)
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.State.Equals(search.State);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }
            if (!string.IsNullOrWhiteSpace(search.WarehouseCode))
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.WarehouseCode.Equals(search.WarehouseCode);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }

            if (search.IsAutomation != null)
            {
                Expression<Func<TakeStockSchedule, bool>> temp = s => s.IsAutomation.Equals(search.IsAutomation);
                expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }
            //if (!string.IsNullOrWhiteSpace(search.Skus))
            //{
            //    string[] skus = search.Skus.Split(';');
            //    Expression<Func<TakeStockSchedule, bool>> temp = o => skus.Contains(o.Sku);
            //    expr = Expression.Lambda<Func<TakeStockSchedule, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);

            //}

            return expr;
        }

        public async Task<SearchScheduleByPageOutputDto> SearchScheduleByPage(SearchScheduleByPageInputDto input)
        {
            int pageIndex = input.Page.Page;
            int pageSize = input.Page.Rows;
            var expr = ScheduleQuery(input.Search);
            var data = takeStockService.SearchScheduleByPage(pageIndex, pageSize, expr);
            var total = data.total;
            var items = data.items;
            return new SearchScheduleByPageOutputDto
            {
                Items = _mapper.Map<List<ScheduleOutputDto>>(items),
                PageItem = new Page.PageOutputDto
                {
                    Total = total,
                    CurrentPage = pageIndex,
                    PageSize = pageSize
                }
            };
        }

        private Expression<Func<TakeStockOrder, bool>> OrderQuery(SearchOrderInputDto search)
        {
            Expression<Func<TakeStockOrder, bool>> expr = s => s.ScheduleId == search.ScheduleId;
            if (!string.IsNullOrWhiteSpace(search.Code))
            {
                Expression<Func<TakeStockOrder, bool>> temp = o => o.Code.Equals(search.Code);
                expr = Expression.Lambda<Func<TakeStockOrder, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.Code.Equals(search.Code));
            }
            if (search.State != null)
            {
                Expression<Func<TakeStockOrder, bool>> temp = o => o.State.Equals(search.State);
                expr = Expression.Lambda<Func<TakeStockOrder, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);
                //expr.And(s => s.State.Equals(search.State));
            }
            if (!string.IsNullOrWhiteSpace(search.Skus))
            {
                string[] skus = search.Skus.Split(';');
                Expression<Func<TakeStockOrder, bool>> temp = o => skus.Contains(o.Sku);
                expr = Expression.Lambda<Func<TakeStockOrder, bool>>(Expression.And(expr.Body, temp.Body), expr.Parameters);

            }

            return expr;
        }

        public async Task<SearchOrderByPageOutputDto> SearchOrderByPage(SearchOrderByPageInputDto input)
        {
            int pageIndex = input.Page.Page;
            int pageSize = input.Page.Rows;
            var expr = OrderQuery(input.Search);
            var data = takeStockService.SearchOrderByPage(pageIndex, pageSize, expr);
            var total = data.total;
            var items = data.items;
            return new SearchOrderByPageOutputDto
            {
                Items = _mapper.Map<List<OrderOutputDto>>(items),
                PageItem = new Page.PageOutputDto
                {
                    Total = total,
                    CurrentPage = pageIndex,
                    PageSize = pageSize
                }
            };
        }

        public async Task<List<LogOutputDto>> SearchOrderLogs(int orderId)
        {
            Expression<Func<TakeStockOrderLog, bool>> expr = l => l.OrderId == orderId;

            var data = takeStockService.SearchOrderLogs(expr);
            return _mapper.Map<List<LogOutputDto>>(data);
        }

        public async Task PushTakeStockMsg()
        {
            //await takeStockService.PushTakeStockMsg(100);
        }
    }
}
