﻿using AutoMapper;
using Domain;
using Domain.TakeStock.Repository;
using EasyNetQ;
using EasyNetQ.AutoSubscribe;
using Hangfire;
using Hangfire.MemoryStorage;
using Hangfire.MySql;
using Hangfire.Redis;
using IService.TakeStock;
using MessageQueue;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.TagHelpers;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Service.TakeStock;
using SqlSugar;
using StackExchange.Redis;
using System;
using System.Linq;
using System.Reflection;
using System.Transactions;
using TakeStock.API.Extensions;
using TakeStock.Application.TakeStock;
using TakeStock.SqlSugar;
using TakeStock.SqlSugar.Repository.TakeStock;

namespace TakeStock.API
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
            Redis = ConnectionMultiplexer.Connect(Configuration.GetConnectionString("Redis"));
        }

        public IConfiguration Configuration { get; }
        public static ConnectionMultiplexer Redis;

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.AddCors(options =>
            {
                options.AddPolicy("Default", builder =>
                {
                    // App:CorsOrigins in appsettings.json can contain more than one address separated by comma.
                    builder
                        .WithOrigins(Configuration["App:CorsOrigins"].Split(",", StringSplitOptions.RemoveEmptyEntries)
                        //.Select(o => o.RemovePostFix("/"))
                        .ToArray())
                        .AllowAnyHeader()
                        .AllowAnyMethod();
                });
            });

            services.AddSingleton(RabbitHutch.CreateBus(Configuration.GetConnectionString("RabbitMqConnection")));
            //services.RegisterEasyNetQ(Configuration.GetConnectionString("RabbitMqConnection"));
            //services.AddSingleton(RabbitHutch.CreateBus(rabbitMqConnection, serviceRegister => serviceRegister.Register(c => new EasyNetQConventions(new DefaultTypeNameSerializer()))));

            services.AddScoped<RabbitMQClient>();
            services.AddScoped<ITakeStockService, TakeStockService>();
            services.AddScoped<DbContext>();
            services.AddScoped(typeof(IRepository<>), typeof(Repository<>));
            services.AddScoped(typeof(ITakeStockScheduleRepository), typeof(TakeStockScheduleRepository));
            services.AddScoped<TakeStockAppService>();
            services.AddScoped(typeof(IConsumeAsync<CreateTakeStockConsumer>), typeof(CreateTakeStockConsumer));

            services.AddAutoMapper();

            RegisterHttpClient(services);
            services.AddHttpClient();


            // Add Hangfire services.
            services.AddHangfire(config =>
            {
                var options = new RedisStorageOptions { Db = 4 };
                config.UseRedisStorage(Redis, options);
                //config.UseMemoryStorage();
            });
            //services.AddHangfire(configuration => configuration.UseRedisStorage(Redis));
            // Add the processing server as IHostedService
            services.AddHangfireServer();

            services.AddMvc()
                .SetCompatibilityVersion(CompatibilityVersion.Version_2_2);
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IHostingEnvironment env, DbContext dbContext)
        {
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }

            app.UseCors("Default");

            var options = new DashboardOptions
            {
                IgnoreAntiforgeryToken = true,
                Authorization = new[]
                {
                    new CustomAuthorizationFilter()
                }
            };
            app.UseHangfireDashboard("/hangfire", options);
            var options2 = new BackgroundJobServerOptions
            {
                WorkerCount = Environment.ProcessorCount * 10,
                Queues = new[] { "freezestock", "rollbackstock", "syncandenabledstock", "checkcomplete", "modifandenabledstock", "canceltakestock", "default" }
            };
            app.UseHangfireServer(options2);

            app.UseMvc();
            //dbContext.Migration();

            //app.UseSubscribe("TakeStockService", AppDomain.CurrentDomain.GetAssemblies().Single(a => a.GetName().Name == "Service"));
        }

        public void RegisterHttpClient(IServiceCollection services)
        {
            string node = "WMS";
            services.AddHttpClient(Configuration[$"{ node }:Name"], c =>
            {
                Setting(node, c); //WMS_QueryStockPoolWMS_QueryStockPool
            });
        }

        public void Setting(string node, System.Net.Http.HttpClient client)
        {
            client.Timeout = TimeSpan.FromSeconds(Convert.ToDouble(Configuration[$"{ node }:Timeout"]));
            client.BaseAddress = new Uri(Configuration[$"{ node }:Uri"]);
        }
    }
}
