﻿using Domain;
using Domain.Domain.TakeStock;
using Domain.Domain.TakeStock.Repository;
using Hangfire;
using IService.TakeStock;
using MessageQueue;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;

namespace Service.TakeStock
{
    public class TakeStockService : ITakeStockService
    {
        private readonly RabbitMQClient mqClient;
        private readonly ITakeStockScheduleRepository scheduleRepository;
        private readonly IRepository<TakeStockOrder> orderRepository;
        private readonly IRepository<TakeStockOrderLog> logRepository;
        private readonly IHttpClientFactory httpClientFactory;

        public TakeStockService(IHttpClientFactory httpClientFactory, RabbitMQClient mqClient, ITakeStockScheduleRepository scheduleRepository, IRepository<TakeStockOrder> orderRepository, IRepository<TakeStockOrderLog> logRepository)
        {
            this.httpClientFactory = httpClientFactory;
            this.mqClient = mqClient;
            this.scheduleRepository = scheduleRepository;
            this.orderRepository = orderRepository;
            this.logRepository = logRepository;
        }

        /// <summary>
        /// 创建盘点记录
        /// </summary>
        /// <param name="schedule"></param>
        /// <returns></returns>
        public async Task<bool> CreateTakeStockOrderAsync(TakeStockSchedule schedule, List<TakeStockOrder> orders)
        {
            return await Task.Run(() =>
            {
                bool isSuccess = scheduleRepository.CreateOrder(schedule, orders);
                if (isSuccess)
                {
                    foreach (var order in orders)
                    {
                        BackgroundJob.Enqueue(() => FreezeStockAsync(order.Id));
                    }
                }
                return isSuccess;
            });
        }

        /// <summary>
        /// 冻结库存
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public async Task<bool> FreezeStockAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TakeStockEnum.TSOrderState.冻结库存)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                //client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/FreezeStock", content);

                bool isFreeze = false;
                if (response.IsSuccessStatusCode)
                {
                    string responseContent = await response.Content.ReadAsStringAsync();
                    isFreeze = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                    TakeStockOrderLog log = new TakeStockOrderLog();
                    log.OrderId = id;

                    if (isFreeze)
                    {
                        order.State = TakeStockEnum.TSOrderState.冻结库存;
                        order.LastModificationTime = DateTime.Now;

                        log.State = TakeStockEnum.TSOrderState.冻结库存;
                        log.Content = "冻结库存成功。";
                        isSuccess = true;
                    }
                    else
                    {
                        order.State = TakeStockEnum.TSOrderState.异常;
                        order.LastModificationTime = DateTime.Now;
                        log.State = TakeStockEnum.TSOrderState.异常;
                        log.Content = "冻结库存失败，WMS返回消息：" + JObject.Parse(responseContent)["message"];
                        log.JsonData = responseContent;
                    }

                    int row = await orderRepository.UpdateAsync(order, "State", "LastModificationTime");
                    log = await logRepository.InsertAsync(log);
                    if (!(row > 0 && log.Id > 0))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => RollbackStockAsync(id));
            else
                BackgroundJob.Schedule(() => FreezeStockAsync(id), DateTimeOffset.UtcNow.AddMinutes(1));

            return isSuccess;
        }

        /// <summary>
        /// 回滚库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        public async Task<bool> RollbackStockAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TakeStockEnum.TSOrderState.释放库存)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("ACOrderService/RollbackStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    isSuccess = JObject.Parse(responseContent)["data"].ToObject<bool>();
                    TakeStockOrderLog log = new TakeStockOrderLog();
                    log.OrderId = id;
                    if (isSuccess)
                    {
                        order.State = TakeStockEnum.TSOrderState.释放库存;
                        order.LastModificationTime = DateTime.Now;

                        log.State = TakeStockEnum.TSOrderState.释放库存;
                        log.Content = $"已释放占用库存。";
                    }
                    else
                    {
                        order.State = TakeStockEnum.TSOrderState.异常;
                        order.LastModificationTime = DateTime.Now;

                        log.State = TakeStockEnum.TSOrderState.异常;
                        log.Content = "释放占用库存失败，WMS返回消息：" + JObject.Parse(responseContent)["message"];
                        log.JsonData = responseContent;
                    }

                    int row = await orderRepository.UpdateAsync(order, "State", "LastModificationTime");
                    log = await logRepository.InsertAsync(log);
                    if (!(row > 0 && log.Id > 0)) isSuccess = false;
                }
            }

            if (isSuccess)
            {
                var schedule = scheduleRepository.Get(order.ScheduleId);
                if(schedule.IsAutomation)
                    BackgroundJob.Enqueue(() => SyncAndEnabledAsync(id));
            }
            else
                BackgroundJob.Schedule(() => RollbackStockAsync(id), DateTimeOffset.UtcNow.AddMinutes(1));

            return isSuccess;
        }

        /// <summary>
        /// 同步和启用库存
        /// </summary>
        /// <param name="warehouseCode"></param>
        /// <param name="sku"></param>
        /// <returns></returns>
        public async Task<bool> SyncAndEnabledAsync(int id)
        {
            var order = orderRepository.Get(id);
            bool isSuccess = false;
            if (order.State == TakeStockEnum.TSOrderState.完成)
                isSuccess = true;
            else
            {
                var client = httpClientFactory.CreateClient("WMS");
                client.DefaultRequestHeaders.Add("Access-Control-Allow-Origin", "*");
                string data = JsonConvert.SerializeObject(new { Data = new { WarehouseCode = order.WarehouseCode, Sku = order.Sku } });
                HttpContent content = new StringContent(data);
                content.Headers.ContentType = new MediaTypeHeaderValue("application/json");
                var response = await client.PostAsync("WMSStockService/SyncAndEnabledStock", content);
                if (response.IsSuccessStatusCode)
                {
                    var responseContent = await response.Content.ReadAsStringAsync();
                    isSuccess = Convert.ToBoolean(JObject.Parse(responseContent)["data"]);
                    TakeStockOrderLog log = new TakeStockOrderLog();
                    log.OrderId = id;
                    if (isSuccess)
                    {
                        order.State = TakeStockEnum.TSOrderState.完成;
                        log.State = TakeStockEnum.TSOrderState.完成;
                        log.Content = $"同步并启用库存，盘点完成。";
                    }
                    else
                    {
                        order.State = TakeStockEnum.TSOrderState.异常;
                        log.State = TakeStockEnum.TSOrderState.异常;
                        log.Content = "盘点失败，WMS返回消息：" + JObject.Parse(responseContent)["message"];
                        log.JsonData = responseContent;
                    }
                    int row = await orderRepository.UpdateAsync(order, "State", "LastModificationTime");
                    log = await logRepository.InsertAsync(log);

                    if (!(row > 0 && log.Id > 0))
                        isSuccess = false;
                }
            }

            if (isSuccess)
                BackgroundJob.Enqueue(() => CheckComplete(order.ScheduleId));
            else
                BackgroundJob.Schedule(() => SyncAndEnabledAsync(id), DateTimeOffset.UtcNow.AddMinutes(1));

            return isSuccess;
        }

        public async Task CheckComplete(int scheduleId)
        {
            bool isComplete = orderRepository.Count(o => o.ScheduleId == scheduleId && (o.State != TakeStockEnum.TSOrderState.完成 || o.State != TakeStockEnum.TSOrderState.取消)) <= 0;
            if (isComplete)
            {
                var schedule = scheduleRepository.Get(scheduleId);
                schedule.State = TakeStockEnum.TSScheduleState.审核;
                schedule.LastModificationTime = DateTime.Now;
                int row = await scheduleRepository.UpdateAsync(schedule, "State", "LastModificationTime");

                //发起报溢报损
                //BackgroundJob.Enqueue(() => GainLoss(schedule.Id));
            }
        }

        public async Task GainLoss(int scheduleId)
        {

        }
    }
}
