﻿using AutoMapper;
using Common.Extensions;
using Domain.Domain.TakeStock;
using IService.TakeStock;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using System.Threading.Tasks;
using TakeStock.Application.TakeStock.Dto;

namespace TakeStock.Application.TakeStock
{
    public class TakeStockAppService
    {
        private readonly ITakeStockService takeStockService;
        private readonly IMapper _mapper;

        public TakeStockAppService(IMapper mapper, ITakeStockService takeStockService)
        {
            this._mapper = mapper;
            this.takeStockService = takeStockService;
        }

        public async Task<bool> CreateTakeStockSchedule(ScheduleInputDto input)
        {
            var str = JsonConvert.SerializeObject(input);
            TakeStockSchedule schedule = JsonConvert.DeserializeObject<TakeStockSchedule>(str);
            List<TakeStockOrder> orders = JObject.Parse(str)["Orders"].ToObject<List<TakeStockOrder>>();
            return await takeStockService.CreateTakeStockOrderAsync(schedule, orders);
        }

        public async Task<bool> ReTry(int scheduleId)
        {
            return await takeStockService.ReTry(scheduleId);
        }

        public async Task<bool> StartTakeStock(StartTakeStockInputDto input)
        {
            return await takeStockService.StartTakeStock(input.Id, input.BeforeQuantity);
        }

        public async Task<bool> Feedback(FeedbackInputDto input)
        {
            return await takeStockService.Feedback(input.Id, input.AfterQuantity, input.Description);
        }

        public async Task<bool> CancelOrder(int id)
        {
            return await takeStockService.CancelOrder(id);
        }

        private Expression<Func<TakeStockSchedule, bool>> ScheduleQuery(SearchScheduleInputDto search)
        {
            Expression<Func<TakeStockSchedule, bool>> expr = s => true;
            if (!string.IsNullOrWhiteSpace(search.Code))
                expr.And(s => s.Code.Equals(search.Code));
            if (search.State != null)
                expr.And(s => s.State.Equals(search.State));

            return expr;
        }

        public async Task<SearchScheduleByPageOutputDto> SearchScheduleByPage(SearchScheduleByPageInputDto input)
        {
            int pageIndex = input.Page.Page;
            int pageSize = input.Page.Rows;
            var expr = ScheduleQuery(input.Search);
            var data = takeStockService.SearchScheduleByPage(pageIndex, pageSize, expr);
            var total = data.total;
            var items = data.items;
            return new SearchScheduleByPageOutputDto
            {
                Items = _mapper.Map<List<ScheduleOutputDto>>(items),
                PageItem = new Page.PageOutputDto
                {
                    Total = total,
                    CurrentPage = pageIndex,
                    PageSize = pageSize
                }
            };
        }
    }
}
