﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.Linq;
using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using ServiceStack.Common.Extensions;
using ServiceStack.Redis;
using JsonSerializer = ServiceStack.Text.JsonSerializer;

namespace BailunRedis
{
    public class RedisBase
    {
        private static string[] ReadWriteHosts = ConfigurationManager.AppSettings["readWriteHosts"].Split(new char[] { ';' });
        private static string[] ReadOnlyHosts = ConfigurationManager.AppSettings["readOnlyHosts"].Split(new char[] { ';' });

        #region Connection
        public static PooledRedisClientManager prcm = CreateManager(ReadWriteHosts, ReadOnlyHosts);

        private static PooledRedisClientManager CreateManager(string[] readWriteHosts, string[] readOnlyHosts)
        {
            // 支持读写分离，均衡负载  
            return new PooledRedisClientManager(readWriteHosts, readOnlyHosts, new RedisClientManagerConfig
            {
                MaxWritePoolSize = 10, // “写”链接池链接数  
                MaxReadPoolSize = 10, // “读”链接池链接数  
                AutoStart = true
            });
        }
        #endregion

        #region Item
        /// <summary> 
        /// 设置单体 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="t"></param> 
        /// <param name="timeSpan"></param> 
        /// <returns></returns> 
        public static bool ItemSet<T>(string key, T t)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {

                return redis.Set<T>(key, t);
            }
        }

        /// <summary> 
        /// 获取单体 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <returns></returns> 
        public static T ItemGet<T>(string key) where T : class
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {

                return redis.Get<T>(key);
            }
        }

        /// <summary> 
        /// 移除单体 
        /// </summary> 
        /// <param name="key"></param> 
        public static bool ItemRemove(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                if (redis.ContainsKey(key))
                    return redis.Remove(key);
                return true;
            }
        }

        #endregion

        #region List
        public static void ListAdd<T>(string key, T t)
        {
            if (t == null)
            {
                return;
            }
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                if (ListContains(key, t))
                {
                    ListRemove(key, t);
                }
                var redisTypedClient = redis.As<T>();
                redisTypedClient.AddItemToList(redisTypedClient.Lists[key], t);
            }
        }
        /// <summary>
        /// 新增批量数据
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="key"></param>
        /// <param name="values"></param>
        public static void ListAddRange<T>(string key, List<T> values)
        {
            if (values.Count > 0)
            {
                using (IRedisClient redis = prcm.GetReadOnlyClient())
                {
                    if (!redis.ContainsKey(key))
                    {
                        var redisTypedClient = redis.As<T>();
                        foreach (var item in values)
                        {
                            redisTypedClient.AddItemToList(redisTypedClient.Lists[key], item);
                        }
                    }
                }
            }
        }

        public static bool ListRemove<T>(string key, T t)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var redisTypedClient = redis.As<T>();
                return redisTypedClient.RemoveItemFromList(redisTypedClient.Lists[key], t) > 0;
            }
        }
        public static void ListRemoveAll<T>(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var redisTypedClient = redis.As<T>();
                redisTypedClient.Lists[key].RemoveAll();
            }
        }

        public static bool ListContains<T>(string key, T t)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var redisTypedClient = redis.As<T>();
                return redisTypedClient.Lists[key].Contains(t);
            }
        }

        public static long ListCount(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.GetListCount(key);
            }
        }

        public static List<T> ListGetRange<T>(string key, int start, int count)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var c = redis.As<T>();
                return c.Lists[key].GetRange(start, start + count - 1);
            }
        }

        public static List<T> ListGetAll<T>(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var c = redis.As<T>();
                var result = c.Lists[key].GetAll();
                return result;
            }
        }

        public static List<T> ListGetPaged<T>(string key, int pageIndex, int pageSize)
        {
            int start = pageSize * (pageIndex - 1);
            return ListGetRange<T>(key, start, pageSize);
        }

        #endregion

        #region Set
        public static void SetAdd<T>(string key, T t)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var redisTypedClient = redis.As<T>();
                redisTypedClient.Sets[key].Add(t);
            }
        }
        public static bool SetContains<T>(string key, T t)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var redisTypedClient = redis.As<T>();
                return redisTypedClient.Sets[key].Contains(t);
            }
        }
        public static bool SetRemove<T>(string key, T t)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var redisTypedClient = redis.As<T>();
                return redisTypedClient.Sets[key].Remove(t);
            }
        }
        #endregion

        #region Hash
        /// <summary> 
        /// 判断某个数据是否已经被缓存 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="dataKey"></param> 
        /// <returns></returns> 
        public static bool HashExist<T>(string key, string dataKey)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.HashContainsEntry(key, dataKey);
            }
        }

        /// <summary> 
        /// 存储数据到hash表 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="dataKey"></param> 
        /// <returns></returns> 
        public static bool HashSet<T>(string key, string dataKey, T t)
        {
            if (t == null)
            {
                return false;
            }
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore,NullValueHandling = NullValueHandling.Ignore};
                var value = JsonConvert.SerializeObject(t, jsSettings);
                //string value = JsonSerializer.SerializeToString(t);
                return redis.SetEntryInHashIfNotExists(key, dataKey, value);
            }
        }

        /// <summary> 
        /// 移除hash中的某值 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="dataKey"></param> 
        /// <returns></returns> 
        public static bool HashRemove(string key, string dataKey)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.RemoveEntryFromHash(key, dataKey);
            }
        }
        /// <summary> 
        /// 移除整个hash 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="dataKey"></param> 
        /// <returns></returns> 
        public static bool HashRemove(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.Remove(key);
            }
        }

        public static void HashSetRange(string key, IEnumerable<KeyValuePair<string, string>> keyValuePairs)
        {
            if (keyValuePairs != null && keyValuePairs.Any())
            {
                using (IRedisClient redis = prcm.GetReadOnlyClient())
                {
                    redis.SetRangeInHash(key, keyValuePairs);
                }
            }

        }
        /// <summary> 
        /// 从hash表获取数据 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="dataKey"></param> 
        /// <returns></returns> 
        public static T HashGet<T>(string key, string dataKey)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                string value = redis.GetValueFromHash(key, dataKey);
                if (value.IsNullOrEmpty())
                {
                    return default(T);
                }
                var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore, NullValueHandling = NullValueHandling.Ignore };
                return JsonConvert.DeserializeObject<T>(value, jsSettings);
            }
        }
        /// <summary> 
        /// 获取整个hash的数据 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <returns></returns> 
        public static List<T> HashGetAll<T>(string key)
        {
            //var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore };
            //var value = JsonConvert.SerializeObject(t, jsSettings);
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var list = redis.GetHashValues(key);
                if (list != null && list.Count > 0)
                {
                    List<T> result = new List<T>();
                    var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore };
                    foreach (var item in list)
                    {
                        //var value = JsonSerializer.DeserializeFromString<T>(item);
                        if (item.IsNullOrEmpty())
                        {
                            return default(List<T>);
                        }
                        var value = JsonConvert.DeserializeObject<T>(item, jsSettings);
                        result.Add(value);
                    }
                    return result;
                }
                return null;
            }
        }

        /// <summary> 
        /// 获取Hash集合数量 
        /// </summary> 
        /// <param name="key">Hashid</param> 
        public static long HashGetCount(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.GetHashCount(key);
            }
        }
        #endregion

        #region SortedSet
        /// <summary> 
        ///  添加数据到 SortedSet 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="t"></param> 
        /// <param name="score"></param> 
        public static bool SortedSetAdd<T>(string key, T t, double score)
        {
            if (t == null)
            {
                return false;
            }
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore };
                var value = JsonConvert.SerializeObject(t, jsSettings);
                //string value = JsonSerializer.SerializeToString<T>(t);

                return redis.AddItemToSortedSet(key, value, score);
            }
        }
        /// <summary> 
        /// 移除数据从SortedSet 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="t"></param> 
        /// <returns></returns> 
        public static bool SortedSetRemove<T>(string key, T t)
        {
            if (t == null)
            {
                return false;
            }
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                //string value = JsonSerializer.SerializeToString<T>(t);
                var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore };
                var value = JsonConvert.SerializeObject(t, jsSettings);
                return redis.RemoveItemFromSortedSet(key, value);
            }
        }

        /// <summary> 
        /// 获取SortedSet的长度 
        /// </summary> 
        /// <param name="key"></param> 
        /// <returns></returns> 
        public static long SortedSetCount(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.GetSortedSetCount(key);
            }
        }

        /// <summary> 
        /// 获取SortedSet的分页数据 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="pageIndex"></param> 
        /// <param name="pageSize"></param> 
        /// <returns></returns> 
        public static List<T> SortedSetGetPaged<T>(string key, int pageIndex, int pageSize)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var list = redis.GetRangeFromSortedSet(key, (pageIndex - 1) * pageSize, pageIndex * pageSize - 1);
                if (list != null && list.Count > 0)
                {
                    var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore };
                    List<T> result = new List<T>();
                    foreach (var item in list)
                    {
                        if (item.IsNullOrEmpty())
                        {
                            return default(List<T>);
                        }
                        //var data = JsonSerializer.DeserializeFromString<T>(item);
                        var data = JsonConvert.DeserializeObject<T>(item, jsSettings);
                        result.Add(data);
                    }
                    return result;
                }
            }
            return null;
        }


        /// <summary> 
        /// 获取SortedSet的全部数据 
        /// </summary> 
        /// <typeparam name="T"></typeparam> 
        /// <param name="key"></param> 
        /// <param name="pageIndex"></param> 
        /// <param name="pageSize"></param> 
        /// <returns></returns> 
        public static List<T> SortedSetGetALL<T>(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var list = redis.GetAllItemsFromList(key);
                if (list != null && list.Count > 0)
                {
                    List<T> result = new List<T>();
                    var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore };
                    foreach (var item in list)
                    {
                        if (item.IsNullOrEmpty())
                        {
                            return default(List<T>);
                        }
                        //var data = JsonSerializer.DeserializeFromString<T>(item);
                        var data = JsonConvert.DeserializeObject<T>(item, jsSettings);
                        result.Add(data);
                    }
                    return result;
                }
            }
            return null;
        }

        #endregion

        /// <summary>
        /// set expire time for spicify key
        /// </summary>
        /// <param name="key"></param>
        /// <param name="timeSpan"></param>
        public static void SetExpire(string key, TimeSpan timeSpan)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                redis.ExpireEntryIn(key, timeSpan);
            }
        }

        /// <summary>
        /// Remove all redis cache
        /// </summary>
        public static void RemoveAll()
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                redis.RemoveAll(redis.GetAllKeys());
            }
        }

        /// <summary>
        /// 获取所有的Key值
        /// </summary>
        /// <returns></returns>
        public static List<string> GetAllKey(string searchParam = "*")
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                return redis.SearchKeys(searchParam);
            }
        }

        public static string GetValues(string key)
        {
            using (IRedisClient redis = prcm.GetReadOnlyClient())
            {
                var jsSettings = new JsonSerializerSettings { ReferenceLoopHandling = ReferenceLoopHandling.Ignore, DateFormatHandling = DateFormatHandling.IsoDateFormat };


                var temp = redis.GetValues(new List<string>() { key });
                if (temp == null || temp.Count == 0) //如果是Hash
                {
                    return JsonConvert.SerializeObject(HashGetAll<object>(key), jsSettings);
                }
                return JsonConvert.SerializeObject(temp, jsSettings);
            }
        }
    }
}