﻿using System;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Linq;
using System.Text;
using System.Web;
using NPOI.HSSF.UserModel;
using NPOI.SS.UserModel;
using NPOI.XSSF.UserModel;
using NPOI.SS.Util;

namespace CommonControl
{
    public static class ExcelUtil
    {
        public static DataTable ExcelToDataTable(string fileName, string sheetName, bool isFirstRowColumn, Dictionary<string, string[]> columnTemplate = null, string[] requireColumns = null,int? maxRows = null)
        {
            ISheet sheet = null;
            DataTable data = new DataTable();
            IWorkbook workbook = null;
            int startRow = 0;
            try
            {
                using (var fs = new FileStream(fileName, FileMode.Open, FileAccess.Read))
                {
                    if (fileName.IndexOf(".xlsx", StringComparison.Ordinal) > 0) // 2007版本
                        workbook = new XSSFWorkbook(fs);
                    else if (fileName.IndexOf(".xls", StringComparison.Ordinal) > 0) // 2003版本
                        workbook = new HSSFWorkbook(fs);
                }

                if (sheetName != null)
                {
                    if (workbook != null)
                    {
                        sheet = workbook.GetSheet(sheetName);
                        if (sheet == null) //如果没有找到指定的sheetName对应的sheet，则尝试获取第一个sheet
                        {
                            sheet = workbook.GetSheetAt(0);
                        }
                    }
                }
                else
                {
                    if (workbook != null) sheet = workbook.GetSheetAt(0);
                }
                if (sheet != null)
                {
                    IRow firstRow = sheet.GetRow(0);
                    int cellCount = firstRow.LastCellNum; //一行最后一个cell的编号 即总的列数

                    if (isFirstRowColumn)
                    {
                        for (int i = firstRow.FirstCellNum; i < cellCount; ++i)
                        {
                            ICell cell = firstRow.GetCell(i);
                            string cellValue = cell?.StringCellValue?.Trim();
                            if (!string.IsNullOrWhiteSpace(cellValue))//列名正确性验证
                            {
                                if (columnTemplate != null && !columnTemplate.First().Value.Contains(cellValue))
                                    throw new Exception($"{columnTemplate.First().Key}不存在列名：{cellValue}！正确列名为：{string.Join(",", columnTemplate.First().Value)}");
                                DataColumn column = new DataColumn(cellValue);
                                data.Columns.Add(column);
                            }
                        }
                        startRow = sheet.FirstRowNum + 1;
                    }
                    else
                    {
                        startRow = sheet.FirstRowNum;
                    }

                    //最后一列的标号
                    int rowCount = sheet.LastRowNum;
                    if (maxRows != null)
                    {
                        if(rowCount> maxRows)
                            throw new Exception($"请拆分文件，一次最多支持{maxRows}条数据");
                    }
                    for (int i = startRow; i <= rowCount; ++i)
                    {
                        IRow row = sheet.GetRow(i);
                        if (row == null || row.Cells.Count == 0 || row.FirstCellNum == -1 || row.Cells.All(d => d.CellType == CellType.Blank)) continue; //没有数据的行默认是null　　　　　　　

                        DataRow dataRow = data.NewRow();
                        for (int j = row.FirstCellNum; j < cellCount; ++j)
                        {
                            var cellvalue = row.GetCell(j);
                            if (cellvalue == null || (cellvalue.ToString().Trim() == "0"))
                            {
                                if (requireColumns != null && requireColumns.Contains(data.Columns[j].ColumnName))
                                {
                                    
                                    //throw new Exception($"第{i}行，第{j}列,【{data.Columns[j].ColumnName}】不能为空或0，必须填写！");
                                }

                                
                            }
                            if (cellvalue != null) dataRow[j] = cellvalue.ToString().Trim();
                            else
                            {
                                dataRow[j] = ""; //string.Empty;
                            }
                        }
                        data.Rows.Add(dataRow);
                    }
                }
                workbook?.Close();
                return data;
            }
            catch (Exception ex)
            {
                workbook?.Close();
                throw new Exception(ex.Message);
            }
        }

        /// <summary>
        /// 生成Excel文件
        /// </summary>
        /// <param name="dt">数据源</param>
        /// <param name="strExcelFileName">文件名</param>
        /// <param name="isWebDownload">如果是web下载，strExcelFileName则仅仅是文件名而非路径名</param>
        public static void GridToExcelByNPOI(DataTable dt, string strExcelFileName,bool isWebDownload = false)
        {
            HSSFWorkbook workbook = new HSSFWorkbook();
            try
            {
                int sheetIndex = 0;
                int dataIndex = 0;
                ICellStyle HeadercellStyle = workbook.CreateCellStyle();
                HeadercellStyle.BorderBottom = NPOI.SS.UserModel.BorderStyle.Thin;
                HeadercellStyle.BorderLeft = NPOI.SS.UserModel.BorderStyle.Thin;
                HeadercellStyle.BorderRight = NPOI.SS.UserModel.BorderStyle.Thin;
                HeadercellStyle.BorderTop = NPOI.SS.UserModel.BorderStyle.Thin;
                HeadercellStyle.Alignment = NPOI.SS.UserModel.HorizontalAlignment.Center;

                ICellStyle cellStyle = workbook.CreateCellStyle();

                //为避免日期格式被Excel自动替换，所以设定 format 为 『@』 表示一率当成text來看
                cellStyle.DataFormat = HSSFDataFormat.GetBuiltinFormat("@");
                cellStyle.BorderBottom = NPOI.SS.UserModel.BorderStyle.Thin;
                cellStyle.BorderLeft = NPOI.SS.UserModel.BorderStyle.Thin;
                cellStyle.BorderRight = NPOI.SS.UserModel.BorderStyle.Thin;
                cellStyle.BorderTop = NPOI.SS.UserModel.BorderStyle.Thin;

                //字体
                NPOI.SS.UserModel.IFont headerfont = workbook.CreateFont();
                headerfont.Boldweight = (short)FontBoldWeight.Bold;
                HeadercellStyle.SetFont(headerfont);


                NPOI.SS.UserModel.IFont cellfont = workbook.CreateFont();
                cellfont.Boldweight = (short)FontBoldWeight.Normal;
                cellStyle.SetFont(cellfont);
                while (dataIndex < dt.Rows.Count)
                {
                    sheetIndex++;
                    ISheet sheet = workbook.CreateSheet($"Sheet{sheetIndex}");

                    //用column name 作为列名
                    int icolIndex = 0;
                    IRow headerRow = sheet.CreateRow(0);
                    foreach (DataColumn item in dt.Columns)
                    {
                        ICell cell = headerRow.CreateCell(icolIndex);
                        cell.SetCellValue(item.ColumnName);
                        cell.CellStyle = HeadercellStyle;
                        icolIndex++;
                    }


                    //建立内容行
                    int iRowIndex = 1;
                    int iCellIndex = 0;
                    for (int count = 0; dataIndex < dt.Rows.Count; dataIndex++, count++)
                    {
                        if (count >= 65000)
                            break;
                        DataRow Rowitem = dt.Rows[dataIndex];
                        IRow DataRow = sheet.CreateRow(iRowIndex);
                        foreach (DataColumn Colitem in dt.Columns)
                        {

                            ICell cell = DataRow.CreateCell(iCellIndex);
                            cell.SetCellValue(Rowitem[Colitem].ToString());
                            cell.CellStyle = cellStyle;
                            iCellIndex++;
                        }
                        iCellIndex = 0;
                        iRowIndex++;
                    }

                    //自适应列宽度
                    for (int i = 0; i < icolIndex; i++)
                    {
                        sheet.AutoSizeColumn(i);
                    }
                }

                //如果没有传路径,就生成用于web下载的流
                if (isWebDownload)
                {
                    using (MemoryStream ms = new MemoryStream())
                    {
                        workbook.Write(ms);
                        ms.Flush();
                        ms.Position = 0;

                        byte[] bytes = ms.GetBuffer();
                        string UserAgent = HttpContext.Current.Request.ServerVariables["http_user_agent"].ToLower();
                        string filename = strExcelFileName + ".xls";
                        if (UserAgent.IndexOf("firefox") <= 0)//火狐,文件名不需要编码
                        {
                            filename = HttpUtility.UrlEncode(filename, Encoding.UTF8);
                        }
                        HttpContext.Current.Response.ContentType = "application/vnd.ms-excel";

                        HttpContext.Current.Response.AppendHeader("Content-Disposition", "attachment;filename=" + filename);
                        HttpContext.Current.Response.BinaryWrite(bytes);
                        HttpContext.Current.Response.End();
                    }
                }
                else
                {
                    if (System.IO.File.Exists(strExcelFileName))
                        System.IO.File.Delete(strExcelFileName);

                    //写Excel
                    FileStream file = new FileStream(strExcelFileName, FileMode.Create);
                    workbook.Write(file);
                    file.Flush();
                    file.Close();
                }

            }
            catch (Exception ex)
            {
                throw;
            }
            finally { workbook = null; }

        }

        /// <summary>
        /// 将DataTable数据导入到excel中
        /// </summary>
        /// <param name="data">要导入的数据</param>
        /// <param name="isColumnWritten">DataTable的列名是否要导入</param>
        /// <param name="fileName">导出文件名全路径</param>
        /// <param name="sheetName">要导入的excel的sheet的名称</param>
        /// <returns>导入数据行数(包含列名那一行)</returns>
        public static int DataTableToExcel(DataTable data, string fileName, bool isColumnWritten, string sheetName = "Sheet1")
        {
            int i = 0;
            int j = 0;
            int count = 0;
            ISheet sheet = null;
            IWorkbook workbook = null;
            string myDir = Path.GetDirectoryName(fileName);
            //判断文件夹是否存在  
            if (!Directory.Exists(myDir))
            {
                //文件夹不存在则创建该文件夹  
                if (myDir != null)
                    Directory.CreateDirectory(myDir);
            }
            using (var fs = new FileStream(fileName, FileMode.Create, FileAccess.ReadWrite))
            {
                if (fileName.IndexOf(".xlsx", StringComparison.Ordinal) > 0) // 2007版本
                    workbook = new XSSFWorkbook();
                else if (fileName.IndexOf(".xls", StringComparison.Ordinal) > 0) // 2003版本
                    workbook = new HSSFWorkbook();

                try
                {
                    if (workbook != null)
                    {
                        sheet = workbook.CreateSheet(sheetName);
                    }
                    else
                    {
                        return -1;
                    }

                    if (isColumnWritten == true) //写入DataTable的列名
                    {
                        IRow row = sheet.CreateRow(0);
                        for (j = 0; j < data.Columns.Count; ++j)
                        {
                            row.CreateCell(j).SetCellValue(data.Columns[j].ColumnName);
                        }
                        count = 1;
                    }
                    else
                    {
                        count = 0;
                    }

                    for (i = 0; i < data.Rows.Count; ++i)
                    {
                        IRow row = sheet.CreateRow(count);
                        for (j = 0; j < data.Columns.Count; ++j)
                        {
                            row.CreateCell(j).SetCellValue(data.Rows[i][j].ToString());
                        }
                        ++count;
                    }
                    workbook.Write(fs); //写入到excel
                    workbook.Close();
                    return count;
                }
                catch (Exception ex)
                {
                    workbook?.Close();
                    throw new Exception(ex.Message);
                }
            }
        }

        /// <summary>
        /// 将excel中的数据导入到DataTable中
        /// </summary>
        /// <param name="sheetName">excel工作薄sheet的名称</param>
        /// <param name="isFirstRowColumn">第一行是否是DataTable的列名</param>
        /// <returns>返回的DataTable</returns>
        public static DataTable ExcelToDataTable(string sheetName, bool isFirstRowColumn, string fileName)
        {
            ISheet sheet = null;
            DataTable data = new DataTable();
            int startRow = 0;
            try
            {
                IWorkbook workbook = null;
                FileStream fs = new FileStream(fileName, FileMode.Open, FileAccess.Read);
                if (fileName.IndexOf(".xlsx") > 0) // 2007版本
                    workbook = new XSSFWorkbook(fs);
                else if (fileName.IndexOf(".xls") > 0) // 2003版本
                    workbook = new HSSFWorkbook(fs);

                if (sheetName != null)
                {
                    sheet = workbook.GetSheet(sheetName);
                    if (sheet == null) //如果没有找到指定的sheetName对应的sheet，则尝试获取第一个sheet
                    {
                        sheet = workbook.GetSheetAt(0);
                    }
                }
                else
                {
                    sheet = workbook.GetSheetAt(0);
                }
                if (sheet != null)
                {
                    IRow firstRow = sheet.GetRow(0);
                    int cellCount = firstRow.LastCellNum; //一行最后一个cell的编号 即总的列数

                    if (isFirstRowColumn)
                    {
                        for (int i = firstRow.FirstCellNum; i < cellCount; ++i)
                        {
                            ICell cell = firstRow.GetCell(i);
                            if (cell != null)
                            {
                                string cellValue = cell.StringCellValue;
                                if (cellValue != null)
                                {
                                    DataColumn column = new DataColumn(cellValue);
                                    data.Columns.Add(column);
                                }
                            }
                        }
                        startRow = sheet.FirstRowNum + 1;
                    }
                    else
                    {
                        startRow = sheet.FirstRowNum;
                    }

                    //最后一列的标号
                    int rowCount = sheet.LastRowNum;
                    for (int i = startRow; i <= rowCount; ++i)
                    {
                        IRow row = sheet.GetRow(i);
                        if (row == null) continue; //没有数据的行默认是null　　　　　　　

                        DataRow dataRow = data.NewRow();
                        for (int j = row.FirstCellNum; j < cellCount; ++j)
                        {
                            if (row.GetCell(j) != null) //同理，没有数据的单元格都默认是null
                                dataRow[j] = row.GetCell(j).ToString();
                        }
                        data.Rows.Add(dataRow);
                    }
                }

                return data;
            }
            catch (Exception ex)
            {
                Console.WriteLine("Exception: " + ex.Message);
                return null;
            }
        }


        /// <summary>
        /// 将excel中的数据，根据sheet分别导入到DataTable中
        /// </summary>
        /// <param name="fileName">文件全路径</param>
        /// <param name="sheetName">excel工作薄sheet的名称</param>
        /// <param name="isFirstRowColumn">第一行是否是DataTable的列名</param>
        /// <returns>返回的DataTable</returns>
        public static Dictionary<string, DataTable> ExcelToDataTablesBySheet(string fileName, bool isFirstRowColumn, Dictionary<string, string[]> columnTemplate = null)
        {
            ISheet sheet = null;
            IWorkbook workbook = null;
            int startRow = 0;
            var dtDictionary = new Dictionary<string, DataTable>();
            try
            {
                using (var fs = new FileStream(fileName, FileMode.Open, FileAccess.Read))
                {
                    if (fileName.IndexOf(".xlsx", StringComparison.Ordinal) > 0) // 2007版本
                        workbook = new XSSFWorkbook(fs);
                    else if (fileName.IndexOf(".xls", StringComparison.Ordinal) > 0) // 2003版本
                        workbook = new HSSFWorkbook(fs);
                }

                if (workbook != null)
                {
                    var sheetCount = workbook.NumberOfSheets;
                    for (int k = 0; k < sheetCount; k++)
                    {
                        var data = new DataTable();
                        sheet = workbook.GetSheetAt(k);
                        if (sheet != null)
                        {
                            var sheetName = sheet.SheetName;
                            IRow firstRow = sheet.GetRow(0);
                            int cellCount = firstRow.LastCellNum; //一行最后一个cell的编号 即总的列数

                            if (isFirstRowColumn)
                            {
                                for (int i = firstRow.FirstCellNum; i < cellCount; ++i)
                                {
                                    ICell cell = firstRow.GetCell(i);
                                    string cellValue = cell?.StringCellValue?.Trim();
                                    if (cellValue != null)
                                    {
                                        if (columnTemplate != null && !columnTemplate.First().Value.Contains(cellValue))
                                            throw new Exception($"{columnTemplate.First().Key}不存在列名：{cellValue}！正确列名为：{string.Join(",", columnTemplate.First().Value)}");
                                        DataColumn column = new DataColumn(cellValue);
                                        data.Columns.Add(column);
                                    }
                                }
                                startRow = sheet.FirstRowNum + 1;
                            }
                            else
                            {
                                startRow = sheet.FirstRowNum;
                            }

                            //最后一列的标号
                            //var isAddRow = true;
                            int rowCount = sheet.LastRowNum;
                            for (int i = startRow; i <= rowCount; ++i)
                            {
                                //isAddRow = true;
                                IRow row = sheet.GetRow(i);
                                if (row == null) continue; //没有数据的行默认是null　　　　　　　

                                DataRow dataRow = data.NewRow();
                                for (int j = row.FirstCellNum; j < cellCount; ++j)
                                {
                                    var cellvalue = row.GetCell(j);
                                    if (cellvalue != null)
                                    {
                                        //if (cellvalue.ToString() != "eBay" && data.Columns[j].ColumnName == "渠道")
                                        //{
                                        //    isAddRow = false;
                                        //    break;
                                        //}
                                        //同理，没有数据的单元格都默认是null
                                        dataRow[j] = cellvalue.ToString().Trim();
                                    }
                                }
                                //if (!isAddRow)
                                //{
                                //    continue;
                                //}
                                data.Rows.Add(dataRow);
                            }
                            if (dtDictionary.ContainsKey(sheetName))
                            {
                                throw new Exception($"sheet名称重复：{sheetName}，请规范命名！");
                            }
                            dtDictionary.Add(sheetName, data);
                        }
                    }
                }

                workbook?.Close();
                return dtDictionary;
            }
            catch (Exception ex)
            {
                workbook?.Close();
                throw new Exception(ex.Message);
            }
        }

        /// <summary>
        /// 初始化导出表格样式
        /// </summary>
        /// <param name="book">Excel文件</param>
        /// <param name="param"></param>
        /// <returns></returns>
        public static ISheet CreateSheet(HSSFWorkbook book, Dictionary<string, int> param, string sheetName = "Sheet1")
        {
            //添加一个sheet
            ISheet sheet1 = book.CreateSheet(sheetName);
            IRow row = sheet1.CreateRow(0);

            //初始化样式
            ICellStyle mStyle = book.CreateCellStyle();
            mStyle.Alignment = HorizontalAlignment.Center;
            mStyle.VerticalAlignment = VerticalAlignment.Center;
            IFont mfont = book.CreateFont();
            mfont.FontHeight = 10 * 20;
            mStyle.SetFont(mfont);

            int i = 0;
            foreach (var item in param)
            {
                //设置列宽
                sheet1.SetColumnWidth(i, item.Value * 256);
                sheet1.SetDefaultColumnStyle(i, mStyle);
                row.CreateCell(i).SetCellValue(item.Key.ToString());
                i++;
            }
            i = 0;
            sheet1.GetRow(0).Height = 28 * 20;
            return sheet1;
        }

        /// <summary>
        ///  初始化导出表格样式 --带表头
        /// </summary>
        /// <param name="book"></param>
        /// <param name="param"></param>
        /// <param name="strTitle"></param>
        /// <param name="sheetName"></param>
        /// <returns></returns>
        public static ISheet CreateSheet(HSSFWorkbook book, Dictionary<string, int> param, string strTitle, string sheetName = "Sheet1")
        {
            //添加一个sheet
            ISheet sheet1 = book.CreateSheet(sheetName);
            IRow row = sheet1.CreateRow(0);
            IRow row1 = sheet1.CreateRow(1);

            ICell cell = row.CreateCell(0);
            cell.SetCellValue(strTitle);
            ICellStyle style = book.CreateCellStyle();
            style.Alignment = HorizontalAlignment.Center;
            style.VerticalAlignment = VerticalAlignment.Center;
            IFont font = book.CreateFont();
            font.FontHeight = 16 * 20;
            style.SetFont(font);
            cell.CellStyle = style;
            var region = new CellRangeAddress(0, 0, 0, param.Count - 1);
            sheet1.AddMergedRegion(region);
            sheet1.GetRow(0).Height = 20 * 20;

            //初始化样式
            ICellStyle mStyle = book.CreateCellStyle();
            mStyle.Alignment = HorizontalAlignment.Center;
            mStyle.VerticalAlignment = VerticalAlignment.Center;
            IFont mfont = book.CreateFont();
            mfont.FontHeight = 10 * 20;
            mStyle.SetFont(mfont);

            int i = 0;
            foreach (var item in param)
            {
                //设置列宽
                sheet1.SetColumnWidth(i, item.Value * 256);
                sheet1.SetDefaultColumnStyle(i, mStyle);
                row1.CreateCell(i).SetCellValue(item.Key.ToString());
                i++;
            }
            i = 0;
            sheet1.GetRow(1).Height = 16 * 20;
            return sheet1;
        }

        /// <summary>
        /// 初始化导出表格样式
        /// </summary>
        /// <param name="book">Excel文件</param>
        /// <param name="param"></param>
        /// <returns></returns>
        public static ISheet CreateSheet2007(XSSFWorkbook book, Dictionary<string, int> param, string sheetName = "Sheet1")
        {
            //添加一个sheet
            ISheet sheet1 = book.CreateSheet(sheetName);
            IRow row = sheet1.CreateRow(0);

            //初始化样式
            ICellStyle mStyle = book.CreateCellStyle();
            mStyle.Alignment = HorizontalAlignment.Center;
            mStyle.VerticalAlignment = VerticalAlignment.Center;
            IFont mfont = book.CreateFont();
            mfont.FontHeight = 10 * 20;
            mStyle.SetFont(mfont);

            int i = 0;
            foreach (var item in param)
            {
                //设置列宽
                sheet1.SetColumnWidth(i, item.Value * 256);
                sheet1.SetDefaultColumnStyle(i, mStyle);
                row.CreateCell(i).SetCellValue(item.Key.ToString());
                i++;
            }
            i = 0;
            sheet1.GetRow(0).Height = 28 * 20;
            return sheet1;
        }

        /// <summary>
        /// 合并单元格
        /// </summary>
        /// <param name="sheet"></param>
        /// <param name="firstRow"></param>
        /// <param name="lastRow"></param>
        /// <param name="firstCell"></param>
        /// <param name="lastCell"></param>
        public static void MergeCell(ISheet sheet, int firstRow, int lastRow, int firstCell, int lastCell)
        {
            var region = new CellRangeAddress(firstRow, lastRow, firstCell, lastCell);
            sheet.AddMergedRegion(region);
            ((HSSFSheet)sheet).SetEnclosedBorderOfRegion(region, BorderStyle.Thin, NPOI.HSSF.Util.HSSFColor.COLOR_NORMAL);
        }
        /// <summary>
        /// 根据Excel列类型获取列的值
        /// </summary>
        /// <param name="cell">Excel列</param>
        /// <returns></returns>
        public static string GetCellValue(ICell cell)
        {
            if (cell == null)
                return string.Empty;
            switch (cell.CellType)
            {
                case CellType.Blank:
                    return string.Empty;
                case CellType.Boolean:
                    return cell.BooleanCellValue.ToString();
                case CellType.Error:
                    return cell.ErrorCellValue.ToString();
                case CellType.Numeric:
                case CellType.Unknown:
                default:
                    return cell.ToString();
                case CellType.String:
                    return cell.StringCellValue;
                case CellType.Formula:
                    try
                    {
                        HSSFFormulaEvaluator e = new HSSFFormulaEvaluator(cell.Sheet.Workbook);
                        e.EvaluateInCell(cell);
                        return cell.ToString();
                    }
                    catch
                    {
                        return cell.NumericCellValue.ToString();
                    }
            }
        }

        public static DataTable ImportExcelFile(string filePath)
        {
            IWorkbook hssfworkbook;
            try
            {
                using (FileStream mfile = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    if (filePath.EndsWith(".xls"))
                    {
                        hssfworkbook = new HSSFWorkbook(mfile);
                    }
                    else
                    {
                        hssfworkbook = new XSSFWorkbook(mfile);//加入支持office2007+
                    }
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }

            ISheet sheet = hssfworkbook.GetSheetAt(0);
            DataTable table = new DataTable();
            IRow headerRow = sheet.GetRow(0);//第一行为标题行
            int cellCount = headerRow.LastCellNum;
            int rowCount = sheet.LastRowNum;

            for (int i = headerRow.FirstCellNum; i < cellCount; i++)
            {
                DataColumn column = new DataColumn(headerRow.GetCell(i).StringCellValue);
                table.Columns.Add(column);
            }
            for (int i = (sheet.FirstRowNum + 1); i <= rowCount; i++)
            {
                IRow row = sheet.GetRow(i);
                DataRow dataRow = table.NewRow();

                if (row != null)
                {
                    for (int j = row.FirstCellNum; j < cellCount; j++)
                    {
                        if (row.GetCell(j) != null)
                            dataRow[j] =GetCellValue(row.GetCell(j));
                    }
                }

                table.Rows.Add(dataRow);
            }
            return table;
        }

        /// <summary>
        /// 将DataTable中数据写入到CSV文件中
        /// </summary>
        /// <param name="dt">提供保存数据的DataTable</param>
        /// <param name="fileName">CSV的文件路径</param>
        public static void SaveCSV(DataTable dt, string fileName)
        {
            FileStream fs = new FileStream(fileName, System.IO.FileMode.Create, System.IO.FileAccess.Write);
            StreamWriter sw = new StreamWriter(fs, System.Text.Encoding.Default);
            string data = "";

            //写出列名称
            for (int i = 0; i < dt.Columns.Count; i++)
            {
                data += dt.Columns[i].ColumnName.ToString();
                if (i < dt.Columns.Count - 1)
                {
                    data += ",";
                }
            }
            sw.WriteLine(data);

            //写出各行数据
            for (int i = 0; i < dt.Rows.Count; i++)
            {
                data = "";
                for (int j = 0; j < dt.Columns.Count; j++)
                {
                    data += ToStr(dt.Rows[i][j]);
                    if (j < dt.Columns.Count - 1)
                    {
                        data += ",";
                    }
                }
                sw.WriteLine(data);
            }

            sw.Close();
            fs.Close();

        }

        /// <summary>
        /// 将DataTable中数据写入到CSV文件中
        /// </summary>
        /// <param name="dt">提供保存数据的DataTable</param>
        /// <param name="fileName">CSV的文件路径</param>
        public static void AppendCSV(DataTable dt, string fileName)
        {
            FileStream fs = new FileStream(fileName, System.IO.FileMode.Append, System.IO.FileAccess.Write);
            StreamWriter sw = new StreamWriter(fs, System.Text.Encoding.Default);
            string data = "";

            //写出各行数据
            for (int i = 0; i < dt.Rows.Count; i++)
            {
                data = "";
                for (int j = 0; j < dt.Columns.Count; j++)
                {
                    data += ToStr(dt.Rows[i][j]);
                    if (j < dt.Columns.Count - 1)
                    {
                        data += ",";
                    }
                }
                sw.WriteLine(data);
            }

            sw.Close();
            fs.Close();

        }

        private static string ToStr(object obj)
        {
            if (obj == null) return "";
            return obj.ToString().Replace(",", "*").Replace("\n", "").Replace("\r", "").Replace("\"","");
        }
    }

    //新建类 重写Npoi流方法
    public class NpoiMemoryStream : MemoryStream
    {
        public NpoiMemoryStream()
        {
            AllowClose = true;
        }

        public bool AllowClose { get; set; }

        public override void Close()
        {
            if (AllowClose)
                base.Close();
        }
    }
}