﻿using System;
using System.Collections.Generic;
using System.Linq;
using Bailun.Cache;
using BailunRedis;
using CommonControl;

namespace Bailun.Cache
{
    public class RedisCacheService
    {
        private static readonly object SyncObj = new object();

        /// <summary>
        /// 获取指定名称的所有缓存集合，如果缓存中不存在，则从数据库中读取
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name">缓存名称</param>
        /// <param name="keyFieldName">作为键的缓存对象属性名称，例如字段名称：Id</param>
        /// <param name="factory">如果缓存中不存在，从数据库中获取的方法</param>
        /// <param name="timeSpan">缓存过期时间</param>
        /// <returns></returns>
        public static List<T> GetList<T>(string name, string keyFieldName, Func<List<T>> factory, TimeSpan timeSpan = default(TimeSpan)) where T : class
        {
            var cacheKey = name;
            var items = RedisBase.HashGetAll<T>(cacheKey);
            if (items == null)
            {
                lock (SyncObj)
                {
                    try
                    {
                        items = RedisBase.HashGetAll<T>(cacheKey);
                        if (items == null)
                        {
                            if (factory != null) items = factory();
                            if (items == null || !items.Any())
                            {
                                return null;
                            }
                            foreach (var item in items)
                            {
                                if (item != null)
                                {
                                    var keyFieldValue = item.GetPropertyValue(keyFieldName);
                                    if (keyFieldValue != null)
                                    {
                                        Add(cacheKey, keyFieldValue.ToString(), item);
                                    }
                                }
                            }
                            if (timeSpan != default(TimeSpan))
                            {
                                SetExpire(cacheKey, timeSpan);
                            }
                        }
                    }
                    catch
                    {
                        //报错则直接读取数据库
                        items = factory();
                    }
                }
            }
            return items;
        }

        /// <summary>
        /// 获取单个缓存对象
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name">缓存名称</param>
        /// <param name="key">缓存对象的键</param>
        /// <param name="factory">如果缓存中不存在，从数据库中获取的方法</param>
        /// <returns></returns>
        public static T Get<T>(string name, object key, Func<object, T> factory) where T : class
        {
            var cacheName = name;
            var cacheKey = key.ToString();
            var item = RedisBase.HashGet<T>(cacheName, cacheKey);
            if (item == null)
            {
                lock (SyncObj)
                {
                    item = RedisBase.HashGet<T>(cacheName, cacheKey);
                    if (item == null)
                    {
                        if (factory != null) item = factory(key);
                        if (item == null)
                        {
                            return null;
                        }
                        Add(cacheName, cacheKey, item);
                    }
                }
            }
            return item;
        }

        /// <summary>
        /// 获取单个缓存对象
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name">缓存名称</param>
        /// <param name="factory">如果缓存中不存在，从数据库中获取的方法</param>
        /// <param name="timeSpan">缓存过期时间</param>
        /// <returns></returns>
        public static T Get<T>(string name, Func<T> factory, TimeSpan timeSpan = default(TimeSpan)) where T : class
        {
            var cacheName = name;
            var item = RedisBase.ItemGet<T>(cacheName);
            if (item == null)
            {
                lock (SyncObj)
                {
                    item = RedisBase.ItemGet<T>(cacheName);
                    if (item == null)
                    {
                        if (factory != null) item = factory();
                        if (item == null)
                        {
                            return null;
                        }
                        Add(cacheName, item);
                        if (timeSpan != default(TimeSpan))
                        {
                            SetExpire(cacheName, timeSpan);
                        }
                    }
                }
            }
            return item;
        }
        /// <summary>
        /// 新增缓存对象
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name">缓存名称</param>
        /// <param name="key">缓存键</param>
        /// <param name="value">缓存对象值</param>
        public static void Add<T>(string name, string key, T value)
        {
            if (value != null)
            {
                RedisBase.HashSet<T>(name, key, value);
            }
        }
        /// <summary>
        /// 新增缓存对象，无键值
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name">缓存名称</param>
        /// <param name="value">缓存对象值</param>
        public static void Add<T>(string name, T value)
        {
            if (value != null)
            {
                RedisBase.ItemSet(name, value);
            }
        }

        /// <summary>
        /// 新增缓存对象，附带过期时间
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name">缓存名称</param>
        /// <param name="value">缓存对象值</param>
        /// <param name="cacheTime">过期时间</param>
        public static void AddWithExpire<T>(string name, T value, TimeSpan cacheTime)
        {
            Add<T>(name, value);
            SetExpire(name, cacheTime);
        }
        private static void SetExpire(string name, TimeSpan cacheTime)
        {
            RedisBase.SetExpire(name, cacheTime);
        }
        /// <summary>
        /// 移除指定名称的缓存集合
        /// </summary>
        /// <param name="name"></param>
        public static void RemoveList(string name)
        {
            RedisBase.HashRemove(name);
        }

        /// <summary>
        /// 移除单个Hash缓存对象
        /// </summary>
        /// <param name="name"></param>
        /// <param name="key"></param>
        public static void Remove(string name, string key)
        {
            RedisBase.HashRemove(name, key);
        }
        /// <summary>
        /// 移除单个缓存对象
        /// </summary>
        /// <param name="name"></param>
        /// <param name="key"></param>
        public static void Remove(string name)
        {
            RedisBase.ItemRemove(name);
        }
        /// <summary>
        /// 更新单个缓存对象
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name"></param>
        /// <param name="key"></param>
        /// <param name="value"></param>
        public static void Update<T>(string name, string key, T value)
        {
            Remove(name, key);
            Add(name, key, value);
        }
    }
}