package com.gogirl.application.market.coupon.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gogirl.application.market.coupon.CouponCustomerRelevanceService;
import com.gogirl.application.market.coupon.MarketService;
import com.gogirl.domain.market.coupon.Coupon;
import com.gogirl.domain.market.coupon.CouponCustomerRelevance;
import com.gogirl.domain.market.discount.DiscountConfig;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.market.coupon.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.coupon.CouponMapper;
import com.gogirl.infrastructure.mapper.market.discount.DiscountConfigMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderManageMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.shared.market.SetUpOuterCouponCommand;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class CouponCustomerRelevanceServiceImpl extends ServiceImpl<CouponCustomerRelevanceMapper, CouponCustomerRelevance> implements CouponCustomerRelevanceService {

    @Resource
    private OrderServeMapper orderServeMapper;
    @Resource
    private OrderManageMapper orderManageMapper;
    @Resource
    private CouponCustomerRelevanceMapper couponCustomerRelevanceMapper;
    @Resource
    private CouponMapper couponMapper;
    @Resource
    private MarketService marketService;
    @Resource
    private DiscountConfigMapper discountConfigMapper;

    @Override
    public List<CouponCustomerRelevance> getOrderCoupon(Integer currentCustomerId, Integer orderId) {

        /*1、查询订单服务和订单*/
        List<OrderServe> orderServeList = orderServeMapper.selectList(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));
        OrderManage orderManage = orderManageMapper.selectById(orderId);
        return couponCustomerRelevanceMapper
                .selectList(new LambdaQueryWrapper<CouponCustomerRelevance>()
                        .eq(CouponCustomerRelevance::getCustomerId, currentCustomerId)
                        //只查询内部券
                        .eq(CouponCustomerRelevance::getSourceType, 0)
                        //只查询正常状态的卡券
                        .eq(CouponCustomerRelevance::getState, 1))
                //
                .stream()
                //只返回可以使用的优惠券
                .filter(couponCustomerRelevance -> {
                    //卡券作用的服务id
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(couponCustomerRelevance.getCouponId());
                    //订单的服务id
                    List<Integer> orderServeIds = orderServeList.stream().map(OrderServe::getServeId).collect(Collectors.toList());
                    //retail
                    List<Integer> retainIds = ListUtil.retainAll(canUserServeIds, orderServeIds);
                    return ListUtil.isNotEmpty(retainIds);
                })
                .peek(couponCustomerRelevance -> {
                    //卡券作用的服务id (mybatis 缓存)
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(couponCustomerRelevance.getCouponId());
                    //免单券
                    List<Integer> freeCouponIds = discountConfigMapper.selectList(new LambdaQueryWrapper<>()).stream().map(DiscountConfig::getCouponId).filter(Objects::nonNull).collect(Collectors.toList());
                    if (freeCouponIds.contains(couponCustomerRelevance.getCouponId())) {
                        canUserServeIds.retainAll(orderServeList.stream()
                                .filter(orderServe -> orderServe.getBindOuterCoupon() == 2)
                                .filter(orderServe -> orderServe.getBindTimesCard() == 2)
                                .map(OrderServe::getServeId)
                                .collect(Collectors.toList()));
                    } else {
                        canUserServeIds.retainAll(orderServeList.stream()
                                .filter(orderServe -> orderServe.getBindOuterCoupon() == 2)
                                .filter(orderServe -> orderServe.getBindTimesCard() == 2)
                                //不是活动价格
                                .filter(orderServe -> !orderServe.isActivity())
                                .map(OrderServe::getServeId)
                                .collect(Collectors.toList()));
                    }
                    //满足使用条件但是不满足冲突条件的卡券
                    couponCustomerRelevance.setCanBeUse(ListUtil.isNotEmpty(canUserServeIds));
                })
                //返回校验通过的卡券
                .filter(CouponCustomerRelevance::isValid)
                //返回到到使用金额条件的卡券
                .filter(couponCustomerRelevance -> couponCustomerRelevance.getReachingAmount().compareTo(orderManage.getTotalPrice()) < 0)
                .collect(Collectors.toList());
    }

    @Override
    public IPage<CouponCustomerRelevance> selectMyCoupon(Integer customerId,
                                                         Integer pageNum,
                                                         Integer pageSize,
                                                         Integer state) {

        IPage<CouponCustomerRelevance> page = new Page<>(pageNum, pageSize);
        page = couponCustomerRelevanceMapper.selectPage(page, new LambdaQueryWrapper<CouponCustomerRelevance>()
                .eq(CouponCustomerRelevance::getCustomerId, customerId)
                .eq(CouponCustomerRelevance::getState, state)
                .eq(CouponCustomerRelevance::getSourceType, CouponCustomerRelevance.SOURCE_TYPE_INNER));
        this.joinCouponCustomerRelevanceMapper(page.getRecords());

        /*校验是否可用*/
        List<CouponCustomerRelevance> records = page.getRecords()
                .stream()
                .peek(couponCustomerRelevance -> {
                    if (couponCustomerRelevance.getState() == 1 && !couponCustomerRelevance.isValid()) {
                        couponCustomerRelevance.setState(3);
                        couponCustomerRelevanceMapper.updateById(couponCustomerRelevance);
                    }
                })
                .filter(couponCustomerRelevance -> couponCustomerRelevance.getState().equals(state))
                .collect(Collectors.toList());

        page.setRecords(records);

        return page;
    }

    @Override
    public void techSubmitCoupons(Integer orderId, List<Integer> couponIds) {
        //查询订单
        OrderManage orderManage = orderManageMapper.selectById(orderId);

        SetUpOuterCouponCommand cmd = SetUpOuterCouponCommand.builder()
                .customerId(orderManage.getOrderUser())
                .orderId(orderId)
                .outerCouponIds(couponIds)
                .build();
        marketService.setUpOuterCoupon(cmd);
        marketService.calcOrderAmount(orderId);
    }

    private void joinCouponCustomerRelevanceMapper(List<CouponCustomerRelevance> couponCustomerRelevanceList) {
        List<Integer> ids = couponCustomerRelevanceList.stream().map(CouponCustomerRelevance::getCouponId).collect(Collectors.toList());
        List<Coupon> couponList = couponMapper.selectList(
                new LambdaQueryWrapper<Coupon>().in(Coupon::getId, ids)
        );
        Map<Integer, List<Coupon>> couponMap = couponList.stream().collect(Collectors.groupingBy(Coupon::getId));
        couponCustomerRelevanceList.forEach(couponCustomerRelevance -> {
            if (ListUtil.isNotEmpty(couponMap.get(couponCustomerRelevance.getCouponId()))) {
                couponCustomerRelevance.setCoupon(couponMap.get(couponCustomerRelevance.getCouponId()).stream().findAny().orElse(null));
            }
        });
    }
}
