package com.gogirl.interfaces.user;

import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.dto.customer.CustomerOrderDetail;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.config.property.GogirlProperties;
import com.gogirl.infrastructure.util.Base64Utils;
import com.gogirl.infrastructure.util.ImageUtils;
import com.gogirl.shared.user.query.qry.CustomerQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.security.InvalidAlgorithmParameterException;
import java.util.List;

@Api(tags = {"用户相关接口"}, value = "用户相关接口")
@RestController
@Slf4j
public class CustomerController {

    @Resource
    private CustomerService customerService;
    @Resource
    private GogirlProperties gogirlProperties;

    @ApiOperation("用户授权接街道地址信息")
    @GetMapping("/customer/xcx/bindAddressInfo")
    public JsonResult<String> bindAddressInfo(@RequestParam Integer customerId,
                                              @RequestParam String province,
                                              @RequestParam String city,
                                              @RequestParam String area,
                                              @RequestParam String street) {
        customerService.bindAddressInfo(customerId, province, city, area, street);
        return JsonResult.success();
    }

    @ApiOperation(("查询会员"))
    @PostMapping("/customer/xcx/query")
    public JsonResult<Customer> query(@RequestBody CustomerQuery qry) {
        Customer customer = customerService.getById(qry.getCustomerId());
        return JsonResult.success(customer);
    }


    @ApiOperation(value = "客户根据code获取token")
    @GetMapping("/customer/xcx/login")
    public JsonResult<String> login(@RequestParam String code) {
        String token = customerService.login(code);
        return JsonResult.success(token);
    }

    @ApiOperation(value = "客户授权后绑定个人信息")
    @RequestMapping(method = {RequestMethod.GET, RequestMethod.POST}, value = "/customer/xcx/authorized1")
    public JsonResult<Customer> authorized1(
            @RequestParam(name = "token", required = false) String token,
            @RequestParam(name = "encryptedData", required = false) String encryptedData,
            @RequestParam(name = "iv", required = false) String iv) throws InvalidAlgorithmParameterException {

        log.info("调用authorized，授权用户信息到程序");
        Customer customer = customerService.authorized1(token, encryptedData, iv);
        return JsonResult.success(customer);
    }

    @ApiOperation(value = "客户授权手机号码")
    @RequestMapping(method = {RequestMethod.POST}, value = "/customer/xcx/authorizedPhone")
    public JsonResult<Customer> authorizedPhone(@RequestHeader String token,
                                                @RequestParam String encryptedData,
                                                @RequestParam String iv) throws InvalidAlgorithmParameterException {
        log.info("手机号码授权,用户信息:");
        Customer customer = customerService.authorizedPhone(token, encryptedData, iv);
        return JsonResult.success(customer);
    }

    @GetMapping("/customer/xcx/getUserInfo")
    @ApiOperation(value = "客户根据token获取用户信息")
    public JsonResult<Customer> getUserInfo(@RequestParam String token) throws Exception {
        log.info("根据token:" + token + ",查询用户信息.");
        Customer customer = customerService.getUserInfo(token);
        return JsonResult.success(customer);
    }

    @ApiOperation(value = "图片上传")
    @PostMapping("/customer/xcx/upload")
    public JsonResult<String> upload(MultipartFile file) throws Exception {
        log.info("图片上传");
        String imgUrl = ImageUtils.saveImage(gogirlProperties.getPicturePath(), file);
        return JsonResult.success(imgUrl);
    }

    @ApiOperation(value = "base64图片上传")
    @PostMapping("/customer/xcx/no_base64Upload")
    public JsonResult<String> base64Upload(@RequestBody String base64) throws Exception {
        log.info("图片上传");
        MultipartFile multipartFile = Base64Utils.base64ToMultipart(base64);
        String imgUrl = ImageUtils.saveImage(gogirlProperties.getPicturePath(), multipartFile);
        return JsonResult.success(imgUrl);
    }

    @ApiOperation(value = "查询用户可选年龄段")
    @PostMapping("/technician/getAgeGroups")
    public JsonResult<List<String>> getAgeGroups() {
        List<String> ageGroups = customerService.getAgeGroups();
        return JsonResult.success(ageGroups);
    }

    @ApiOperation(value = "用户详情数据查看")
    @PostMapping("/technician/queryCustomerDetail")
    public JsonResult<CustomerOrderDetail> queryCustomerDetail(@RequestHeader String token,
                                                               @RequestParam Integer customerId,
                                                               @RequestParam(required = false) Integer orderId) {

        CustomerOrderDetail customerOrderDetail = customerService.queryCustomerDetail(customerId, orderId);
        return JsonResult.success(customerOrderDetail);
    }

    @ApiOperation(value = "用户详情数据修改")
    @PostMapping("/technician/updateCustomerDetail")
    public JsonResult<Void> updateCustomerDetail(@RequestHeader String token,
                                                 @RequestParam(required = false) Integer birthdayMonth,
                                                 @RequestParam(required = false) Integer birthdayDay,
                                                 @RequestParam(required = false) String ageGroup,
                                                 @RequestParam Integer customerId,
                                                 @RequestParam(required = false) Integer orderId,
                                                 @RequestParam(required = false) Integer orderServeId,
                                                 @RequestParam(required = false) String storeRecordRealName,
                                                 @RequestParam(required = false) String sex,
                                                 @RequestParam(required = false) Integer age,
                                                 @RequestParam(required = false) String job,
                                                 @RequestParam(required = false) String preference,
                                                 @RequestParam(required = false) String character,
                                                 @RequestParam(required = false) Integer customerSource) {
        customerService.updateCustomerDetail(birthdayMonth, birthdayDay, ageGroup, customerId, orderId, orderServeId, storeRecordRealName, sex, age, job, preference, character, customerSource);
        return JsonResult.success();
    }

    @ApiOperation(value = "h5用户登录")
    @GetMapping("/customer/no_h5Login")
    public JsonResult<String> h5Login(@RequestParam Integer orderId) {
        String token = customerService.h5Login(orderId);
        return JsonResult.success(token);
    }

    @ApiOperation(value = "超时测试接口")
    @RequestMapping("/customer/testTimeOut")
    public JsonResult<Void> testTimeOut() throws InterruptedException {
        Thread.sleep(36000L);
        return JsonResult.success();
    }
}
