package com.gogirl.application.user.customer.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.application.xcx.QRCodeService;
import com.gogirl.application.xcx.WechatService;
import com.gogirl.domain.order.serve.ScheduleManage;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.domain.user.customer.CustomerDetail;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.common.util.StringUtils;
import com.gogirl.infrastructure.mapper.order.serve.ScheduleManageMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerDetailMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerMapper;
import com.gogirl.infrastructure.util.WXCore;
import com.gogirl.infrastructure.util.lock.Openid1Lock;
import com.gogirl.shared.member.wx.query.dto.Code2SessionResult;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.springframework.stereotype.Service;

import java.security.InvalidAlgorithmParameterException;
import java.util.Date;
import java.util.List;

@Service
@AllArgsConstructor
@Slf4j
public class CustomerServiceImpl extends ServiceImpl<CustomerMapper, Customer> implements CustomerService {

    private final CustomerMapper customerMapper;
    private final WechatService wechatService;

    @Override
    public Customer selectByPhone(String phone) {
        return customerMapper.selectByPhone(phone);
    }

    @Override
    public Customer selectByCustomerId(Integer scheduledUser) {
        return customerMapper.selectById(scheduledUser);
    }

    @Override
    public String login(String code) {

        log.info("调用login，授权用户信息到程序 code:{}", code);

        Openid1Lock lock = Openid1Lock.getInsatance();


        Code2SessionResult result = wechatService.getCustomerOpenidByCode(code);

        String token = IdWorker.getIdStr();

        //创建token
        GogirlToken gogirlToken = GogirlToken.builder()
                .code(code)
                .sysId(1)
                .openid(result.getOpenid())
                .sessionKey(result.getSessionKey())
                .createTime(new Date())
                .updateTime(new Date())
                .unionid(result.getUnionid())
                .token(token)
                .build();

        String openid = gogirlToken.getOpenid();

        try {
            lock.lock(openid1GetInt(openid));
            //变量为当前登陆用户
            Customer customer;

            customer = customerMapper.selectOne(new LambdaQueryWrapper<Customer>().eq(Customer::getOpenid1, openid));
            //不存在小程序用户
            if (customer == null) {
                //token unionid 不为空
                if (StringUtils.isNotEmpty(result.getUnionid())) {

                    customer = customerMapper.selectOne(new LambdaQueryWrapper<Customer>().eq(Customer::getUnionid, result.getUnionid()));
                    //不存在公众号用户
                    if (customer == null) {
                        customer = Customer.builder()
                                .customerSource(4)
                                //小程序openid
                                .openid1(openid)
                                .orderTimes(0)
                                .registerTime(new Date())
                                .scheduledTimes(0)
                                //微信授权
                                .source(3)
                                //状态 正常
                                .state("1")
                                .updateTime(new Date())
                                .build();

                        customerMapper.insert(customer);
                    }
                    // 存在公众号用户
                    else {
                        customer.setOpenid1(openid);
                        customerMapper.updateById(customer);
                    }
                }
                //union id为空
                else {
                    customer = Customer.builder()
                            .customerSource(4)
                            //小程序openid
                            .openid1(openid)
                            .orderTimes(0)
                            .registerTime(new Date())
                            .scheduledTimes(0)
                            //微信授权
                            .source(3)
                            //状态 正常
                            .state("1")
                            .updateTime(new Date())
                            .build();
                    customerMapper.insert(customer);
                }

            }

            gogirlToken.setCustomerId(customer.getId());
            gogirlTokenService.save(gogirlToken);

        } finally {
            lock.unlock(openid1GetInt(openid));
        }
        return token;
    }

    @Override
    public void bindAddressInfo(Integer customerId, String province, String city, String area, String street) {
        Customer customer = customerMapper.selectById(customerId);

        customer.setProvince(province);
        customer.setCity(city);
        customer.setArea(area);
        customer.setStreet(street);
        customerMapper.updateById(customer);
    }

    private final GogirlTokenService gogirlTokenService;

    @Override
    public Customer authorized1(String token, String encryptedData, String iv) throws InvalidAlgorithmParameterException {

        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);
        JSONObject data = WXCore.decrypt(encryptedData, gogirlToken.getSessionKey(), iv);
        String unionid = (String) data.get("unionId");
        String openid = (String) data.get("openId");
        Integer gender = (Integer) data.get("gender");
        String avatarUrl = (String) data.get("avatarUrl");
        String city = (String) data.get("city");
        String country = (String) data.get("country");
        String province = (String) data.get("province");
        String nickName = filterEmoji((String) data.get("nickName"));

        Customer customer = customerMapper.selectOne(new LambdaQueryWrapper<Customer>().eq(Customer::getOpenid1, openid));

        customer.setSex(gender == 1 ? "男" : gender == 2 ? "女" : "未知");
        customer.setHeadimgurl(avatarUrl);
        customer.setNickname(nickName);
        customer.setProvince(province);
        customer.setCountry(country);
        customer.setCity(city);
        customer.setUnionid(unionid);
        customerMapper.updateById(customer);
        return customer;
    }

    private final ScheduleManageMapper scheduleManageMapper;

    @Override
    public Customer authorizedPhone(String token, String encryptedData, String iv) throws InvalidAlgorithmParameterException {
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);

        JSONObject data = WXCore.decrypt(encryptedData, gogirlToken.getSessionKey(), iv);

        String phone = (String) data.get("purePhoneNumber");

        gogirlToken.setPhone(phone);
        gogirlTokenService.updateByToken(gogirlToken);

        Customer customer = customerMapper.selectById(gogirlToken.getCustomerId());


        Customer phoneCustomer = customerMapper.selectOne(new LambdaQueryWrapper<Customer>().eq(Customer::getPhone, phone));

        //如果存在手机号
        if (phoneCustomer != null) {

            phoneCustomer.setSex(customer.getSex());
            phoneCustomer.setHeadimgurl(customer.getHeadimgurl());
            phoneCustomer.setNickname(customer.getNickname());
            phoneCustomer.setProvince(customer.getProvince());
            phoneCustomer.setCountry(customer.getCountry());
            phoneCustomer.setCity(customer.getCity());
            phoneCustomer.setUnionid(customer.getUnionid());

            phoneCustomer.setCustomerSource(customer.getCustomerSource());
            phoneCustomer.setOpenid1(customer.getOpenid1());
            phoneCustomer.setRegisterTime(customer.getRegisterTime());
            phoneCustomer.setScheduledTimes(customer.getScheduledTimes());
            phoneCustomer.setSource(customer.getSource());
            phoneCustomer.setState(customer.getState());
            phoneCustomer.setUpdateTime(new Date());

            customerMapper.updateById(phoneCustomer);
            customerMapper.deleteById(customer);

            List<ScheduleManage> scheduleManageList = scheduleManageMapper.selectList(new LambdaQueryWrapper<ScheduleManage>()
                    .eq(ScheduleManage::getTelephone, phone));
            if (ListUtil.isNotEmpty(scheduleManageList)) {
                scheduleManageList.forEach(scheduleManage -> {
                    scheduleManage.setTelephone(phone);
                    scheduleManageMapper.updateById(scheduleManage);
                });
            }
            //清除token
            gogirlTokenService.remove(token);
            return phoneCustomer;
        }
        //不存在手机号
        else {
            customer.setPhone(phone);
            customerMapper.updateById(customer);
            return customer;
        }
    }

    private final CustomerDetailMapper customerDetailMapper;
    private final CustomerBalanceMapper customerBalanceMapper;

    private final QRCodeService qrCodeService;

    @Override
    public Customer getUserInfo(String token) throws Exception {
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);

        Customer customer = customerMapper.selectById(gogirlToken.getCustomerId());
        CustomerDetail customerDetail = customerDetailMapper.selectOne(new LambdaQueryWrapper<CustomerDetail>().eq(CustomerDetail::getCustomerId, customer.getId()));
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>().eq(CustomerBalance::getCustomerId, customer.getId()));


        customer.setCustomerDetail(customerDetail);
        customer.setCustomerBalance(customerBalance);

        if (StringUtils.isEmpty(customer.getMyQrcode()) && StringUtils.isNotEmpty(customer.getPhone())) {
            String myQrCode = qrCodeService.myQrCode(customer.getId().toString(), customer.getPhone());
            customer.setMyQrcode(myQrCode);
            customer.setId(customer.getId());
            customer.setMyQrcode(myQrCode);
            customerMapper.updateById(customer);
        }

        return customer;
    }

    //todo 替换成hashcode
    //字符串中所有字符相加得到一个int
    private int openid1GetInt(String openid1) {
        StringBuilder sb = new StringBuilder(openid1);
        int sum = 0;
        int length = sb.length();
        for (int i = 20; i < length; i++) {
            sum += sb.charAt(i);
        }
        return sum;
    }

    /**
     * 将emoji表情替换成空串
     *
     * @param source
     * @return 过滤后的字符串
     */
    private static String filterEmoji(String source) {
        if (source != null && source.length() > 0) {
            return source.replaceAll("[\ud800\udc00-\udbff\udfff\ud800-\udfff]", "");
        } else {
            return source;
        }
    }
}
