package com.gogirl.interfaces.store;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.CareerCmdService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.assembler.CareerDTOAssembler;
import com.gogirl.assembler.TrainingStartClassDTOAssembler;
import com.gogirl.assembler.WorksAssembler;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.store.career.Career;
import com.gogirl.domain.store.career.Works;
import com.gogirl.domain.store.store.*;
import com.gogirl.domain.store.training.SkillSku;
import com.gogirl.domain.store.training.TechnicianScore;
import com.gogirl.domain.store.training.TrainingStartClass;
import com.gogirl.domain.store.training.TrainingStartClassLogTechnician;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.store.store.StoreDataMapper;
import com.gogirl.infrastructure.mapper.store.store.TechnicianScoreMapper;
import com.gogirl.infrastructure.mapper.store.training.SkillSkuMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassLogMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassLogTechnicianMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.shared.store.command.SignUpTrainingCommand;
import com.gogirl.shared.store.command.SubmitFeedbackCommand;
import com.gogirl.shared.store.command.SubmitWorksCommand;
import com.gogirl.shared.store.query.dto.CareerDTO;
import com.gogirl.shared.store.query.dto.TrainingStartClassDTO;
import com.gogirl.shared.store.query.dto.WorksDTO;
import com.gogirl.shared.store.query.qry.TrainingStartClassQuery;
import com.gogirl.shared.store.query.qry.WorksQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.text.ParseException;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 美甲师职业生涯接口
 */
@Api(tags = "version3.0:美甲师职业生涯接口", value = "version3.0:美甲师职业生涯接口")
@RestController
@AllArgsConstructor
public class CareerController {

    private final GogirlTokenService gogirlTokenService;

    private final TrainingStartClassLogMapper trainingStartClassLogMapper;
    private final TechnicianScoreMapper technicianScoreMapper;
    private final TrainingStartClassLogTechnicianMapper trainingStartClassLogTechnicianMapper;
    private final SkillSkuMapper skillSkuMapper;
    private final TrainingStartClassMapper trainingStartClassMapper;


    private final CareerCmdService careerCmdService;
    private final StoreDataMapper storeDataMapper;
    private final OrderServeMapper orderServeMapper;
    private final CustomerBalanceRecordMapper customerBalanceRecordMapper;

    @ApiOperation("提交美甲作品")
    @PostMapping("/technician/career/submitWorks")
    public JsonResult<Void> submitWorks(@RequestHeader String token,
                                        @RequestBody SubmitWorksCommand cmd) {
        Integer technicianId = SessionUtils.getTechnicianId();
        cmd.setTechnicianId(technicianId);
        careerCmdService.submitWorks(cmd);
        return JsonResult.success();
    }

    @ApiOperation(("查询美甲师作品"))
    @PostMapping("/technician/career/queryWorks")
    public JsonResult<Page<WorksDTO>> queryWorks(@RequestHeader String token,
                                                 @RequestBody WorksQuery qry) throws ParseException {
        Integer technicianId = SessionUtils.getTechnicianId();
        qry.setTechnicianId(technicianId);
        IPage<Works> worksIPage = careerCmdService.queryWorks(qry);
        return JsonResult.success((Page<WorksDTO>) worksIPage.convert(worksAssembler));
    }

    private final WorksAssembler worksAssembler;
    private final CareerDTOAssembler careerDTOAssembler;

    @ApiOperation("报名技能培训")
    @PostMapping("/technician/career/signUpTraining")
    public JsonResult<Void> signUpTraining(@RequestBody SignUpTrainingCommand cmd) {
        careerCmdService.signUpTraining(cmd);
        return JsonResult.success();
    }

    @ApiOperation("填写技能培训反馈")
    @PostMapping("/technician/career/submitFeedBack")
    public JsonResult<Void> submitFeedBack(@RequestBody SubmitFeedbackCommand cmd) {
        careerCmdService.submitFeedBack(cmd);
        return JsonResult.success();
    }

    private final TrainingStartClassDTOAssembler trainingStartClassDTOAssembler;

    @ApiOperation("课程列表")
    @PostMapping("/technician/career/queryPageClass")
    public JsonResult<Page<TrainingStartClassDTO>> queryPageClass(@RequestBody TrainingStartClassQuery qry) {
        IPage<TrainingStartClass> page = careerCmdService.queryPageClass(qry);
        return JsonResult.success((Page<TrainingStartClassDTO>) page.convert(trainingStartClassDTOAssembler));
    }

    @ApiOperation("培训记录")
    @GetMapping("/technician/career/queryPageStartClassLog")
    public JsonResult<IPage<TrainingStartClassLogTechnician>> queryPageStartClassLog(@RequestParam Integer pageNum,
                                                                                     @RequestParam Integer pageSize,
                                                                                     @RequestHeader String token) {
        Integer technicianId = SessionUtils.getTechnicianId();
        IPage<TrainingStartClassLogTechnician> pageRequest = new Page<>(pageNum, pageSize);
        IPage<TrainingStartClassLogTechnician> page = trainingStartClassLogTechnicianMapper.selectPage(pageRequest, new LambdaQueryWrapper<TrainingStartClassLogTechnician>().eq(TrainingStartClassLogTechnician::getTechnicianId, technicianId));

        List<Integer> ids = page.getRecords().stream().map(TrainingStartClassLogTechnician::getTrainingStartClassId).collect(Collectors.toList());
        if (ListUtil.isNotEmpty(ids)) {
            List<TrainingStartClass> trainingStartClassesList = trainingStartClassMapper.selectBatchIds(ids);
            page.getRecords().forEach(trainingStartClassLogTechnician -> {
                trainingStartClassLogTechnician.setTrainingStartClass(
                        trainingStartClassesList.stream().filter(trainingStartClass -> trainingStartClass.getId().equals(trainingStartClassLogTechnician.getTrainingStartClassId())).findAny().orElse(null));
            });
        }
        return JsonResult.success(page);
    }

    @ApiOperation("成长历程")
    @GetMapping("/technician/career/career")
    public JsonResult<CareerDTO> career(@RequestHeader String token) {
        Integer technicianId = SessionUtils.getTechnicianId();
        Career career = careerCmdService.career(technicianId);
        return JsonResult.success(careerDTOAssembler.apply(career));
    }

    @ApiOperation("门店数据")
    @GetMapping("/technician/career/storeData")
    public JsonResult<StoreData> storeData(@RequestHeader String token,
                                           @RequestParam String month) {
        Integer technicianId = SessionUtils.getTechnicianId();
        GogirlToken gogirlToken = SessionUtils.getTechnicianToken();
        StoreData storeData = storeDataMapper.selectOne(new LambdaQueryWrapper<StoreData>().eq(StoreData::getDepartmentId, gogirlToken.getDepartmentId()));
        //总订单业绩
        List<StoreTradeAmount> storeTradeAmountList = storeDataMapper.storeTradeAmount(month);

        //总开卡业绩
        StoreTradeAmount storeTradeAmount = new StoreTradeAmount();
        storeTradeAmount.setAmount(storeDataMapper.chargeAchieve(month));
        storeTradeAmount.setServeType("开卡");
        storeTradeAmountList.add(storeTradeAmount);

        //个人业绩
        List<StoreTechnicianAchieve> storeTechnicianAchieveList = storeDataMapper.storeTechnicianAchieve(month);
        storeData.setStoreTradeAmountList(storeTradeAmountList);
        storeData.setStoreTechnicianAchieveList(storeTechnicianAchieveList);

        //总业绩排行
        List<StoreTechnicianTotalAchieve> storeTechnicianTotalAchieveList = storeTechnicianAchieveList.stream().collect(Collectors.groupingBy(StoreTechnicianAchieve::getTechnicianId))
                .values().stream().map(list -> {
                    StoreTechnicianTotalAchieve storeTechnicianTotalAchieve = new StoreTechnicianTotalAchieve();
                    storeTechnicianTotalAchieve.setAchievement(list.stream().map(StoreTechnicianAchieve::getAchievement).reduce(BigDecimal.ZERO, BigDecimal::add));
                    storeTechnicianTotalAchieve.setTechnicianId(list.stream().map(StoreTechnicianAchieve::getTechnicianId).findAny().orElse(null));
                    storeTechnicianTotalAchieve.setTechnicianName(list.stream().map(StoreTechnicianAchieve::getTechnicianName).findAny().orElse(null));
                    storeTechnicianTotalAchieve.setTechnicianPicture(list.stream().findAny().get().getTechnicianPicture());

                    return storeTechnicianTotalAchieve;
                })
                .sorted(Comparator.comparing(StoreTechnicianTotalAchieve::getAchievement).reversed())
                .collect(Collectors.toList());

        storeData.setStoreTechnicianTotalAchieveList(storeTechnicianTotalAchieveList);

        //总订单业绩排行
        List<StoreTechnicianTotalAchieve> storeTechnicianTotalOrderAchieveList = storeTechnicianAchieveList.stream()
                .filter(storeTechnicianAchieve -> !storeTechnicianAchieve.getServeType().equals("开卡"))
                .collect(Collectors.groupingBy(StoreTechnicianAchieve::getTechnicianId))
                .values().stream().map(list -> {
                    StoreTechnicianTotalAchieve storeTechnicianTotalAchieve = new StoreTechnicianTotalAchieve();
                    storeTechnicianTotalAchieve.setAchievement(list.stream().map(StoreTechnicianAchieve::getAchievement).reduce(BigDecimal.ZERO, BigDecimal::add));
                    storeTechnicianTotalAchieve.setTechnicianId(list.stream().map(StoreTechnicianAchieve::getTechnicianId).findAny().orElse(null));
                    storeTechnicianTotalAchieve.setTechnicianName(list.stream().map(StoreTechnicianAchieve::getTechnicianName).findAny().orElse(null));
                    storeTechnicianTotalAchieve.setTechnicianPicture(list.stream().findAny().get().getTechnicianPicture());

                    return storeTechnicianTotalAchieve;
                })
                .sorted(Comparator.comparing(StoreTechnicianTotalAchieve::getAchievement).reversed())
                .collect(Collectors.toList());

        storeData.setStoreTechnicianTotalOrderAchieveList(storeTechnicianTotalOrderAchieveList);


        //总充值业绩排行
        List<StoreTechnicianTotalAchieve> storeTechnicianTotalChargeAchieveList = storeTechnicianAchieveList.stream()
                .filter(storeTechnicianAchieve -> storeTechnicianAchieve.getServeType().equals("开卡"))
                .collect(Collectors.groupingBy(StoreTechnicianAchieve::getTechnicianId))
                .values().stream().map(list -> {
                    StoreTechnicianTotalAchieve storeTechnicianTotalAchieve = new StoreTechnicianTotalAchieve();
                    storeTechnicianTotalAchieve.setAchievement(list.stream().map(StoreTechnicianAchieve::getAchievement).reduce(BigDecimal.ZERO, BigDecimal::add));
                    storeTechnicianTotalAchieve.setTechnicianId(list.stream().map(StoreTechnicianAchieve::getTechnicianId).findAny().orElse(null));
                    storeTechnicianTotalAchieve.setTechnicianName(list.stream().map(StoreTechnicianAchieve::getTechnicianName).findAny().orElse(null));
                    storeTechnicianTotalAchieve.setTechnicianPicture(list.stream().findAny().get().getTechnicianPicture());

                    return storeTechnicianTotalAchieve;
                })
                .sorted(Comparator.comparing(StoreTechnicianTotalAchieve::getAchievement).reversed())
                .collect(Collectors.toList());

        storeData.setStoreTechnicianTotalChargeAchieveList((storeTechnicianTotalChargeAchieveList));

        return JsonResult.success(storeData);
    }

    @ApiOperation("分页查询订单业绩")
    @GetMapping("/technician/career/queryPageAchieve")
    public JsonResult<Page<OrderServe>> queryPageAchieve(@RequestHeader String token,
                                                         @RequestParam String type,
                                                         @RequestParam Integer pageNum,
                                                         @RequestParam Integer pageSize,
                                                         @RequestParam(required = false) String month) {

        Page<OrderServe> page = new Page<>(pageNum, pageSize);
        Integer technicianId = SessionUtils.getTechnicianId();
        Page<OrderServe> result = orderServeMapper.queryPageAchieve(page, technicianId, type, month);
        return JsonResult.success(result);
    }

    @ApiOperation("分页查询订单业绩")
    @GetMapping("/technician/career/queryPageTechnicianAchieve")
    public JsonResult<Page<TechnicianAchieve>> queryPageTechnicianAchieve(@RequestHeader String token,
                                                                          @RequestParam String type,
                                                                          @RequestParam Integer pageNum,
                                                                          @RequestParam Integer pageSize,
                                                                          @RequestParam(required = false) String month,
                                                                          @RequestParam(required = false) Integer technicianId) {

        Page<OrderServe> page = new Page<>(pageNum, pageSize);

        if (type.equals("开卡")) {
            Page<TechnicianAchieve> result = orderServeMapper.queryPageChargeTechnicianAchieve(page, technicianId, month);
            return JsonResult.success(result);
        } else {
            Page<TechnicianAchieve> result = orderServeMapper.queryPageOrderTechnicianAchieve(page, technicianId, type, month);
            return JsonResult.success(result);
        }
    }

    @ApiOperation("分页查询开卡业绩")
    @GetMapping("/technician/career/queryPageChargeAchieve")
    public JsonResult<Page<CustomerBalanceRecord>> queryPageChargeAchieve(@RequestHeader String token,
                                                                          @RequestParam Integer pageNum,
                                                                          @RequestParam Integer pageSize,
                                                                          @RequestParam(required = false) String month) {

        Page<CustomerBalanceRecord> page = new Page<>(pageNum, pageSize);
        Integer technicianId = SessionUtils.getTechnicianId();
        Page<CustomerBalanceRecord> result = customerBalanceRecordMapper.queryPageAchieve(page, technicianId, month);
        return JsonResult.success(result);
    }

    @ApiOperation("查询美甲师评分")
    @GetMapping("/technician/career/queryPageTechScore")
    public JsonResult<List<TechnicianScore>> queryPageTechScore(@RequestHeader String token) {

        Integer technicianId = SessionUtils.getTechnicianId();

        List<TechnicianScore> list = technicianScoreMapper.selectList(
                new LambdaQueryWrapper<TechnicianScore>().eq(TechnicianScore::getTechnicianId, technicianId)
        );
        if (ListUtil.isNotEmpty(list)) {
            List<SkillSku> skillSkuList = skillSkuMapper.selectBatchIds(list.stream().map(TechnicianScore::getSkillSkuId).collect(Collectors.toList()));
            list.forEach(technicianScore -> technicianScore.setSkillSku(skillSkuList.stream().filter(skillSku -> skillSku.getId().equals(technicianScore.getSkillSkuId())).findAny().orElse(null)));
        }
        return JsonResult.success(list);
    }

}
