package com.gogirl.application.user.customer.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.store.store.StoreUser;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.infrastructure.common.exception.ErrorCode;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.mapper.store.store.UserManageMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerMapper;
import com.gogirl.shared.user.ConsumerCommand;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
@Slf4j
@Transactional
public class CustomerBalanceServiceImpl extends ServiceImpl<CustomerBalanceMapper, CustomerBalance> implements CustomerBalanceService {

    @Resource
    private CustomerBalanceMapper customerBalanceMapper;
    @Resource
    private CustomerBalanceRecordMapper balanceRecordDao;
    @Resource
    private CustomerBalanceRecordMapper customerBalanceRecordMapper;

    @Resource
    private UserManageMapper userManageMapper;
    @Resource
    private OrderManageService orderManageService;

    @Resource
    CustomerMapper customerMapper;

    @Override
    public CustomerBalance getCustomerBalance(int customerId) {
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>()
                .eq(CustomerBalance::getCustomerId, customerId));

        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(customerId);
            customerBalanceMapper.insert(customerBalance);
        }
        return customerBalance;
    }

    @Override
    public void consumer(ConsumerCommand cmd) {
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>()
                .eq(CustomerBalance::getCustomerId, cmd.getCustomerId()));

        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(cmd.getCustomerId());
            customerBalanceMapper.insert(customerBalance);
        }

        if (customerBalance.getBalance() < cmd.getAmount()) {
            throw new RRException(ErrorCode.CS_2002);
        }

        //余额支付
        CustomerBalanceRecord record = CustomerBalanceRecord
                .builder()
                .bestowAmount(0)
                .oldBalance(customerBalance.getBalance())
                .currentBalance(customerBalance.getBalance() - cmd.getAmount())
                .orderAmount(cmd.getAmount())
                .customerId(cmd.getCustomerId())
                .departmentId(cmd.getDepartmentId())
                .orderId(String.valueOf(cmd.getOrderId()))
                .discount(1.00)
                .orderState(cmd.getOrderStatus())
                //来源 会员卡收款
                .source(2)
                //类型 预约订单扣款
                .type(-1)
                .time(new Date())
                .build();

        customerBalanceRecordMapper.insert(record);
        customerBalance.setBalance(customerBalance.getBalance() - cmd.getAmount());
        customerBalance.setTotalExpenditure(customerBalance.getTotalExpenditure() + cmd.getAmount());
        customerBalanceMapper.updateById(customerBalance);
    }


    @Override
    public IPage<CustomerBalanceRecord> getBalanceRecordCard(Integer customerId, int pageNum, int pageSize) {
        CustomerBalanceRecord param = new CustomerBalanceRecord();
        param.setCustomerId(customerId);

        IPage<CustomerBalanceRecord> page = new Page<>(pageNum, pageSize);
        //1、查询消费记录
        page = balanceRecordDao.getBalanceRecordCard(page, customerId);

        //2、如果是预约订单消费，关联查询预约订单
        page.getRecords().stream()
                .filter(balanceRecord -> balanceRecord.getType().equals(-1))
                .forEach(balanceRecord -> {
                    //消费记录关联的订单
                    OrderManage orderManage = orderManageService.queryOrder(Integer.valueOf(balanceRecord.getOrderId()));
                    balanceRecord.setOrderManage(orderManage);
                    //deprecated
                    if (balanceRecord.getOrderAmount() != null && balanceRecord.getOrderAmount() != 0 && balanceRecord.getOrderManage() != null) {
                        balanceRecord.getOrderManage().setTotalPrice(new BigDecimal(balanceRecord.getOrderAmount()).divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP));
                    }
                });

        //设置推荐人名字
        List<StoreUser> allStoreUser = userManageMapper.selectList(new QueryWrapper<>());
        Map<String, String> idMapName = new HashMap<>();

        for (StoreUser storeUser : allStoreUser) {
            idMapName.put(String.valueOf(storeUser.getId()), storeUser.getName());
        }
        //
        for (CustomerBalanceRecord customerBalanceRecord : page.getRecords()) {
            String refereeId = customerBalanceRecord.getRefereeId();
            if (refereeId != null && !refereeId.isEmpty()) {
                String[] refereeIdArr = refereeId.split(",");
                StringBuilder refereeName = new StringBuilder();
                for (String s : refereeIdArr) {
                    if (s != null && !s.isEmpty()) {
                        refereeName.append(",").append(idMapName.getOrDefault(s, s));
                    }
                }
                customerBalanceRecord.setRefereeId(refereeName.substring(1));
            }
        }
        return page;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void transferBalance(String sourcePhone, String targetPhone) {

        Customer sourceCustomer = customerMapper.selectByPhone(sourcePhone);
        Customer targetCustomer = customerMapper.selectByPhone(targetPhone);

        if (sourceCustomer == null || targetCustomer == null) {
            throw new RRException("用户不存在");
        }

        CustomerBalance sourceCustomerBalance = customerBalanceMapper.getByCustomerId(sourceCustomer.getId());
        if (sourceCustomerBalance == null) {
            throw new RRException("源用户异常");
        }

        Integer transferAmount = sourceCustomerBalance.getBalance();

        CustomerBalanceRecord sourceCustomerBalanceRecord = new CustomerBalanceRecord();
        sourceCustomerBalanceRecord.setCustomerId(sourceCustomer.getId());
        sourceCustomerBalanceRecord.setOrderAmount(transferAmount);
        sourceCustomerBalanceRecord.setSource(CustomerBalanceRecord.SOURCE_TRANSFER);
        sourceCustomerBalanceRecord.setType(CustomerBalanceRecord.TYPE_TRANSFER_OUT);
        customerBalanceRecordMapper.insert(sourceCustomerBalanceRecord);

        sourceCustomerBalance.setBalance(0);
        customerBalanceMapper.updateById(sourceCustomerBalance);


        CustomerBalance targetCustomerBalance = customerBalanceMapper.getByCustomerId(targetCustomer.getId());
        if (targetCustomerBalance == null) {
            targetCustomerBalance = new CustomerBalance();
            targetCustomerBalance.setCustomerId(targetCustomer.getId());
            customerBalanceMapper.insert(targetCustomerBalance);
        }


        CustomerBalanceRecord targetCustomerBalanceRecord = new CustomerBalanceRecord();
        targetCustomerBalanceRecord.setCustomerId(targetCustomer.getId());
        targetCustomerBalanceRecord.setOrderAmount(transferAmount);
        targetCustomerBalanceRecord.setSource(CustomerBalanceRecord.SOURCE_TRANSFER);
        targetCustomerBalanceRecord.setType(CustomerBalanceRecord.TYPE_TRANSFER_IN);
        customerBalanceRecordMapper.insert(targetCustomerBalanceRecord);

        targetCustomerBalance.setBalance(targetCustomerBalance.getBalance() + transferAmount);
        customerBalanceMapper.updateById(targetCustomerBalance);


    }
}
