package com.gogirl.application.order.serve.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.github.binarywang.wxpay.bean.order.WxPayMpOrderResult;
import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.gogirl.application.market.coupon.MarketService;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.order.serve.PayService;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.domain.order.serve.MultiPaymentType;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.infrastructure.common.config.property.GogirlProperties;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.JsonUtilByFsJson;
import com.gogirl.infrastructure.common.util.SessionUtils;
import com.gogirl.infrastructure.mapper.market.discount.DiscountConfigMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderManageMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerMapper;
import com.gogirl.infrastructure.service.wx.WxPayService;
import com.gogirl.shared.member.BalanceWxPayQuery;
import com.gogirl.shared.member.UnifiedOrderQuery;
import com.gogirl.shared.order.WxPayOrderQuery;
import com.gogirl.shared.user.ConsumerCommand;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;
import java.util.List;

@Service
@AllArgsConstructor
@Slf4j
@Transactional
public class PayServiceImpl implements PayService {


    private final CustomerBalanceMapper customerBalanceMapper;
    private final CustomerBalanceRecordMapper customerBalanceRecordMapper;
    private final CustomerMapper customerMapper;
    private final DiscountConfigMapper discountConfigMapper;

    private final OrderManageService orderManageService;
    private final MarketService marketService;
    private final WxPayService wxPayService;


    private final GogirlProperties gogirlProperties;
    private final CustomerBalanceService customerBalanceService;
    private final OrderManageMapper orderManageMapper;

    @Override
    public void consumeBalance(Integer currentCustomerId, Integer orderId) {

        //pre、订单状态检测
        OrderManage orderManage = orderManageService.queryOrder(orderId);
        if (!orderManage.getStatus().equals(OrderManage.STATUS_UN_PAY)) {
            throw new RRException("订单状态异常");
        }

//        this.checkCustomerBalance(currentCustomerId, orderId);


        /*1、扣除余额*/
        ConsumerCommand consumerCmd = ConsumerCommand.builder()
                .amount(orderManage.getTotalPaymentAmount().multiply(new BigDecimal(100)).intValue())
                .customerId(currentCustomerId)
                .departmentId(orderManage.getDepartmentId())
                .orderId(orderId)
                .orderStatus(orderManage.getStatus())
                .brandId(orderManage.getBrandId())
                .build();
        customerBalanceService.consumer(consumerCmd);

        /*2、更新订单状态*/
        orderManage.setRemark("余额支付");
        orderManageService.updateById(orderManage);

        //支付时间
        orderManage.setPayTime(new Date());
        //支付方式
        orderManage.setPaymentType(OrderManage.PAYMENT_BALANCE);
        //状态待评价
        orderManage.setStatus(OrderManage.STATUS_UN_COMMENT);
        //支付用户
        orderManage.setPayUser(currentCustomerId);
        //是否代付
        orderManage.setIsFriendPayByPayUserAndOrderUser();

        orderManageService.updateById(orderManage);


        /*step3.更新卡券状态*/
        log.debug("更新卡券状态");
        marketService.finishPayOrder(orderManage.getId(), orderManage.getBrandId());
    }


    @Override
    public WxPayMpOrderResult unifiedOrder(UnifiedOrderQuery qry) throws UnknownHostException, WxPayException {

        OrderManage orderManage = orderManageService.queryOrder(qry.getOrderId());

        if (!orderManage.getStatus().equals(OrderManage.STATUS_UN_PAY)) {
            throw new RRException("订单状态异常");
        }
        WxPayOrderQuery qry2 = WxPayOrderQuery
                .builder()
                .currentCustomerId(qry.getCustomerId())
                .openid(qry.getOpenid())
                //避免同单不同金额的情况
                .orderNo(String.format("%06d", orderManage.getTotalPaymentAmount().multiply(new BigDecimal(100)).intValue()).substring(0, 6) + orderManage.getOrderNo())
                .totalPaymentAmount(orderManage.getTotalPaymentAmount())
                .build();

        WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest =
                WxPayUnifiedOrderRequest.newBuilder()
                        .openid(qry.getOpenid())
                        .tradeType("JSAPI")
                        //todo
                        .notifyUrl(gogirlProperties.getNotifyOrder())
                        .spbillCreateIp(InetAddress.getLocalHost().getHostAddress())
                        .totalFee(qry2.getTotalPaymentAmount().multiply(new BigDecimal(100)).intValue())
                        .outTradeNo(qry2.getOrderNo())
                        .body("gogirl美甲美睫")
                        .build();
        log.info("**微信下单参数构造**,{}", wxPayUnifiedOrderRequest.toString());

        wxPayUnifiedOrderRequest.setNotifyUrl(gogirlProperties.getNotifyOrder());
        //微信统一下单
        return wxPayService.createJsapiOrder(wxPayUnifiedOrderRequest, orderManage.getBrandId());

    }

    private void checkCustomerBalance(Integer customerId, Integer orderId) throws RRException {
        OrderManage orderManage = orderManageService.getById(orderId);
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>().eq(CustomerBalance::getCustomerId, customerId));

        if (customerBalance == null) {
            throw new RRException("您不是会员，请直接使用pos支付");
        }
        if (orderManage.getTotalPaymentAmount().multiply(new BigDecimal(100)).intValue() < customerBalance.getBalance()) {
            throw new RRException("请使用余额支付");
        }
        if (customerBalance.getBalance() <= 0) {
            throw new RRException("余额为0，请直接使用pos支付");
        }
    }


    @Override
    public WxPayMpOrderResult balanceWxPay(BalanceWxPayQuery qry) throws UnknownHostException, WxPayException {

        //1、订单状态检查
        OrderManage orderManage = orderManageService.queryOrder(qry.getOrderId());
        if (!orderManage.getStatus().equals(OrderManage.STATUS_UN_PAY)) {
            throw new RRException("订单状态异常");
        }

        //2、余额检查
        this.checkCustomerBalance(qry.getCustomerId(), qry.getOrderId());
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>().eq(CustomerBalance::getCustomerId, qry.getCustomerId()));


        //3、微信统一下单
        BigDecimal leftTotalPaymentAmount = orderManage.getTotalPaymentAmount().subtract(new BigDecimal(customerBalance.getBalance()).divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP));
        WxPayOrderQuery wxPayOrderQuery = WxPayOrderQuery.builder()
                //避免同单不同金额的情况
                .orderNo(String.format("%06d", leftTotalPaymentAmount.multiply(new BigDecimal(100)).intValue()).substring(0, 6) + orderManage.getOrderNo())
                .currentCustomerId(qry.getCustomerId())
                .openid(qry.getOpenid())
                .totalPaymentAmount(leftTotalPaymentAmount)
                .build();

        WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest =
                WxPayUnifiedOrderRequest.newBuilder()
                        .openid(qry.getOpenid())
                        .tradeType("JSAPI")
                        //todo
                        .notifyUrl(gogirlProperties.getNotifyBalanceWxPay())
                        .spbillCreateIp(InetAddress.getLocalHost().getHostAddress())
                        .totalFee(wxPayOrderQuery.getTotalPaymentAmount().multiply(new BigDecimal(100)).intValue())
                        .outTradeNo(wxPayOrderQuery.getOrderNo())
                        .body("gogirl美甲美睫")
                        .build();

        log.info("**微信下单参数构造**,{}", wxPayUnifiedOrderRequest.toString());

        orderManage.setPayUser(SessionUtils.getCustomerId());
        orderManageMapper.updateById(orderManage);
        //微信统一下单
        return wxPayService.createJsapiOrder(wxPayUnifiedOrderRequest, orderManage.getBrandId());
    }


    @Override
    public void balancePosPay(Integer currentCustomerId, Integer orderId) {

        //1、订单状态检查
        OrderManage orderManage = orderManageService.getById(orderId);
        if (!orderManage.getStatus().equals(OrderManage.STATUS_UN_PAY)) {
            throw new RRException("订单状态异常");
        }

        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>().eq(CustomerBalance::getCustomerId, currentCustomerId));
        this.checkCustomerBalance(currentCustomerId, orderId);

        //3、更新订单
        //支付类型：请余额pos支付
        orderManage.setPaymentType(13);

        //多种方式支付字段构建
        BigDecimal balanceAmount = new BigDecimal(customerBalance.getBalance()).divide(new BigDecimal(100), 100, BigDecimal.ROUND_HALF_UP);

        MultiPaymentType balancePay = new MultiPaymentType(2, balanceAmount);
        MultiPaymentType posPay = new MultiPaymentType(5, orderManage.getTotalPaymentAmount().subtract(balanceAmount));

        List<MultiPaymentType> multiPaymentTypeList = Lists.newArrayList(balancePay, posPay);

        orderManage.setMultiplePaymentType(JsonUtilByFsJson.beanToJson(multiPaymentTypeList));

        //待确认支付
        orderManage.setStatus(8);
        orderManage.setUpdateTime(new Date());
        orderManageService.updateById(orderManage);
    }

    @Override
    public void chosePayType(Integer orderId, Integer payType) {


        //1、订单状态检查
        OrderManage orderManage = orderManageMapper.selectById(orderId);
        if (!orderManage.getStatus().equals(OrderManage.STATUS_UN_PAY)) {
            throw new RRException("订单状态异常");
        }

        //2、更新订单状态
        //客户申请pos机支付
        if (payType.equals(5)) {
            //状态待确认
            orderManage.setStatus(OrderManage.STATUS_UN_CONFIRM);
            //支付类型状态pos机
            orderManage.setPaymentType(OrderManage.PAYMENT_POS);
            orderManage.setUpdateTime(new Date());
            orderManageMapper.updateById(orderManage);
        }
        //客户申请团购支付
        else if (payType.equals(7)) {
            //状态待确认
            orderManage.setStatus(OrderManage.STATUS_UN_CONFIRM);
            //支付类型状态pos机
            orderManage.setPaymentType(OrderManage.PAYMENT_DZ);
            orderManage.setUpdateTime(new Date());
            orderManageMapper.updateById(orderManage);
        }
        //非法类型
        else {
            throw new RRException(500, "invalid payType");
        }
    }
}
