package com.gogirl.interfaces.user;

import com.github.pagehelper.PageInfo;
import com.gogirl.application.market.CouponService;
import com.gogirl.application.market.MarketService;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.market.serve.Coupon;
import com.gogirl.domain.market.serve.CouponCustomerRelevance;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.StringUtils;
import com.gogirl.infrastructure.mapper.market.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.CouponMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.xcx.GogirlConfigMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@RestController
@Api(tags = {"version2.0: 优惠券"}, value = "version2.0: 优惠券")
@AllArgsConstructor
@Slf4j
public class CouponController {

    private final CouponService couponService;
    private final GogirlTokenService gogirlTokenService;
    private final OrderManageService orderManageService;
    private final GogirlConfigMapper gogirlConfigMapper;
    private final CouponCustomerRelevanceMapper couponCustomerRelevanceMapper;
    private final OrderServeMapper orderServeMapper;
    private final MarketService marketService;
    private final CouponMapper couponMapper;

    @ApiOperation(value = "查询优惠券配置详情")
    @GetMapping("/customer/selectCouponByPrimaryKey")
    public JsonResult<Coupon> selectCouponByPrimaryKey(Integer id) {
        Coupon coupon = couponService.getById(id);
        return JsonResult.success(coupon);
    }

    @ApiOperation(value = "用户根据订单号查询可用优惠券")
    @GetMapping("/customer/getOrderCoupon")
    public JsonResult<List<CouponCustomerRelevance>> getOrderCoupon(@RequestHeader String token,
                                                                    @RequestParam Integer orderId) {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));
        List<CouponCustomerRelevance> list = couponService.getOrderCoupon(currentCustomerId, orderId);
        return JsonResult.success(list);
    }

    @ApiOperation(value = "我的优惠券列表")
    @GetMapping("/customer/getMyCoupon")
    public JsonResult<PageInfo<CouponCustomerRelevance>> getMyCoupon(String token,
                                                                     Integer pageNum,
                                                                     Integer pageSize,
                                                                     Integer state,
                                                                     Boolean containRemoveCoupon) {
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);
        Integer customerId = gogirlToken.getCustomerId();
        log.info("查询领取优惠券的用户列表CouponCustomerRelevance:" + customerId.toString());
        PageInfo<CouponCustomerRelevance> pageInfo = couponService.selectMyCoupon(customerId, pageNum, pageSize, state, containRemoveCoupon);
        return JsonResult.success(pageInfo);
    }

    @ApiOperation(value = "查询是否可领新人优惠券,是新人返回券信息,不是新人返回null;")
    @GetMapping("/customer/checkNewCustomer")
    public JsonResult<Coupon> checkNewCustomer(String token) {
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);
        //顾客id
        Integer customerId = SessionUtils.getCustomer().getId();
        Coupon coupon = couponService.checkNewCustomer(customerId);
        return JsonResult.success(coupon);
    }

    @ApiOperation(value = "发放新人优惠券")
    @PostMapping("/customer/sendNewCustomerCouponXcx")
    public JsonResult<Coupon> sendNewCustomerCouponXcx(String token) {

        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);

        Integer customerId = SessionUtils.getCustomer().getId();
        Coupon coupon = couponService.sendNewCustomerCouponXcx(customerId);
        return JsonResult.success(coupon);
    }

    @ApiOperation(value = "小程序发券")
    @PostMapping("/customer/sendCouponXcx")
    public JsonResult<Coupon> sendCouponXcx(String token,
                                            Integer couponId) {
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);

        Integer customerId = SessionUtils.getCustomer().getId();

        log.info("发放优惠券customerId:" + customerId.toString());
        log.info("发放优惠券couponId:" + couponId.toString());

        //返回消息
        Coupon coupon = couponService.sendCoupon(couponId, customerId);
        return JsonResult.success(coupon);
    }


    @ApiOperation(value = "美甲师提交外部券")
    @PostMapping("/technician/techSubmitCoupons")
    public JsonResult<Void> techSubmitCoupons(@RequestHeader String token,
                                              @RequestParam Integer orderId,
                                              @RequestParam String couponIds) {
        StoreTechnician storeTechnician = SessionUtils.getStoreTechnician();
        log.info("美甲师 {} 提交外部券,couponId", storeTechnician.getName());

        List<Integer> idList = StringUtils.isNotEmpty(couponIds)
                ? Lists.newArrayList(couponIds.split(",")).stream().map(Integer::valueOf).collect(Collectors.toList())
                : Lists.newArrayList();

        couponService.techSubmitCoupons(orderId, idList);
        return JsonResult.success();
    }

    @ApiOperation(value = "美甲师根据订单号查询可用外部券")
    @GetMapping("/technician/getOrderExternalCoupon")
    public JsonResult<List<Coupon>> getOrderExternalCoupon(@RequestHeader String token,
                                                           @RequestParam Integer orderId) {

        log.info("用户根据订单号查询可用优惠券");
        List<Coupon> couponList = couponService.getOrderExternalCoupon(orderId);
        return JsonResult.success(couponList);
    }
}
