package com.gogirl.application.store.store.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.TakeLeaveEventCmdService;
import com.gogirl.assembler.ApplyTakeLeaveCommandAssembler;
import com.gogirl.assembler.OverTimeLogCommandAssembler;
import com.gogirl.domain.store.oa.OverTimeRecordLog;
import com.gogirl.domain.store.oa.OvertimeRecord;
import com.gogirl.domain.store.oa.TakeLeaveEvent;
import com.gogirl.domain.store.store.Message;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.JsonUtilByFsJson;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.store.oa.OverTimeRecordLogMapper;
import com.gogirl.infrastructure.mapper.store.oa.OvertimeRecordMapper;
import com.gogirl.infrastructure.mapper.store.oa.TakeLeaveEventMapper;
import com.gogirl.infrastructure.mapper.store.store.MessageMapper;
import com.gogirl.infrastructure.mapper.store.store.StoreManageMapper;
import com.gogirl.infrastructure.mapper.store.store.StoreTechnicianMapper;
import com.gogirl.shared.store.ApplyTakeLeaveCommand;
import com.gogirl.shared.store.ApprovalTakeLeaveCommand;
import com.gogirl.shared.store.PageApplyQuery;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
@Transactional
@Slf4j
public class TakeLeaveEventCmdServiceImpl implements TakeLeaveEventCmdService {

    /**
     * assembler
     */
    private final ApplyTakeLeaveCommandAssembler applyTakeLeaveCommandAssembler;
    private final OverTimeLogCommandAssembler overTimeLogCommandAssembler;

    /**
     * repository
     */
    private final StoreTechnicianMapper storeTechnicianRepository;
    private final TakeLeaveEventMapper takeLeaveEventRepository;
    private final OvertimeRecordMapper overtimeRecordRepository;
    private final OverTimeRecordLogMapper overTimeRecordLogRepository;
    private final MessageMapper messageMapper;
    private final StoreManageMapper storeManageMapper;

    @Override
    public void applyTakeLeave(ApplyTakeLeaveCommand cmd) {

        List<TakeLeaveEvent> appliedList = takeLeaveEventRepository.appliedList(cmd.getApplyTechnicianId(), cmd.getStartTime().getTime(), cmd.getEndTime().getTime());
        if (ListUtil.isNotEmpty(appliedList)) {
            throw new RRException("重复时间提交");
        }

        TakeLeaveEvent takeLeaveEvent = applyTakeLeaveCommandAssembler.apply(cmd);
        takeLeaveEvent.setStatus(TakeLeaveEvent.STATUS_UN_APPROVAL);

        takeLeaveEvent.setTimeLength(cmd.getTimeLength());

        //处理人
        StoreTechnician storeTechnician = storeTechnicianRepository.selectById(cmd.getDealingTechnicianId());
        takeLeaveEvent.setDealingTechnicianName(storeTechnician.getName());
        takeLeaveEvent.setCreateTime(System.currentTimeMillis());

        //
        takeLeaveEventRepository.insert(takeLeaveEvent);

        //调休
        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_ADJUST_REST)) {

            List<OverTimeRecordLog> overTimeRecordLogList = cmd.getOverTimeLogCommandList()
                    .stream().map(overTimeLogCommandAssembler)
                    .peek(overTimeRecordLog -> overTimeRecordLog.setTakeLeaveEventId(takeLeaveEvent.getId()))
                    .collect(Collectors.toList());

            //检查总调休时间
            if (takeLeaveEvent.getTimeLength()
                    .compareTo(overTimeRecordLogList.stream().map(OverTimeRecordLog::getLengthTime).reduce(BigDecimal.ZERO, BigDecimal::add)) != 0) {

                log.info("申请时间:{}", takeLeaveEvent.getTimeLength());
                log.info("选择的时间{},", overTimeRecordLogList.stream().map(OverTimeRecordLog::getLengthTime).reduce(BigDecimal.ZERO, BigDecimal::add));
                throw new RRException(500, "调休时间有误");
            }

            overTimeRecordLogList.forEach(overTimeRecordLog -> {
                //加班记录
                OvertimeRecord overtimeRecord = overtimeRecordRepository.selectById(overTimeRecordLog.getOverTimeRecordId());
                //扣除这一次申请的时间
                overtimeRecord.setLeftTimeLength(overtimeRecord.getLeftTimeLength().subtract(overTimeRecordLog.getLengthTime()));
                //检查调休时间和加班时间匹配
                if (overtimeRecord.getLeftTimeLength().compareTo(BigDecimal.ZERO) < 0) {
                    throw new RRException(500, "超出加班时间");
                }
                if (overtimeRecord.getLeftTimeLength().compareTo(BigDecimal.ZERO) == 0) {
                    //不能调休
                    overtimeRecord.setStatus(2);
                }
                overtimeRecord.setLastUpdateTime(System.currentTimeMillis());
                //更新加班使用情况
                overtimeRecordRepository.updateById(overtimeRecord);
            });

            //写入加班使用情况日志
            overTimeRecordLogList.forEach(overTimeRecordLogRepository::insert);
        }

        takeLeaveEvent.setApplyStoreTechnician(storeTechnicianRepository.selectById(cmd.getApplyTechnicianId()));
        Message message = Message.builder()
                .content("请假审核通知")
                .isRead(0)
                .paramJson(JsonUtilByFsJson.beanToJson(takeLeaveEvent))
                .technicianId(takeLeaveEvent.getDealingTechnicianId())
                .time(new Date())
                .title("请假审核")
                .type(1)
                .tab(2)
                .build();
        messageMapper.insert(message);
    }

    @Transactional
    @Override
    public void approvalTakeLeave(ApprovalTakeLeaveCommand cmd) {

        TakeLeaveEvent takeLeaveEvent = takeLeaveEventRepository.selectById(cmd.getTakeLeaveEventId());

        if (!takeLeaveEvent.getStatus().equals(TakeLeaveEvent.STATUS_UN_APPROVAL)) {
            throw new RRException(500, "审批状态异常");
        }

//        //加班申请通过 添加加班记录
//        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_OVER_TIME)
//                && cmd.getApproval().equals(TakeLeaveEvent.STATUS_APPROVAL_PASS)) {
//
//            OvertimeRecord overtimeRecord = OvertimeRecord.builder()
//                    .createTime(System.currentTimeMillis())
//                    .startTime(takeLeaveEvent.getStartTime())
//                    .endTime(takeLeaveEvent.getEndTime())
//                    .lastUpdateTime(System.currentTimeMillis())
//                    .leftTimeLength(takeLeaveEvent.getTimeLength())
//                    .status(1)
//                    .totalTimeLength(takeLeaveEvent.getTimeLength())
//                    .technicianId(takeLeaveEvent.getApplyTechnicianId())
//                    .takeLeaveEventId(cmd.getTakeLeaveEventId())
//                    .build();
//            overtimeRecordRepository.insert(overtimeRecord);
//        }


        //调休申请拒绝
        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_ADJUST_REST)
                && cmd.getApproval().equals(TakeLeaveEvent.STATUS_APPROVAL_REFUSE)) {
            Long takeLeaveEventId = cmd.getTakeLeaveEventId();
            List<OverTimeRecordLog> overTimeRecordLogList = overTimeRecordLogRepository
                    .selectList(new LambdaQueryWrapper<OverTimeRecordLog>().eq(OverTimeRecordLog::getTakeLeaveEventId, takeLeaveEventId));

            overTimeRecordLogList.stream().collect(Collectors.groupingBy(OverTimeRecordLog::getOverTimeRecordId))
                    .forEach((overTimeRecordId, list) -> {
                        OvertimeRecord overtimeRecord = overtimeRecordRepository.selectById(overTimeRecordId);
                        BigDecimal timeLength = list.stream().map(OverTimeRecordLog::getLengthTime).reduce(BigDecimal.ZERO, BigDecimal::add);
                        overtimeRecord.setLeftTimeLength(overtimeRecord.getLeftTimeLength().add(timeLength));
                        overtimeRecordRepository.insert(overtimeRecord);
                    });

            overTimeRecordLogRepository.deleteBatchIds(overTimeRecordLogList.stream().map(OverTimeRecordLog::getId).collect(Collectors.toList()));
        }

        takeLeaveEvent.setStatus(cmd.getApproval());
        takeLeaveEvent.setRejectReason(cmd.getRejectReason());
        takeLeaveEventRepository.updateById(takeLeaveEvent);

        StoreTechnician storeTechnician = storeTechnicianRepository.selectById(takeLeaveEvent.getApplyTechnicianId());
        takeLeaveEvent.setApplyStoreTechnician(storeTechnician);

        Message message = Message.builder()
                .content("请假申请通过了")
                .isRead(0)
                .paramJson(JsonUtilByFsJson.beanToJson(takeLeaveEvent))
                .technicianId(takeLeaveEvent.getApplyTechnicianId())
                .time(new Date())
                .title("请假申请")
                .type(3)
                .tab(2)
                .build();
        if (cmd.getApproval().equals(TakeLeaveEvent.STATUS_APPROVAL_REFUSE)) {
            message.setContent("请假申请被拒绝了");
        }
        messageMapper.insert(message);
    }

    @Override
    public TakeLeaveEvent queryTakeLeave(Long id) {
        TakeLeaveEvent takeLeaveEvent = takeLeaveEventRepository.selectById(id);
        StoreTechnician storeTechnician = storeTechnicianRepository.selectById(takeLeaveEvent.getApplyTechnicianId());
        takeLeaveEvent.setApplyStoreTechnicianDTO(storeTechnician);
        return takeLeaveEvent;
    }

    @Override
    public IPage<TakeLeaveEvent> queryPageTakeLeave(PageApplyQuery qry) {

        LambdaQueryWrapper<TakeLeaveEvent> wrapper = new LambdaQueryWrapper<>();
        if (qry.getApplyTechnicianId() != null) {
            wrapper.eq(TakeLeaveEvent::getApplyTechnicianId, qry.getApplyTechnicianId());
        }
        if (qry.getDealingTechnicianId() != null) {
            wrapper.eq(TakeLeaveEvent::getDealingTechnicianId, qry.getDealingTechnicianId());
        }
        if (qry.getStatus() != null) {
            wrapper.eq(TakeLeaveEvent::getStatus, qry.getStatus());
        }
        wrapper.orderByDesc(TakeLeaveEvent::getCreateTime);
        IPage<TakeLeaveEvent> page = new Page<>(qry.getPageNum(), qry.getPageSize());

        page = takeLeaveEventRepository.selectPage(page, wrapper);

        if (ListUtil.isEmpty(page.getRecords())) {
            return page;
        }
        List<Integer> ids = page.getRecords().stream().map(TakeLeaveEvent::getApplyTechnicianId).collect(Collectors.toList());
        List<StoreTechnician> storeTechnicianList = storeTechnicianRepository.selectBatchIds(ids);
        Map<Integer, List<StoreTechnician>> map = storeTechnicianList.stream().collect(Collectors.groupingBy(StoreTechnician::getId));

        page.getRecords().forEach(takeLeaveEvent -> {
            if (ListUtil.isNotEmpty(map.get(takeLeaveEvent.getApplyTechnicianId()))) {
                StoreTechnician storeTechnician = map.get(takeLeaveEvent.getApplyTechnicianId()).stream().findAny().orElse(null);
                takeLeaveEvent.setApplyStoreTechnicianDTO(storeTechnician);
            }
        });
        return page;
    }


    @Override
    public List<StoreTechnician> queryCheckTechnicianList(Integer technicianId, Integer departmentId) {
        StoreManage storeManage = storeManageMapper.selectById(departmentId);
        StoreTechnician storeTechnician;
        if (storeManage.getMasterUserId().equals(technicianId))
            storeTechnician = storeTechnicianRepository.selectById(storeManage.getOperateUserId());
        else
            storeTechnician = storeTechnicianRepository.selectById((storeManage.getMasterUserId()));
        return Lists.newArrayList(storeTechnician);
    }

    @Override
    public void adminRefuseTakeLeaveEvent(Integer takeLeaveEventId) {
        TakeLeaveEvent takeLeaveEvent = takeLeaveEventRepository.selectById(takeLeaveEventId);

        //如果是通过的加班申请被驳回
        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_OVER_TIME) &&
                takeLeaveEvent.getStatus().equals(TakeLeaveEvent.STATUS_APPROVAL_PASS)) {

            OvertimeRecord overtimeRecord = overtimeRecordRepository.selectOne(new LambdaQueryWrapper<OvertimeRecord>()
                    .eq(OvertimeRecord::getTakeLeaveEventId, takeLeaveEvent.getId()));

            List<OverTimeRecordLog> overTimeRecordLogList = overTimeRecordLogRepository.selectList(new LambdaQueryWrapper<OverTimeRecordLog>()
                    .eq(OverTimeRecordLog::getOverTimeRecordId, overtimeRecord.getId()));

            overTimeRecordLogList.forEach(overTimeRecordLog -> {

            });
        }
    }
}
