package com.gogirl.application.market.coupon.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gogirl.application.market.coupon.CouponService;
import com.gogirl.domain.common.xcx.GogirlConfig;
import com.gogirl.domain.market.coupon.Coupon;
import com.gogirl.domain.market.coupon.CouponCustomerRelevance;
import com.gogirl.domain.market.coupon.CouponExcludeDetailed;
import com.gogirl.domain.market.discount.DiscountConfig;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.common.util.StringUtils;
import com.gogirl.infrastructure.mapper.common.xcx.GogirlConfigMapper;
import com.gogirl.infrastructure.mapper.market.coupon.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.coupon.CouponExcludeDetailedMapper;
import com.gogirl.infrastructure.mapper.market.coupon.CouponMapper;
import com.gogirl.infrastructure.mapper.market.discount.DiscountConfigMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderManageMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 卡券配置服务
 */
@Service
@Slf4j
public class CouponServiceImpl extends ServiceImpl<CouponMapper, Coupon> implements CouponService {

    @Resource
    private CouponMapper couponMapper;
    @Resource
    private CouponCustomerRelevanceMapper couponCustomerRelevanceMapper;
    @Resource
    private OrderServeMapper orderServeMapper;
    @Resource
    private OrderManageMapper orderManageMapper;
    @Resource
    private GogirlConfigMapper gogirlConfigMapper;
    @Resource
    CouponExcludeDetailedMapper couponExcludeDetailedMapper;
    @Resource
    private DiscountConfigMapper discountConfigMapper;

    @Override
    public Coupon checkNewCustomer(Integer customerId) {
        //新用户配置
        GogirlConfig gogirlConfig = gogirlConfigMapper.selectByPrimaryKey(2);

        if (gogirlConfig == null || StringUtils.isEmpty(gogirlConfig.getValue())) {
            throw new RRException("请联系管理员配置发放哪张新人优惠券");
        }
        //新人券id
        Integer couponId = Integer.valueOf(gogirlConfig.getValue());

        //用户的新人券
        List<CouponCustomerRelevance> list = couponCustomerRelevanceMapper.selectList(new LambdaQueryWrapper<CouponCustomerRelevance>()
                .eq(CouponCustomerRelevance::getCustomerId, customerId)
                .eq(CouponCustomerRelevance::getCouponId, couponId));
        //用户的订单
        Integer orderTimes = orderManageMapper.countOrderTimes(customerId);

        //如果用户已经有订单或者已经领取过新人券，就不是新人，返回null
        if (list.size() > 0 || orderTimes > 0) {
            return null;
        }
        //新人返回券新人券配置
        return this.getById(couponId);
    }

    @Override
    public Coupon sendNewCustomerCouponXcx(Integer customerId) {
        GogirlConfig gogirlConfig = gogirlConfigMapper.selectByPrimaryKey(2);

        if (gogirlConfig == null || StringUtils.isEmpty(gogirlConfig.getValue())) {
            throw new RRException("请联系管理员配置发放哪张新人优惠券");
        }
        Integer couponId = Integer.valueOf(gogirlConfig.getValue());
        return sendCoupon(couponId, customerId);
    }

    @Override
    public Coupon sendCoupon(Integer couponId, Integer customerId) {
        log.info("发放优惠券customerId:{},发放优惠券couponId:{}", customerId, couponId);

        /*1、发券校验*/
        Coupon coupon = this.getById(couponId);
        if (coupon == null) {
            throw new RRException("找不到该优惠券");
        }
        //判断优惠券是否还有
        if (coupon.getAllQuantity() <= coupon.getReceiveQuantity()) {
            throw new RRException("优惠券已经领完");
        }

        //判断该用户是否到达领取限制
        int receiveCouponNam = couponCustomerRelevanceMapper.countRelevanceNum(couponId, customerId);
        if (coupon.getLimitQuantity() != null && coupon.getLimitQuantity() != 0 && coupon.getLimitQuantity() <= receiveCouponNam) {
            throw new RRException("抱歉,该优惠券最多可领取" + coupon.getLimitQuantity() + "张,你已有" + receiveCouponNam + "张该优惠券.");
        }

        /*2、发放卡券*/
        CouponCustomerRelevance couponCustomerRelevance = CouponCustomerRelevance.getInstance(coupon);

        //是否免单券
        DiscountConfig discountConfig = discountConfigMapper.selectOne(new LambdaQueryWrapper<DiscountConfig>()
                .eq(DiscountConfig::getCouponId, couponId));
        if (discountConfig != null) {
            couponCustomerRelevance.setIsFreeVoucher(1);
        }
        couponCustomerRelevance.setCustomerId(customerId);
        couponCustomerRelevanceMapper.insert(couponCustomerRelevance);

        /*3、修改卡券配置已领取人数*/
        coupon.setReceiveQuantity(coupon.getReceiveQuantity() + 1);
        couponMapper.updateById(coupon);

        //todo 修改返回值类型
        coupon.setValidStartTime(couponCustomerRelevance.getValidStartTime());
        coupon.setValidEndTime(couponCustomerRelevance.getValidEndTime());
        return coupon;
    }

    @Override
    public List<Coupon> getOrderExternalCoupon(Integer orderId) {
        OrderManage orderManage = orderManageMapper.selectById(orderId);
        List<OrderServe> orderServeList = orderServeMapper.selectList(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));
        return couponMapper
                .getOrderExternalCoupon(orderManage.getDepartmentId())
                .stream()
                //过滤所有可用在服务上的
                .filter(coupon -> {
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(coupon.getId());
                    canUserServeIds.retainAll(orderServeList.stream().map(OrderServe::getServeId).collect(Collectors.toList()));
                    return ListUtil.isNotEmpty(canUserServeIds);
                })
                .peek(coupon -> {
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(coupon.getId());
                    canUserServeIds.retainAll(orderServeList.stream()
                            .filter(orderServe -> orderServe.getBindInnerCoupon() == 2)
                            .filter(orderServe -> !orderServe.isActivity())
                            .map(OrderServe::getServeId)
                            .collect(Collectors.toList()));
                    coupon.setCanBeUse(ListUtil.isNotEmpty(canUserServeIds));
                })
                //过滤达到可用金额
                .filter(coupon -> coupon.getReachingAmount().compareTo(orderManage.getTotalPrice()) < 0)
                .collect(Collectors.toList());
    }

    @Override
    public Boolean existCouponConflict(Collection<Integer> couponIdList) {
        LinkedList<Integer> linkList = new LinkedList<>(couponIdList);
        if (linkList.size() < 2) {
            return false;
        }
        //终止条件
        else if (linkList.size() == 2) {
            CouponExcludeDetailed couponExcludeDetailed = couponExcludeDetailedMapper.selectExcludeDetailed(linkList.get(0), linkList.get(1));
            return couponExcludeDetailed != null;
        }
        //
        else {
            boolean existCouponConflict = false;
            Integer first = linkList.pollFirst();
            for (Integer couponId : linkList) {
                CouponExcludeDetailed couponExcludeDetailed = couponExcludeDetailedMapper.selectExcludeDetailed(couponId, first);
                if (couponExcludeDetailed != null) {
                    existCouponConflict = true;
                    break;
                }
            }
            if (existCouponConflict) {
                return true;
            }
            return this.existCouponConflict(linkList);
        }
    }

    @Override
    public void checkCouponConflict(Collection<Integer> couponIdList) throws RRException {
        LinkedList<Integer> linkList = new LinkedList<>(couponIdList);


        //终止条件
        if (linkList.size() == 2) {
            Integer couponId = linkList.get(0);
            Integer excludeCouponId = linkList.get(1);

            CouponExcludeDetailed couponExcludeDetailed = couponExcludeDetailedMapper.selectExcludeDetailed(couponId, excludeCouponId);
            if (couponExcludeDetailed != null) {
                Coupon coupon = this.getById(couponId);
                Coupon excludeCoupon = this.getById(excludeCouponId);
                throw new RRException("%s和%s不能同时使用", coupon.getName(), excludeCoupon.getName());
            }
        }
        //
        else if (linkList.size() > 2) {
            Integer first = linkList.pollFirst();
            for (Integer couponId : linkList) {
                CouponExcludeDetailed couponExcludeDetailed = couponExcludeDetailedMapper.selectExcludeDetailed(couponId, first);
                if (couponExcludeDetailed != null) {
                    Coupon coupon = this.getById(couponExcludeDetailed.getCouponId());
                    Coupon excludeCoupon = this.getById(couponExcludeDetailed.getExcludeCouponId());
                    throw new RRException("%s和%s不能同时使用", coupon.getName(), excludeCoupon.getName());
                }
            }

            this.checkCouponConflict(linkList);
        }
    }


}
