package com.gogirl.application.order.member.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.binarywang.wxpay.bean.notify.WxPayOrderNotifyResult;
import com.github.binarywang.wxpay.bean.order.WxPayMpOrderResult;
import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.gogirl.application.market.coupon.CouponService;
import com.gogirl.application.market.discount.DiscountConfigService;
import com.gogirl.application.market.gift.GiftCardCustomerRelevanceService;
import com.gogirl.application.market.gift.GiftCardService;
import com.gogirl.application.order.member.IChargeOrderService;
import com.gogirl.application.order.sys.IWxOrderFlowService;
import com.gogirl.application.store.store.CustomerDepartmentRelevanceService;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.domain.common.xcx.GogirlToken;
import com.gogirl.domain.market.discount.DiscountConfig;
import com.gogirl.domain.market.gift.GiftCard;
import com.gogirl.domain.market.gift.GiftCardCustomerRelevance;
import com.gogirl.domain.order.member.ChargeOrder;
import com.gogirl.domain.order.sys.WxOrderFlow;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.infrastructure.common.config.property.GogirlProperties;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.SessionUtils;
import com.gogirl.infrastructure.mapper.market.discount.DiscountConfigMapper;
import com.gogirl.infrastructure.mapper.order.member.ChargeOrderMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerMapper;
import com.gogirl.infrastructure.service.wx.WxPayService;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-03-08
 */
@Service
@AllArgsConstructor
@Transactional(rollbackFor = Exception.class)
@Slf4j
public class ChargeOrderServiceImpl extends ServiceImpl<ChargeOrderMapper, ChargeOrder> implements IChargeOrderService {

    private final CustomerMapper customerMapper;
    private final DiscountConfigMapper discountConfigMapper;
    private final CustomerBalanceMapper customerBalanceMapper;

    private final CustomerBalanceRecordMapper customerBalanceRecordMapper;
    private final CouponService couponService;

    private final IWxOrderFlowService wxOrderFlowService;

    private final GiftCardService giftCardService;

    private final GiftCardCustomerRelevanceService giftCardCustomerRelevanceService;

    @Override
    public void notifyChargeOrder(WxPayOrderNotifyResult result) {

        if (!result.getReturnCode().equals("SUCCESS") || !result.getResultCode().equals("SUCCESS")) {
            throw new RRException("回调失败");
        }

        Customer customer = customerMapper.selectOne(new LambdaQueryWrapper<Customer>().eq(Customer::getOpenid1, result.getOpenid()));

        CustomerBalance customerBalance = customerBalanceMapper.selectOne(
                new LambdaQueryWrapper<CustomerBalance>().eq(CustomerBalance::getCustomerId, customer.getId()));

        LocalDateTime now = LocalDateTime.now();
        AtomicReference<Integer> currentDayAmountSum = new AtomicReference<>(0);
        Optional.ofNullable(customerBalanceRecordMapper.selectList(
                new LambdaQueryWrapper<CustomerBalanceRecord>()
                        .eq(CustomerBalanceRecord::getCustomerId, 27117)
                        .in(CustomerBalanceRecord::getType, Arrays.asList(CustomerBalanceRecord.TYPE_CHARGE, CustomerBalanceRecord.TYPE_FIRST_CHARGE))
                        .between(CustomerBalanceRecord::getTime, LocalDateTime.of(now.toLocalDate(), LocalTime.MIN), LocalDateTime.of(now.toLocalDate(), LocalTime.MAX))
        )).ifPresent(list -> currentDayAmountSum.set(list.stream().map(CustomerBalanceRecord::getOrderAmount).reduce(0, Integer::sum)));

        DiscountConfig discountConfig = discountConfigMapper.selectList(new LambdaQueryWrapper<DiscountConfig>().eq(DiscountConfig::getBrandId, customer.getBrandId()))
                .stream().filter(config -> config.getChargeAmount() <= result.getTotalFee() + currentDayAmountSum.get())
                .max(Comparator.comparing(DiscountConfig::getChargeAmount)).orElseThrow(NullPointerException::new);

        try {
            couponService.sendCoupon(discountConfig.getCouponId(), customer.getId());
        } catch (RRException e) {
            log.error(e.getMessage());
        }

        //礼品卡
        if (discountConfig.getGiftCardId() != null) {
            GiftCard giftCard = giftCardService.getById(discountConfig.getGiftCardId());
            if (giftCard != null) {
                GiftCardCustomerRelevance giftCardCustomerRelevance = GiftCardCustomerRelevance.getInstance(giftCard, customer.getId());
                giftCardCustomerRelevanceService.save(giftCardCustomerRelevance);
            }
        }


        //首次充值
        if (customerBalance == null || (customerBalance.getBalance() == 0 && customerBalanceRecordMapper.selectCount(
                new LambdaQueryWrapper<CustomerBalanceRecord>().eq(CustomerBalanceRecord::getCustomerId, customer.getId())
                        .in(CustomerBalanceRecord::getType, Lists.newArrayList(CustomerBalanceRecord.TYPE_CHARGE, CustomerBalanceRecord.TYPE_FIRST_CHARGE))
        ) == 0)) {
            if (customerBalance == null) {
                customerBalance = CustomerBalance.builder()
                        //充值金额+赠送金额
                        .balance(result.getTotalFee())
                        .customerId(customer.getId())
                        .firstChargeTime(new Date())
                        .level(discountConfig.getLevel())
                        //赠送金额
                        .totalBestow(0)
                        //总充值金额
                        .totalCharge(result.getTotalFee())
                        .totalExpenditure(0)
                        .updateTime(new Date())
                        .brandId(customer.getBrandId())
                        .build();
                customerBalance.setDiscountConfig(discountConfig);
            } else {
                customerBalance.setBalance(result.getTotalFee());
                customerBalance.setFirstChargeTime(new Date());
                customerBalance.setLevel(discountConfig.getLevel());
                customerBalance.setTotalBestow(0);
                customerBalance.setTotalCharge(result.getTotalFee());
                customerBalance.setTotalExpenditure(0);
                customerBalance.setUpdateTime(new Date());
                customerBalance.setDiscountConfig(discountConfig);
            }


            //会员卡不存在
            if (customerBalance.getId() == null) {
                customerBalanceMapper.insert(customerBalance);
            }
            //会员卡存在
            else {
                customerBalanceMapper.updateById(customerBalance);
            }

            CustomerBalanceRecord customerBalanceRecord = CustomerBalanceRecord.builder()
                    .time(new Date())
                    .orderId(result.getOutTradeNo())
                    .orderAmount(result.getTotalFee())
                    .discount(discountConfig.getDiscount())
                    .currentBalance(customerBalance.getBalance())
                    .bestowAmount(0)
                    //fF首次充值
                    .type(2)
                    .customerId(customer.getId())
                    //微信支付
                    .source(1)
                    .brandId(customer.getBrandId())
                    .build();
            customerBalanceRecordMapper.insert(customerBalanceRecord);
        }

        //充值
        else {
            //更新用户余额
            customerBalance.setBalance(customerBalance.getBalance() + result.getTotalFee());
            customerBalance.setLevel(discountConfig.getLevel());
            customerBalance.setUpdateTime(new Date());

            //总送
            customerBalance.setTotalCharge(customerBalance.getTotalCharge() + result.getTotalFee());
            customerBalanceMapper.updateById(customerBalance);

            //
            CustomerBalanceRecord customerBalanceRecord = CustomerBalanceRecord.builder()
                    .time(new Date())
                    .orderId(result.getOutTradeNo())
                    .orderAmount(result.getTotalFee())
                    .discount(discountConfig.getDiscount())
                    .currentBalance(customerBalance.getBalance())
                    .bestowAmount(0)
                    //首次充值
                    .type(1)
                    .customerId(customer.getId())
                    //微信支付
                    .source(1)
                    .brandId(customer.getBrandId())
                    .build();
            customerBalanceRecordMapper.insert(customerBalanceRecord);
        }

        /*2、写入充值订单*/
        ChargeOrder chargeOrder = ChargeOrder.builder()
                .chargeOrderNo(result.getOutTradeNo())
                .chargeAmount(new BigDecimal(result.getTotalFee() / 100))
                .discountConfigId(discountConfig.getId())
                .customerId(customer.getId())
                .status(1)
                .brandId(customer.getBrandId())
                .build();

        this.save(chargeOrder);

        //微信流水
        WxOrderFlow wxOrderFlow = WxOrderFlow.getInstance(result);
        wxOrderFlow.setSourceFrom(WxOrderFlow.CUSTOMER);
        wxOrderFlow.setOutTradeNo(result.getOutTradeNo());
        wxOrderFlow.setOrderType(WxOrderFlow.CHARGE_ORDER);
        wxOrderFlow.setBrandId(customer.getBrandId());
        wxOrderFlowService.log(wxOrderFlow);


    }

    private final GogirlProperties gogirlProperties;
    private final WxPayService wxPayService;

    @Override
    public WxPayMpOrderResult customerChargeBalance(Integer chargeId) throws UnknownHostException, WxPayException {

        GogirlToken gt = SessionUtils.getCustomerToken();

        DiscountConfig discountConfig = discountConfigMapper.selectById(chargeId);

        SimpleDateFormat yyyyMMddHHmmss = new SimpleDateFormat("yyyyMMddHHmmss");
        WxPayUnifiedOrderRequest request = WxPayUnifiedOrderRequest.newBuilder()
                .deviceInfo("设备号")
                .body("充值" + discountConfig.getChargeAmount() / 100 + "赠送" + discountConfig.getBestowAmount() / 100)
                .detail("详情")
                .attach("c")
                .outTradeNo(IdWorker.getIdStr())
                .totalFee(discountConfig.getChargeAmount())
                .spbillCreateIp(InetAddress.getLocalHost().getHostAddress())
                .timeStart(yyyyMMddHHmmss.format(new Date()))
                .notifyUrl(gogirlProperties.getNotifyCharge())
                .tradeType("JSAPI")
                .openid(SessionUtils.getCustomerToken().getOpenid())
                .sceneInfo("gogirl美甲美睫沙龙")
                .build();

        WxPayMpOrderResult result = wxPayService.createJsapiOrder(request, gt.getBrandId());
        return result;
    }

    private final CustomerService customerService;
    private final CustomerDepartmentRelevanceService customerDepartmentRelevanceService;
    private final DiscountConfigService discountConfigService;

    @Override
    public void chargeCustomerCard(String name, String phone, Double amount, Integer source, String refereeId, String remark) {
        GogirlToken gogirlToken = SessionUtils.getTechnicianToken();
        Customer customer = customerService.selectByPhone(phone, gogirlToken.getBrandId());
        Integer departmentId = gogirlToken.getDepartmentId();

        log.info("{} 充值给 {}", gogirlToken.getTechnicianId(), phone);

        //新建会员
        if (customer == null) {
            customer = new Customer();
            customer.setStoreRecordRealName(name);
            customer.setPhone(phone);
            customer.setBrandId(gogirlToken.getBrandId());
            customerService.save(customer);
        }

        //更新会员的店员记录名字
        else if (name != null && !name.isEmpty()) {
            customer.setStoreRecordRealName(name);
            customerService.updateById(customer);
        }

        //店铺会员关联
        customerDepartmentRelevanceService.insertDepartmentRelevanceIfNotExist(customer.getId(), departmentId, 4, new Date(), gogirlToken.getBrandId());

        //检查当天是否存在其他充值记录
        LocalDateTime now = LocalDateTime.now();
        AtomicReference<Integer> currentDayAmountSum = new AtomicReference<>(0);
        Optional.ofNullable(customerBalanceRecordMapper.selectList(
                new LambdaQueryWrapper<CustomerBalanceRecord>()
                        .eq(CustomerBalanceRecord::getCustomerId, customer.getId())
                        .in(CustomerBalanceRecord::getType, Arrays.asList(CustomerBalanceRecord.TYPE_CHARGE, CustomerBalanceRecord.TYPE_FIRST_CHARGE))
                        .between(CustomerBalanceRecord::getTime, LocalDateTime.of(now.toLocalDate(), LocalTime.MIN), LocalDateTime.of(now.toLocalDate(), LocalTime.MAX))
        )).ifPresent(list -> currentDayAmountSum.set(list.stream().map(CustomerBalanceRecord::getOrderAmount).reduce(0, Integer::sum)));

        //折扣信息
        DiscountConfig discountConfig = discountConfigService.selectByCharge((int) (amount * 100) + currentDayAmountSum.get(), gogirlToken.getBrandId());

        //会员卡
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>()
                .eq(CustomerBalance::getCustomerId, customer.getId()));

        //礼品卡
        if (discountConfig.getGiftCardId() != null) {
            GiftCard giftCard = giftCardService.getById(discountConfig.getGiftCardId());
            if (giftCard != null) {
                GiftCardCustomerRelevance giftCardCustomerRelevance = GiftCardCustomerRelevance.getInstance(giftCard, customer.getId());
                giftCardCustomerRelevanceService.save(giftCardCustomerRelevance);
            }
        }

        //充值类型
        int type;
        if (customerBalance == null) {
            type = 2;
        } else if (customerBalance.getBalance() == 0 && customerBalanceRecordMapper.selectCount(
                new LambdaQueryWrapper<CustomerBalanceRecord>().eq(CustomerBalanceRecord::getCustomerId, customer.getId())
                        .in(CustomerBalanceRecord::getType, Lists.newArrayList(CustomerBalanceRecord.TYPE_CHARGE, CustomerBalanceRecord.TYPE_FIRST_CHARGE))) == 0) {
            type = 2;
        } else {
            type = 1;
        }

        //如果会员卡不存在就新增会员卡
        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(customer.getId(), gogirlToken.getBrandId());
            customerBalanceMapper.insert(customerBalance);
        }
        int chargeAmount = (int) (amount * 100);

        //充值订单id
        String orderId = IdWorker.getIdStr();

        //更新余额
        customerBalance.setBalance(customerBalance.getBalance() + chargeAmount);
        customerBalance.setTotalCharge(customerBalance.getTotalCharge() + chargeAmount);
        customerBalance.setUpdateTime(new Date());

        //会员等级
        customerBalance.setLevel(discountConfig.getLevel());

        //折扣率
        customerBalance.setDiscountConfig(discountConfig);
        customerBalanceMapper.updateById(customerBalance);


        //充值流水
        CustomerBalanceRecord customerBalanceRecord = CustomerBalanceRecord.builder()
                .currentBalance(customerBalance.getBalance())
                .customerId(customer.getId())
                //充送金额由定时任务计算
                .bestowAmount(0)
                .departmentId(gogirlToken.getDepartmentId())
                .departmentName(gogirlToken.getDepartmentName())
                .discount(discountConfig.getDiscount())
                .orderAmount(chargeAmount)
                .orderId(orderId)
                //支付方式
                .source(source)
                .refereeId(refereeId)
                .remark(remark)
                //订单类型
                .type(type)
                .time(new Date())
                .build();
        customerBalanceRecordMapper.insert(customerBalanceRecord);

        try {
            if (discountConfig.getCouponId() != null && discountConfig.getCouponId() != 0) {
                couponService.sendCoupon(discountConfig.getCouponId(), customer.getId());
            }
        } catch (Exception e) {
            log.error("美甲师充值送券失败 msg:{}", e.getMessage());
        }
    }
}
