package com.gogirl.application.market.timescard.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gogirl.application.market.timescard.TimesCardService;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.domain.market.timescard.TimesCardCustomerRelevance;
import com.gogirl.domain.market.timescard.TimesCardType;
import com.gogirl.domain.market.timescard.TimesCardTypeContent;
import com.gogirl.domain.order.market.TimesCardOrder;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.infrastructure.common.util.SessionUtils;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardTypeContentMapper;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardTypeMapper;
import com.gogirl.infrastructure.mapper.order.market.TimesCardOrderMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.shared.user.ConsumerCommand;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;

@Service
@AllArgsConstructor
@Slf4j
public class TimesCardServiceImpl extends ServiceImpl<TimesCardTypeMapper, TimesCardType> implements TimesCardService {

    private final TimesCardTypeContentMapper timesCardTypeContentMapper;
    private final TimesCardTypeMapper timesCardTypeMapper;
    private final CustomerBalanceService customerBalanceService;
    private final CustomerBalanceMapper customerBalanceMapper;
    private final TimesCardCustomerRelevanceMapper timesCardCustomerRelevanceMapper;
    private final TimesCardOrderMapper timesCardOrderMapper;

    public List<TimesCardType> getTimesCardTypeList(Integer brandId) {

        return timesCardTypeMapper.selectList(new LambdaQueryWrapper<TimesCardType>()
                .eq(TimesCardType::getStatus, TimesCardType.STATUS_ON)
                .eq(TimesCardType::getBrandId, brandId));
    }

    public TimesCardType getAggregate(Integer id) {
        TimesCardType timesCardType = this.getById(id);
        List<TimesCardTypeContent> timesCardTypeContentList = timesCardTypeContentMapper.selectList(
                new LambdaQueryWrapper<TimesCardTypeContent>().eq(TimesCardTypeContent::getCardTypeId, id)
        );
        return timesCardTypeMapper.selectAggregate(id);
    }

    @Override
    public TimesCardType getTimesCardTypeDetail(Integer id) {
        return timesCardTypeMapper.selectAggregate(id);
    }

    @Override
    @Transactional
    public void balancePayTimesCard(Integer timesCardId, Integer customerId, Integer brandId) {
        TimesCardType timesCardType = this.getTimesCardTypeDetail(timesCardId);
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>()
                .eq(CustomerBalance::getCustomerId, customerId));

        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(customerId, brandId);
            customerBalanceMapper.insert(customerBalance);
        }

        //总需要付款金额（分）=单次业绩单价（元）*总次数*100*折扣率
        BigDecimal totalPayAmount = timesCardType.getPayAmount()
                .multiply(new BigDecimal(timesCardType.getSumTimes()))
                .multiply(new BigDecimal("100"))
                .multiply(customerBalance.getDiscountRate())
                .setScale(0, RoundingMode.HALF_DOWN);

        /*1、次卡*/
        TimesCardCustomerRelevance timesCardCustomerRelevance = TimesCardCustomerRelevance.builder()
                //次卡类型id
                .cardTypeId(timesCardId)
                //次卡类型名称
                .name(timesCardType.getName())
                //会员id
                .customerId(customerId)
                .sumTimes(timesCardType.getSumTimes())
                .usedTimes(0)
                .createTime(new Date())
                .discountAmount(timesCardType.getDiscountAmount().add(timesCardType.getPayAmount().subtract(totalPayAmount.divide(new BigDecimal("100")))))
                .payAmount(totalPayAmount.divide(new BigDecimal("100")))
                .refereeId(null)
                .status(TimesCardCustomerRelevance.STATUS_NORMAL)
                .validStartTime(new Date())
                .validEndTime(new Date(System.currentTimeMillis() + timesCardType.getValidDays() * 24 * 60 * 60 * 1000L))
                .brandId(timesCardType.getBrandId())
                .build();

        timesCardCustomerRelevanceMapper.insert(timesCardCustomerRelevance);

        String outTradeNo = IdWorker.getIdStr();

        TimesCardOrder timesCardOrder = TimesCardOrder.builder()
                .timesCardOrderNo(outTradeNo)
                .timesCardCustomerRelevanceId(timesCardCustomerRelevance.getId())
                .timesCardName(timesCardCustomerRelevance.getName())
                .timesCardTypeId(timesCardCustomerRelevance.getCardTypeId())
                .createTime(LocalDateTime.now())
                .customerId(timesCardCustomerRelevance.getCustomerId())
                .payAmount(timesCardCustomerRelevance.getPayAmount())
                .status(1)
                .brandId(timesCardType.getBrandId())
                .build();

        timesCardOrderMapper.insert(timesCardOrder);

        /*1、扣除余额*/
        ConsumerCommand consumerCmd = ConsumerCommand.builder()
                .amount(totalPayAmount.intValue())
                .customerId(customerId)
                //.departmentId(mallOrder.getSelfTakeDepartmentId())
                .orderId(timesCardOrder.getId().longValue())
                //.orderStatus(mallOrder.getStatus())
                .brandId(timesCardOrder.getBrandId())
                .type(6) //次卡
                .build();
        customerBalanceService.consumer(consumerCmd);
    }


}
