package com.gogirl.interfaces.xcx;

import com.github.binarywang.wxpay.bean.order.WxPayMpOrderResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.pagehelper.PageInfo;
import com.gogirl.application.order.member.IChargeOrderService;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.order.serve.PayService;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.infrastructure.util.lock.OrderIdLock;
import com.gogirl.shared.member.order.command.BalanceWxPayQuery;
import com.gogirl.shared.member.order.query.UnifiedOrderQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.net.UnknownHostException;
import java.util.Optional;


@RestController
@Api(tags = {"version3.0: 预约订单支付接口"}, value = "支付")
@AllArgsConstructor
@Slf4j
public class PayController {


    /**
     * domain service
     */
    private final GogirlTokenService gogirlTokenService;
    private final CustomerBalanceService balanceService;
    private final OrderManageService orderManageService;


    /**
     * cqrs service
     */
    private final PayService payService;


    @ApiOperation(value = "微信支付预约")
    @PostMapping("/customer/pay/unifiedOrder")
    public JsonResult<WxPayMpOrderResult> unifiedOrder(@RequestHeader String token,
                                                       @RequestParam @NotNull Integer orderId) throws WxPayException, UnknownHostException {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        log.info("**微信支付订单**,会员:{}，订单id:{} ", currentCustomerId, orderId);

        String openid = gogirlTokenService.getByToken(token).getOpenid();
        UnifiedOrderQuery qry = new UnifiedOrderQuery(currentCustomerId, orderId, openid);
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            WxPayMpOrderResult result = payService.unifiedOrder(qry);
            return JsonResult.success(result);
        } finally {
            lock.unlock(orderId);
        }
    }

    @ApiOperation(value = "会员卡支付预约订单")
    @PostMapping("/customer/pay/consumeBalance")
    public JsonResult<Void> consumeBalance(@RequestHeader String token,
                                           @RequestParam Integer orderId) {
        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        log.info("**会员卡支付订单**,会员:{},订单id:{}", currentCustomerId, orderId);
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            payService.consumeBalance(currentCustomerId, orderId);
        } finally {
            lock.unlock(orderId);
        }
        return JsonResult.success();
    }

    @ApiOperation(value = "申请支付预约订单，待店员确认是否完成订单。5：刷pos机7：大众点评团购")
    @PostMapping("/customer/pay/chosePayType")
    public JsonResult<Void> chosePayType(@RequestHeader String token,
                                         @RequestParam @NotNull Integer orderId,
                                         @RequestParam @NotNull Integer payType) {

        log.info("*****用户选择支付方式,支付类型{}", payType);
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            payService.chosePayType(orderId, payType);
        } finally {
            lock.unlock(orderId);
        }
        return JsonResult.success();
    }

    @ApiOperation(value = "余额微信支付")
    @GetMapping("/customer/pay/balanceWxPay")
    public JsonResult<WxPayMpOrderResult> balanceWxPay(@RequestHeader String token,
                                                       @RequestParam @NotNull Integer orderId) throws WxPayException, UnknownHostException {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));
        log.info("**余额微信支付订单**,会员:{}，订单id:{} ", currentCustomerId, orderId);

        String openid = gogirlTokenService.getByToken(token).getOpenid();
        BalanceWxPayQuery qry = new BalanceWxPayQuery(currentCustomerId, orderId, openid);
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            WxPayMpOrderResult result = payService.balanceWxPay(qry);
            return JsonResult.success(result);
        } finally {
            lock.unlock(orderId);
        }
    }

    @ApiOperation(value = "预约订单会员余额和pos支付")
    @GetMapping("/customer/pay/balancePosPay")
    public JsonResult<Void> balancePosPay(@RequestHeader String token,
                                          @RequestParam @NotNull Integer orderId) throws WxPayException, UnknownHostException {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));
        log.info("**余额pos支付订单**,会员:{}，订单id:{} ", currentCustomerId, orderId);
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            payService.balancePosPay(currentCustomerId, orderId);
        } finally {
            lock.unlock(orderId);
        }
        return JsonResult.success();
    }

    private final IChargeOrderService chargeOrderService;

    @ApiOperation(value = "用户端充值会员卡")
    @PostMapping("/customer/pay/chargeBalance")
    public JsonResult<?> customerChargeBalance(@RequestHeader String token,
                                               @RequestParam Integer chargeId) throws UnknownHostException, WxPayException {
        log.info("获取微信支付签名token:" + token + "；chargeId:" + chargeId);
        WxPayMpOrderResult result = chargeOrderService.customerChargeBalance(chargeId);
        return JsonResult.success(result);
    }

    @ApiOperation("朋友代付")
    @GetMapping("/customer/pay/friendPayOrder/{orderId}")
    public JsonResult<Integer> friendPayOrder(@RequestHeader String token,
                                              @PathVariable Integer orderId,
                                              @ApiParam("朋友手机号") @RequestParam String phone) {
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            orderManageService.friendPayOrder(orderId, phone);
        } finally {
            lock.unlock(orderId);
        }
        return JsonResult.success();
    }


    @ApiOperation(value = "查看消费记录")
    @GetMapping("/customer/pay/getBalanceRecordCard")
    public JsonResult<PageInfo<CustomerBalanceRecord>> getBalanceRecordCard(Integer customerId,
                                                                            int pageNum,
                                                                            int pageSize) {
        PageInfo<CustomerBalanceRecord> pageInfo = balanceService.getBalanceRecordCard(customerId, pageNum, pageSize);
        return JsonResult.success(pageInfo);
    }


    @ApiOperation(value = "美甲师端确认收款。2：会员卡收款；5：刷pos机；7：大众点评团购，8.微信扫二维码支付")
    @PostMapping("/technician/pay/receiveMoney")
    public JsonResult<Void> receiveMoney(@RequestHeader String token,
                                         @NotNull Integer orderId,
                                         @NotNull Integer payType) {

        GogirlToken gt = gogirlTokenService.getByToken(token);
        Integer technicianId = SessionUtils.getTechnicianId();
        OrderIdLock lock = OrderIdLock.getInsatance();
        try {
            lock.lock(orderId);
            orderManageService.confirmOrder(orderId, payType, technicianId);
        } finally {
            lock.unlock(orderId);
        }

        return JsonResult.success();
    }


    @ApiOperation(value = "美甲师端，充值会员卡")
    @PostMapping("/technician/pay/chargeCustomerCard")
    public JsonResult<Void> chargeCustomerCard(@RequestHeader String token,
                                               String name,
                                               String phone,
                                               Double amount,
                                               Integer source,
                                               String refereeId,
                                               String remark) {
        chargeOrderService.chargeCustomerCard(name, phone, amount, source, refereeId, remark);
        return JsonResult.success();
    }
}
