package com.gogirl.interfaces.wxappmember.user;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.gogirl.application.market.CouponService;
import com.gogirl.application.market.MarketService;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.market.serve.Coupon;
import com.gogirl.domain.market.serve.CouponCustomerRelevance;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.user.Customer;
import com.gogirl.domain.xcx.GogirlConfig;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.common.util.StringUtils;
import com.gogirl.infrastructure.mapper.market.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.CouponMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.xcx.GogirlConfigMapper;
import com.gogirl.shared.market.command.SetUpOuterCouponCommand;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

@RestController
@Api(tags = {"version2.0: 优惠券"}, value = "version2.0: 优惠券")
@AllArgsConstructor
@Slf4j
public class CouponController {

    private final CouponService couponService;
    private final GogirlTokenService gogirlTokenService;
    private final OrderManageService orderManageService;
    private final GogirlConfigMapper gogirlConfigMapper;
    private final CouponCustomerRelevanceMapper couponCustomerRelevanceMapper;
    private final OrderServeMapper orderServeMapper;
    private final MarketService marketService;
    private final CouponMapper couponMapper;

    @ApiOperation(value = "查询优惠券详情", notes = "查询优惠券详情")
    @GetMapping("/customer/selectCouponByPrimaryKey")
    public JsonResult<Coupon> selectCouponByPrimaryKey(Integer id) {
        Coupon coupon = couponService.getById(id);
        if (coupon == null) {
            throw new RRException(500, "无该优惠券");
        } else {
            return new JsonResult<>(true, JsonResult.APP_DEFINE_SUC, coupon);
        }

    }

    @ApiOperation(value = "用户根据订单号查询可用优惠券", notes = "用户根据订单号查询可用优惠券")
    @GetMapping("/customer/getOrderCoupon")
    public JsonResult<List<CouponCustomerRelevance>> getOrderCoupon(@RequestHeader String token,
                                                                    @RequestParam Integer orderId) {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getTokenByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        List<OrderServe> orderServeList = orderServeMapper.selectList(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));
        OrderManage orderManage = orderManageService.getById(orderId);
        List<CouponCustomerRelevance> list = couponCustomerRelevanceMapper
                .selectList(new LambdaQueryWrapper<CouponCustomerRelevance>()
                        .eq(CouponCustomerRelevance::getCustomerId, currentCustomerId)
                        .eq(CouponCustomerRelevance::getSourceType, 0)
                        .ne(CouponCustomerRelevance::getState, 2)
                        .ne(CouponCustomerRelevance::getState, 3))
                .stream()
                //过滤所有可用在服务上的
                .filter(couponCustomerRelevance -> {
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(couponCustomerRelevance.getCouponId());
                    canUserServeIds.retainAll(orderServeList.stream().map(OrderServe::getServeId).collect(Collectors.toList()));
                    return ListUtil.isNotEmpty(canUserServeIds);
                })
                //过滤可用
                .filter(CouponCustomerRelevance::isValid)
                //过滤达到可用金额
                .filter(couponCustomerRelevance -> couponCustomerRelevance.getReachingAmount().compareTo(orderManage.getTotalPrice()) < 0)
                .collect(Collectors.toList());

        return JsonResult.success(list);

    }

    @ApiOperation(value = "我的优惠券列表,查所有时不用传pageNum,pageSize")
    @GetMapping("/customer/getMyCoupon")
    public JsonResult<PageInfo<CouponCustomerRelevance>> getMyCoupon(String token,
                                                                     Integer pageNum,
                                                                     Integer pageSize,
                                                                     Integer state,
                                                                     Boolean containRemoveCoupon) {
        if (token == null) {
            return new JsonResult<>(false, String.format("PARAM_NULL", "token"), null);
        }
        GogirlToken gogirlToken = gogirlTokenService.getTokenByToken(token);
        if (gogirlToken == null) {
            return new JsonResult<>(false, "TOKEN_NULL_ERR", null);
        }
        if (gogirlToken.getCustomerId() == null) {
            return new JsonResult<>(false, "TOKEN_NULL_AUTHORIZED_MSG", null);
        }
        Integer customerId = gogirlToken.getCustomerId();
        log.info("查询领取优惠券的用户列表CouponCustomerRelevance:" + customerId.toString());
        if (pageNum != null && pageSize != null) {
            PageHelper.startPage(pageNum, pageSize);
        }
        List<CouponCustomerRelevance> list = couponService.selectMyCoupon(customerId);
        long nowTime = new Date().getTime();
        for (int i = 0; i < list.size(); i++) {
            if (containRemoveCoupon != null && containRemoveCoupon == false && list.get(i).getCouponId().equals(66)) {//移除卸甲券
                list.remove(i);
                i--;
                continue;
            }
            CouponCustomerRelevance couponCustomerRelevance = list.get(i);
            if (nowTime > couponCustomerRelevance.getValidEndTime().getTime() && couponCustomerRelevance.getState() == 1) {
                couponCustomerRelevance.setState(3);
            }
            if (state != null && state != 0 && couponCustomerRelevance != null && couponCustomerRelevance.getState() != null && !couponCustomerRelevance.getState().equals(state)) {
                list.remove(i);
                i--;
            }
        }
        PageInfo<CouponCustomerRelevance> pageInfo = new PageInfo<>(list);
        return new JsonResult<>(true, JsonResult.APP_DEFINE_SUC, pageInfo);
    }

    @ApiOperation(value = "查询是否可领新人优惠券,是新人返回券信息,不是新人返回null;")
    @GetMapping("/customer/checkNewCustomer")
    public JsonResult<Coupon> checkNewCustomer(String token) {
        if (token == null) {
            return new JsonResult<>(false, String.format("PARAM_NULL", "token"), null);
        }
        GogirlToken gogirlToken = gogirlTokenService.getTokenByToken(token);
        if (gogirlToken == null) {
            return new JsonResult<>(false, "TOKEN_NULL_ERR", null);
        }
        if (gogirlToken.getCustomer() == null) {
            return new JsonResult<>(false, "TOKEN_NULL_AUTHORIZED_MSG", null);
        }

        //顾客id
        Integer customerId = gogirlToken.getCustomer().getId();

        //新用户配置
        GogirlConfig gogirlConfig = gogirlConfigMapper.selectByPrimaryKey(2);

        if (gogirlConfig == null
                || gogirlConfig.getValue() == null
                || gogirlConfig.getValue().isEmpty()) {
            return new JsonResult<>(false, "请联系管理员配置发放哪张新人优惠券");
        }

        List<Integer> couponIdList = Arrays.stream(gogirlConfig.getValue().split(",")).map(Integer::valueOf).collect(Collectors.toList());
        if (ListUtil.isEmpty(couponIdList)) {
            return new JsonResult<>(false, "请联系管理员配置发放哪张新人优惠券");
        }

        //判断是否有领过相关的优惠券
        List<CouponCustomerRelevance> list = couponService.selectMyNewCustomerCoupon(customerId, couponIdList);
        if (list.size() > 0) {
            return new JsonResult<>(true, "您曾经领取过新人优惠券", null);
        }

        Integer orderTimes = orderManageService.countOrderTimes(gogirlToken.getCustomerId());

        //没有订单,确认是新人,发券
        if (orderTimes == null || orderTimes == 0) {
            Coupon c = couponService.getById(couponIdList.get(0));

            return new JsonResult<>(true, JsonResult.APP_DEFINE_SUC, c);
        }

        return new JsonResult<>(true, "只有新客才能领取优惠券", null);
    }

    //新人优惠券,后台配置具体发那张券
    @ApiOperation(value = "新人优惠券,后台配置具体发那张券")
    @PostMapping("/customer/sendNewCustomerCouponXcx")
    public JsonResult<Coupon> sendNewCustomerCouponXcx(String token) {

        GogirlToken gogirlToken = gogirlTokenService.getTokenByToken(token);

        Integer customerId = gogirlToken.getCustomer().getId();

        GogirlConfig gogirlConfig = gogirlConfigMapper.selectByPrimaryKey(2);
        if (gogirlConfig == null || gogirlConfig.getValue() == null || gogirlConfig.getValue().isEmpty()) {
            return new JsonResult<>(false, "请联系管理员配置发放哪张新人优惠券");
        }
        String[] couponIds = gogirlConfig.getValue().split(",");
        Integer couponId = null;
        List<Integer> coupontIdList = new ArrayList<>();
        for (int i = 0; i < couponIds.length; i++) {
            if (couponIds[i] != null && !couponIds[i].isEmpty()) {
                Integer itemId = Integer.parseInt(couponIds[0]);
                if (i == 0) {
                    couponId = itemId;
                }
                coupontIdList.add(itemId);
            }
        }
        if (coupontIdList.size() > 0) {//判断是否有领过相关的优惠券
            List<CouponCustomerRelevance> list = couponService.selectMyNewCustomerCoupon(customerId, coupontIdList);
            if (list.size() > 0) {
                return new JsonResult<>(false, "您曾经领取过新人优惠券", null);
            }
        }
        log.info("发放优惠券customerId:" + customerId.toString());
        log.info("发放优惠券couponId:" + couponId.toString());

        Customer customer = gogirlToken.getCustomer();
        //返回消息
        return sendCoupon(couponId, customerId, customer);
    }

    //小程序发券
    @ApiOperation(value = "小程序发券")
    @PostMapping("/customer/sendCouponXcx")
    public JsonResult<Coupon> sendCouponXcx(String token, Integer couponId) {
        if (token == null) {
            return new JsonResult<>(false, String.format("PARAM_NULL", "token"), null);
        }
        GogirlToken gogirlToken = gogirlTokenService.getTokenByToken(token);
        if (gogirlToken == null) {
            return new JsonResult<>(false, "TOKEN_NULL_ERR", null);
        }
        if (gogirlToken.getCustomer() == null) {
            return new JsonResult<>(false, "TOKEN_NULL_AUTHORIZED_MSG", null);
        }
        Integer customerId = gogirlToken.getCustomer().getId();
        if (couponId == null) {
            return new JsonResult<>(false, String.format("PARAM_NULL", "couponId"), null);
        }
        log.info("发放优惠券customerId:" + customerId.toString());
        log.info("发放优惠券couponId:" + couponId.toString());

        Customer customer = gogirlToken.getCustomer();
        //返回消息
        return sendCoupon(couponId, customerId, customer);
    }

    private JsonResult<Coupon> sendCoupon(Integer couponId, Integer customerId, Customer customer) {
        Coupon coupon = couponService.getById(couponId);
        if (coupon == null) {
            return new JsonResult<>(false, "找不到该优惠券", null);
        }

        //判断优惠券是否还有
        if (coupon.getAllQuantity() <= coupon.getReceiveQuantity()) {
            return new JsonResult<>(false, "优惠券已经领完", null);
        }
        //判断该用户是否限领,null过,0过,有但限制不过,不限制过
        CouponCustomerRelevance record = new CouponCustomerRelevance();
        record.setCustomerId(customerId);
        record.setCouponId(couponId);
        int row = couponService.countRelevanceNum(record);
        if (coupon.getLimitQuantity() != null && coupon.getLimitQuantity() != 0 && coupon.getLimitQuantity() <= row) {
            return new JsonResult<>(false, "抱歉,该优惠券最多可领取" + coupon.getLimitQuantity() + "张,你已有" + row + "张该优惠券.", null);
        }

        CouponCustomerRelevance couponCustomerRelevance = new CouponCustomerRelevance();
        couponCustomerRelevance.setCustomerId(customerId);
        couponCustomerRelevance.setCouponId(couponId);
        couponCustomerRelevance.setState(1);
        couponCustomerRelevance.setCode(couponService.getRandomCode());
        couponCustomerRelevance.setReceiveTime(new Date());

        //冗字段
        couponCustomerRelevance.setCouponName(coupon.getName());
        couponCustomerRelevance.setDiscountAmount(coupon.getDiscountAmount());
        couponCustomerRelevance.setReachingAmount(coupon.getReachingAmount());
        couponCustomerRelevance.setDiscountPercent(coupon.getDiscountPercent());
        couponCustomerRelevance.setType(coupon.getType());
        couponCustomerRelevance.setSourceType(coupon.getSourceType());
        couponCustomerRelevance.setCondition1(coupon.getCondition1());
        couponCustomerRelevance.setPayAmount(coupon.getPayAmount());
        couponCustomerRelevance.setSourceType(coupon.getSourceType());
        couponCustomerRelevance.setSourceFrom("新人券");

        if (coupon.getValidType() == 1) {
            couponCustomerRelevance.setValidStartTime(coupon.getValidStartTime());//优惠券原有开始和结束时间
            couponCustomerRelevance.setValidEndTime(coupon.getValidEndTime());
        } else if (coupon.getValidType() == 2) {
            long today = new Date().getTime();
            long day7 = today + 86400000L * coupon.getValidDate();
            couponCustomerRelevance.setValidStartTime(new Date(today));//从现在开始
            couponCustomerRelevance.setValidEndTime(new Date(day7));//七天后过期
        }
        int id = couponService.insertSelective(coupon, couponCustomerRelevance);


        coupon.setValidStartTime(couponCustomerRelevance.getValidStartTime());
        coupon.setValidEndTime(couponCustomerRelevance.getValidEndTime());
        return new JsonResult<>(true, JsonResult.APP_DEFINE_SUC, coupon);
    }

    @ApiOperation(value = "美甲师提交外部券")
    @PostMapping("/technician/techSubmitCoupons")
    public JsonResult<Void> techSubmitCoupons(@RequestHeader String token,
                                              @RequestParam Integer orderId,
                                              @RequestParam String couponIds) {
        StoreTechnician storeTechnician = gogirlTokenService.getTokenByToken_t(token).getStoreTechnician();
        log.info("美甲师 {} 提交外部券,couponId", storeTechnician.getName());

        List<Integer> idList = StringUtils.isNotEmpty(couponIds)
                ? Lists.newArrayList(couponIds.split(",")).stream().map(Integer::valueOf).collect(Collectors.toList())
                : Lists.newArrayList();

        //查询订单
        OrderManage orderManage = orderManageService.getById(orderId);

        SetUpOuterCouponCommand cmd = SetUpOuterCouponCommand.builder()
                .customerId(orderManage.getOrderUser())
                .orderId(orderId)
                .outerCouponIds(idList)
                .build();
        marketService.setUpOuterCoupon(cmd);
        marketService.setterOrder(orderId);
        return JsonResult.success();
    }

    @ApiOperation(value = "美甲师根据订单号查询可用外部券")
    @GetMapping("/technician/getOrderExternalCoupon")
    public JsonResult<List<Coupon>> getOrderExternalCoupon(@RequestHeader String token,
                                                           @RequestParam Integer orderId) {
        log.info("用户根据订单号查询可用优惠券");
        OrderManage orderManage = orderManageService.getById(orderId);
        List<OrderServe> orderServeList = orderServeMapper.selectList(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));
        List<Coupon> couponList = couponMapper
                .selectList(new LambdaQueryWrapper<Coupon>().eq(Coupon::getSourceType, 1))
                .stream()
                //过滤所有可用在服务上的
                .filter(coupon -> {
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(coupon.getId());
                    canUserServeIds.retainAll(orderServeList.stream().map(OrderServe::getServeId).collect(Collectors.toList()));
                    return ListUtil.isNotEmpty(canUserServeIds);
                })
                //过滤达到可用金额
                .filter(coupon -> coupon.getReachingAmount().compareTo(orderManage.getTotalPrice()) < 0)
                .collect(Collectors.toList());


        return JsonResult.success(couponList);
    }
}
