package com.gogirl.interfaces.market.coupon;

import com.gogirl.application.market.coupon.CouponService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.market.coupon.Coupon;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.infrastructure.util.lock.CustomerIdLock;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@Api(tags = {"优惠券配置接口"}, value = "优惠券配置接口")
@AllArgsConstructor
@Slf4j
public class CouponController {

    private final CouponService couponService;
    private final GogirlTokenService gogirlTokenService;

    @ApiOperation(value = "根据配置id查询优惠券配置详情")
    @GetMapping("/customer/selectCouponByPrimaryKey")
    public JsonResult<Coupon> selectCouponByPrimaryKey(@RequestParam Integer id) {
        Coupon coupon = couponService.getById(id);
        return JsonResult.success(coupon);
    }


    @ApiOperation(value = "查询是否可领新人优惠券,是新人返回券信息,不是新人返回null;")
    @GetMapping("/customer/checkNewCustomer")
    public JsonResult<Coupon> checkNewCustomer(@RequestHeader String token) {
        Integer customerId = SessionUtils.getCustomerId();
        Coupon coupon = couponService.checkNewCustomer(customerId);
        return JsonResult.success(coupon);
    }

    @ApiOperation(value = "发放新人优惠券")
    @PostMapping("/customer/sendNewCustomerCouponXcx")
    public JsonResult<Coupon> sendNewCustomerCouponXcx(@RequestHeader String token) {
        Integer customerId = SessionUtils.getCustomerId();
        CustomerIdLock lock = CustomerIdLock.getInstance();
        try {
            lock.lock(customerId);
            Coupon coupon = couponService.sendNewCustomerCouponXcx(customerId);
            return JsonResult.success(coupon);
        } finally {
            lock.unlock(customerId);
        }
    }

    @ApiOperation(value = "小程序发券")
    @PostMapping("/customer/sendCouponXcx")
    public JsonResult<Coupon> sendCouponXcx(@RequestHeader String token,
                                            @RequestParam Integer couponId) {
        Integer customerId = SessionUtils.getCustomerId();
        log.info("发放优惠券couponId:" + couponId.toString());
        //返回消息
        Coupon coupon = couponService.sendCoupon(couponId, customerId);
        return JsonResult.success(coupon);
    }


    @ApiOperation(value = "美甲师根据订单号查询可用外部券")
    @GetMapping("/technician/getOrderExternalCoupon")
    public JsonResult<List<Coupon>> getOrderExternalCoupon(@RequestHeader String token,
                                                           @RequestParam Integer orderId) {

        log.info("用户根据订单号查询可用优惠券");
        List<Coupon> couponList = couponService.getOrderExternalCoupon(orderId);
        return JsonResult.success(couponList);
    }
}
