package com.gogirl.interfaces.order.serve;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.order.serve.OrderServeService;
import com.gogirl.application.order.serve.ScheduleManageService;
import com.gogirl.application.order.serve.impl.OrderManageDTOAssembler;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.order.serve.OrderComment;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.order.serve.OrderServeDescribe;
import com.gogirl.domain.product.mall.PurchaseSku;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.annotation.GogirlShop;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.mapper.order.mall.PurchaseSkuMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderCommentMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeDescribeMapper;
import com.gogirl.shared.order.serve.query.dto.ordermanage.OrderManageDTO;
import com.gogirl.shared.order.serve.query.qry.ordermanage.ListOrderQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.Optional;

@Api(tags = "version3.0: 预约订单接口")
@RestController
@AllArgsConstructor
@Slf4j
public class OrderManageController {

    /**
     * domain service
     */
    private final GogirlTokenService gogirlTokenService;

    /**
     * domain service
     */
    private final OrderManageService orderManageService;

    private final OrderServeService orderServeService;


    /**
     * repository
     */
    private final OrderServeDescribeMapper orderServeDescribeMapper;
    private final PurchaseSkuMapper purchaseSkuMapper;

    private final OrderCommentMapper orderCommentMapper;
    private final OrderManageDTOAssembler orderManageDTOAssembler;
    private final ScheduleManageService scheduleManageService;

    @ApiOperation(value = "查询我的客照")
    @GetMapping(value = "/customer/ordermanage/queryMyOrderPhoto")
    public JsonResult<Page<OrderManageDTO>> queryMyOrderPhoto(@RequestHeader String token,
                                                              @RequestParam Integer pageNum,
                                                              @RequestParam Integer pageSize) {

        GogirlToken gogirlToken = gogirlTokenService.getTokenByToken(token);
        ListOrderQuery qry = new ListOrderQuery();
        qry.setCustomerId(gogirlToken.getCustomerId());
        qry.setPageNum(pageNum);
        qry.setPageSize(pageSize);

        Page<OrderManageDTO> orderManageDTOPage = (Page<OrderManageDTO>) orderManageService.queryListOrder(qry).convert(orderManageDTOAssembler);
        return JsonResult.success(orderManageDTOPage);
    }

    @ApiOperation(value = "查询我的订单")
    @GetMapping("/customer/ordermanage/queryMyOrderForPage")
    public JsonResult<IPage<OrderManageDTO>> queryMyOrderForPage(@RequestHeader String token,
                                                                 @RequestParam Integer pageNum,
                                                                 @RequestParam Integer pageSize) {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getTokenByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        ListOrderQuery qry = new ListOrderQuery();
        qry.setCustomerId(currentCustomerId);
        qry.setPageNum(pageNum);
        qry.setPageSize(pageSize);

        //不显示未核销订单
        qry.setShowUnCheck(false);

        Page<OrderManageDTO> orderManageDTOPage = (Page<OrderManageDTO>) orderManageService.queryListOrder(qry).convert(orderManageDTOAssembler);
        return JsonResult.success(orderManageDTOPage);
    }

    @ApiOperation(value = "查询订单")
    @GetMapping(value = "/customer/ordermanage/queryOrderForDetail")
    @GogirlShop
    public JsonResult<OrderManageDTO> customerQueryOrderForDetail(@RequestParam Integer id) {
        OrderManage orderManage = orderManageService.queryOrder(id);
        return JsonResult.success(orderManageDTOAssembler.apply(orderManage));
    }

    @ApiOperation(value = "查询所有可选的描述")
    @GetMapping("/technician/ordermanage/listOrderServeDescribe")
    public JsonResult<List<OrderServeDescribe>> listOrderServeDescribe(@RequestParam Integer type) {

        List<OrderServeDescribe> orderServeDescribeList = orderServeDescribeMapper.selectByType(type);
        return JsonResult.success(orderServeDescribeList);
    }

    @ApiOperation("根据skuName查询sku")
    @PostMapping("//technician/ordermanage/selectBySkuName")
    public JsonResult<PurchaseSku> selectBySkuName(String skuName) {
        PurchaseSku purchaseSku = purchaseSkuMapper.selectBySkuName(skuName);
        return JsonResult.success(purchaseSku);
    }

    @ApiOperation("查询款式/色号和款式描述")
    @GetMapping("/technician/ordermanage/getOrderServeDetail")
    public JsonResult<OrderServe> getOrderServeDetail(@RequestParam("orderServeId") Integer orderServeId) {
        OrderServe orderServe = orderServeService.getOrderServeDetail(orderServeId);
        return JsonResult.success(orderServe);
    }

    @ApiOperation("美甲师录入款式/色号和款式描述")
    @PostMapping("/technician/ordermanage/recordOrderServeData")
    @GogirlShop
    public JsonResult<Void> recordOrderServeData(@RequestBody OrderServe param) {
        orderManageService.recordOrderServeData(param);
        return JsonResult.success();
    }

    @ApiOperation(value = "美甲师录入订单资料")
    @PostMapping("/technician/ordermanage/recordOrderData")
    public JsonResult<Void> recordOrderData(@RequestHeader String token,
                                            @RequestParam("orderId") Integer orderId,
                                            @RequestParam("remark") String remark,
                                            @RequestParam("userFeedback") String userFeedback,
                                            @RequestParam("picturePath") String picturePath,
                                            @RequestParam String afterSaleIsBuyCard,
                                            @RequestParam String afterSaleBuyCardReason) {
        orderManageService.recordOrderData(orderId, remark, userFeedback, picturePath, afterSaleIsBuyCard, afterSaleBuyCardReason);
        return JsonResult.success();
    }

    @ApiOperation(value = "查询会员消费记录")
    @GetMapping("/technician/ordermanage/queryCustomerOrder")
    @GogirlShop
    public JsonResult<Map<String, Object>> queryCustomerOrder(@RequestHeader String token,
                                                              @RequestParam Integer customerId,
                                                              @RequestParam Integer pageNum,
                                                              @RequestParam Integer pageSize) {
        Map<String, Object> result = orderManageService.queryCustomerOrder(token, customerId, pageNum, pageSize);
        return JsonResult.success(result);
    }

    @ApiOperation(value = "根据订单id查询订单详情，原接口")
    @GetMapping("/technician/ordermanage/queryOrderForDetail")
    public JsonResult<OrderManageDTO> queryOrderForDetail(@RequestParam Integer id) {
        OrderManage orderManage = orderManageService.queryOrder(id);
        return JsonResult.success(orderManageDTOAssembler.apply(orderManage));
    }

    @ApiOperation(value = "店员查询店铺订单列表")
    @GetMapping("/technician/ordermanage/queryListOrder")
    public JsonResult<Page<OrderManageDTO>> queryListOrder(@RequestHeader String token,
                                                           @RequestParam Integer status,
                                                           @RequestParam Integer pageNum,
                                                           @RequestParam Integer pageSize,
                                                           @RequestParam String customerPhone) {

        StoreTechnician storeTechnician = gogirlTokenService.getTokenByToken_t(token).getStoreTechnician();

        ListOrderQuery qry = ListOrderQuery.builder()
                .pageSize(pageSize).pageNum(pageNum)
                //店铺id
                .departmentId(storeTechnician.getDepartmentId())
                //是否显示未核算订单
                .showUnCheck(true)
                //会员手机号
                .customerPhone(customerPhone)
                //订单状态
                .status(status)
                .build();

        Page<OrderManage> page = (Page<OrderManage>) orderManageService.queryListOrder(qry);
        Page<OrderManageDTO> orderManageDTOPage = (Page<OrderManageDTO>) page.convert(orderManageDTOAssembler);
        return JsonResult.success(orderManageDTOPage);
    }

    @ApiOperation(value = "美甲师接单前确认检查")
    @GetMapping("/technician/ordermanage/checkBeforeTakeOrders")
    public JsonResult<String> checkBeforeTakeOrders(@RequestHeader String token,
                                                    @RequestParam Integer scheduleId) {

        StoreTechnician storeTechnician = gogirlTokenService.getTokenByToken_t(token).getStoreTechnician();
        Integer technicianId = storeTechnician.getTechnicianId();
        scheduleManageService.checkBeforeTakeOrders(technicianId, scheduleId);
        return JsonResult.success();
    }

    @ApiOperation("核算订单")
    @GetMapping("/technician/ordermanage/checkOrder")
    public JsonResult<Void> checkOrder(@RequestHeader String token,
                                       @RequestParam String orderId) {
        orderManageService.checkOrder(orderId);
        return JsonResult.success();
    }

    @ApiOperation("更新订单")
    @PostMapping("/technician/ordermanage/updateOrderManage")
    public JsonResult<Void> updateOrderManage(@RequestHeader String token,
                                              @RequestBody OrderManage param) {
        orderManageService.updateOrderManage(param);
        return JsonResult.success();
    }


    @ApiOperation("查看评论")
    @GetMapping("/technician/ordermanage/getMyOrderComment")
    public JsonResult<IPage<OrderComment>> getMyOrderComment(@RequestParam Integer pageNum,
                                                             @RequestParam Integer pageSize,
                                                             @RequestHeader String token) {
        StoreTechnician storeTechnician = gogirlTokenService.getTokenByToken_t(token).getStoreTechnician();
        Page<OrderComment> page = new Page<>(pageNum, pageSize);
        IPage<OrderComment> orderCommentIPage = orderCommentMapper.queryMyCommentForPage(page, storeTechnician.getTechnicianId());
        return JsonResult.success(orderCommentIPage);
    }
}