package com.gogirl.interfaces.order.mall;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.order.mall.MallOrder;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.PageUtils;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.shared.market.ListMallOrderReqDTO;
import com.gogirl.shared.market.SubmitMallOrderCommand;
import com.gogirl.shared.member.order.command.ImmediatePurchaseCommand;
import com.gogirl.shared.order.mall.command.CancelMallOrderCommand;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.Optional;

@Api(tags = "商城订单接口", value = "商城订单接口")
@RestController
@AllArgsConstructor
public class MallOrderController {

    private final MallOrderService mallOrderService;
    private final GogirlTokenService gogirlTokenService;


    @ApiOperation("提交订单")
    @PostMapping("/customer/mallOrder/submitMallOrder")
    public JsonResult<MallOrder> submitMallOrder(@RequestHeader String token,
                                                 @RequestBody SubmitMallOrderCommand cmd) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCustomerId(currentCustomerId);
        mallOrderService.submitMallOrder(cmd);
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(cmd.getExecutionResult());
        return JsonResult.success(mallOrder);
    }

    @ApiOperation("立即购买")
    @PostMapping("/customer/mallOrder/immediatePurchase")
    public JsonResult<MallOrder> immediatePurchase(@RequestHeader String token,
                                                   @RequestBody ImmediatePurchaseCommand cmd) {

        Integer currentCustomerId = SessionUtils.getCustomerId();
        mallOrderService.immediatePurchase(cmd);
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(cmd.getExecutionResult());
        return JsonResult.success(mallOrder);
    }


    @ApiOperation("订单确认收货")
    @GetMapping("/customer/mallOrder/takeDeliveryOfMallOrder")
    public JsonResult<Integer> takeDeliveryOfMallOrder(@RequestHeader String token,
                                                       @PathVariable Long mallOrderId) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        mallOrderService.takeDeliveryOfMallOrder(currentCustomerId, mallOrderId);
        return JsonResult.success();
    }

    @ApiOperation("我的订单列表")
    @PostMapping("/customer/mallOrder/listMallOrderAggregate")
    public JsonResult<IPage<MallOrder>> listMallOrderAggregate(@RequestHeader String token,
                                                               @RequestBody ListMallOrderReqDTO reqDTO) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        Page<MallOrder> page = PageUtils.castPageReq(reqDTO, MallOrder.class);
        MallOrder mallOrder = ListMallOrderReqDTO.convertToMallOrder(reqDTO);
        mallOrder.setCustomerId(currentCustomerId);
        IPage<MallOrder> data = mallOrderService.listMallOrderAggregate(page, mallOrder);
        return JsonResult.success(data);
    }

    @ApiOperation("查看订单详情")
    @GetMapping("/customer/mallOrder/getMallOrderAggregate/{mallOrderId}")
    public JsonResult<MallOrder> getMallOrderAggregate(@RequestHeader String token,
                                                       @PathVariable Long mallOrderId) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(mallOrderId);
        return JsonResult.success(mallOrder);
    }

    @ApiOperation(("/用户取消订单"))
    @PostMapping("/customer/mallOrder/cancelMallOrder")
    public JsonResult<Void> cancelMallOrder(@RequestHeader String token, @RequestBody CancelMallOrderCommand cmd) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCurrentCustomerId(currentCustomerId);
        mallOrderService.cancelMallOrder(cmd);
        return JsonResult.success();
    }


    @ApiModelProperty("/查询运费")
    @GetMapping("/customer/mallOrder/queryFreight")
    public JsonResult<BigDecimal> queryFreight(@RequestParam BigDecimal totalPrice) {
        BigDecimal freight = totalPrice.compareTo(new BigDecimal(100)) > 0 ? new BigDecimal("0.00") : new BigDecimal("6.00");
        return JsonResult.success(freight);
    }

}
