package com.gogirl.application.market.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.gogirl.application.market.CouponService;
import com.gogirl.application.market.MarketService;
import com.gogirl.domain.market.coupon.Coupon;
import com.gogirl.domain.market.coupon.CouponCustomerRelevance;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.xcx.GogirlConfig;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.market.coupon.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.coupon.CouponMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderManageMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.xcx.GogirlConfigMapper;
import com.gogirl.shared.market.command.SetUpOuterCouponCommand;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
@Slf4j
public class CouponServiceImpl extends ServiceImpl<CouponMapper, Coupon> implements CouponService {

    private final CouponMapper couponMapper;
    private final CouponCustomerRelevanceMapper couponCustomerRelevanceMapper;
    private final OrderServeMapper orderServeMapper;
    private final OrderManageMapper orderManageMapper;
    private final GogirlConfigMapper gogirlConfigMapper;


    private final MarketService marketService;

    @Override
    public int countRelevanceNum(Integer couponId, Integer customerId) {
        return couponCustomerRelevanceMapper.countRelevanceNum(couponId, customerId);
    }

    @Override
    public int insertSelective(Coupon coupon, CouponCustomerRelevance record) {
        if (record.getState() == null) {
            record.setState(1);
        }
        if (record.getReceiveTime() == null) {
            record.setReceiveTime(new Date());
        }
        if ((record.getValidStartTime() == null || record.getValidEndTime() == null) && coupon != null) {
            if (coupon.getValidType() == 1) {
                //优惠券原有开始和结束时间
                record.setValidStartTime(coupon.getValidStartTime());
                record.setValidEndTime(coupon.getValidEndTime());
            } else if (coupon.getValidType() == 2) {
                long today = new Date().getTime();
                long day7 = today + 86400000L * coupon.getValidDate();
                //从现在开始
                record.setValidStartTime(new Date(today));
                //七天后过期
                record.setValidEndTime(new Date(day7));
            }
        }

        //修改已领取人数
        Coupon c = couponMapper.selectById(record.getCouponId());
        c.setReceiveQuantity(c.getReceiveQuantity() + 1);
        couponMapper.updateById(c);


        couponCustomerRelevanceMapper.insert(record);
        return record.getId();
    }


    @Override
    public List<CouponCustomerRelevance> selectMyNewCustomerCoupon(Integer customerId, List<Integer> couponIdList) {

        List<CouponCustomerRelevance> couponCustomerRelevanceList = couponCustomerRelevanceMapper.selectList(
                new LambdaQueryWrapper<CouponCustomerRelevance>().eq(CouponCustomerRelevance::getCustomerId, couponIdList)
                        .in(CouponCustomerRelevance::getCouponId, couponIdList)
        );
        this.joinCouponCustomerRelevanceMapper(couponCustomerRelevanceList);
        return couponCustomerRelevanceList;
    }

    @Override
    public List<CouponCustomerRelevance> getOrderCoupon(Integer currentCustomerId, Integer orderId) {
        List<OrderServe> orderServeList = orderServeMapper.selectList(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));
        OrderManage orderManage = orderManageMapper.selectById(orderId);
        return couponCustomerRelevanceMapper
                .selectList(new LambdaQueryWrapper<CouponCustomerRelevance>()
                        .eq(CouponCustomerRelevance::getCustomerId, currentCustomerId)
                        .eq(CouponCustomerRelevance::getSourceType, 0)
                        .ne(CouponCustomerRelevance::getState, 2)
                        .ne(CouponCustomerRelevance::getState, 3))
                .stream()
                //过滤所有可用在服务上的
                .filter(couponCustomerRelevance -> {
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(couponCustomerRelevance.getCouponId());
                    canUserServeIds.retainAll(orderServeList.stream().map(OrderServe::getServeId).collect(Collectors.toList()));
                    return ListUtil.isNotEmpty(canUserServeIds);
                })
                //过滤可用
                .filter(CouponCustomerRelevance::isValid)
                //过滤达到可用金额
                .filter(couponCustomerRelevance -> couponCustomerRelevance.getReachingAmount().compareTo(orderManage.getTotalPrice()) < 0)
                .collect(Collectors.toList());
    }

    @Override
    public PageInfo<CouponCustomerRelevance> selectMyCoupon(Integer customerId, Integer pageNum, Integer pageSize, Integer state, Boolean containRemoveCoupon) {

        if (pageNum != null && pageSize != null) {
            PageHelper.startPage(pageNum, pageSize);
        }
        List<CouponCustomerRelevance> couponCustomerRelevanceList = couponCustomerRelevanceMapper.selectList(new LambdaQueryWrapper<CouponCustomerRelevance>().eq(CouponCustomerRelevance::getCustomerId, customerId));
        this.joinCouponCustomerRelevanceMapper(couponCustomerRelevanceList);


        long nowTime = new Date().getTime();
        for (int i = 0; i < couponCustomerRelevanceList.size(); i++) {
            if (containRemoveCoupon != null && containRemoveCoupon == false && couponCustomerRelevanceList.get(i).getCouponId().equals(66)) {
                couponCustomerRelevanceList.remove(i);
                i--;
                continue;
            }
            CouponCustomerRelevance couponCustomerRelevance = couponCustomerRelevanceList.get(i);
            if (nowTime > couponCustomerRelevance.getValidEndTime().getTime() && couponCustomerRelevance.getState() == 1) {
                couponCustomerRelevance.setState(3);
            }
            if (state != null && state != 0 && couponCustomerRelevance != null && couponCustomerRelevance.getState() != null && !couponCustomerRelevance.getState().equals(state)) {
                couponCustomerRelevanceList.remove(i);
                i--;
            }
        }
        PageInfo<CouponCustomerRelevance> pageInfo = new PageInfo<>(couponCustomerRelevanceList);
        return pageInfo;
    }

    @Override
    public Coupon checkNewCustomer(Integer customerId) {
        //新用户配置
        GogirlConfig gogirlConfig = gogirlConfigMapper.selectByPrimaryKey(2);

        if (gogirlConfig == null
                || gogirlConfig.getValue() == null
                || gogirlConfig.getValue().isEmpty()) {
            throw new RRException("请联系管理员配置发放哪张新人优惠券");

        }
        List<Integer> couponIdList = Arrays.stream(gogirlConfig.getValue().split(",")).map(Integer::valueOf).collect(Collectors.toList());
        if (ListUtil.isEmpty(couponIdList)) {
            throw new RRException("请联系管理员配置发放哪张新人优惠券");
        }


        //判断是否有领过相关的优惠券
        List<CouponCustomerRelevance> list = couponCustomerRelevanceMapper.selectList(new LambdaQueryWrapper<CouponCustomerRelevance>()
                .eq(CouponCustomerRelevance::getCustomerId, customerId)
                .in(CouponCustomerRelevance::getCouponId, couponIdList));

        if (list.size() > 0) {
            return null;
        }
        Integer orderTimes = orderManageMapper.countOrderTimes(customerId);

        //没有订单,确认是新人,发券
        if (orderTimes == null || orderTimes == 0) {
            return this.getById(couponIdList.get(0));
        }
        return null;
    }

    @Override
    public Coupon sendNewCustomerCouponXcx(Integer customerId) {
        GogirlConfig gogirlConfig = gogirlConfigMapper.selectByPrimaryKey(2);
        if (gogirlConfig == null || gogirlConfig.getValue() == null || gogirlConfig.getValue().isEmpty()) {
            throw new RRException("请联系管理员配置发放哪张新人优惠券");
        }
        String[] couponIds = gogirlConfig.getValue().split(",");
        Integer couponId = null;

        List<Integer> coupontIdList = new ArrayList<>();
        for (int i = 0; i < couponIds.length; i++) {
            if (couponIds[i] != null && !couponIds[i].isEmpty()) {
                Integer itemId = Integer.parseInt(couponIds[0]);
                if (i == 0) {
                    couponId = itemId;
                }
                coupontIdList.add(itemId);
            }
        }
        if (coupontIdList.size() > 0) {//判断是否有领过相关的优惠券
            List<CouponCustomerRelevance> list = this.selectMyNewCustomerCoupon(customerId, coupontIdList);
            if (list.size() > 0) {
                throw new RRException("您曾经领取过新人优惠券");
            }
        }
        log.info("发放优惠券customerId:" + customerId.toString());
        log.info("发放优惠券couponId:" + couponId.toString());

        //返回消息
        return sendCoupon(couponId, customerId);
    }

    @Override
    public Coupon sendCoupon(Integer couponId, Integer customerId) {
        Coupon coupon = this.getById(couponId);
        if (coupon == null) {
            throw new RRException("找不到该优惠券");
        }

        //判断优惠券是否还有
        if (coupon.getAllQuantity() <= coupon.getReceiveQuantity()) {
            throw new RRException("优惠券已经领完");
        }
        //判断该用户是否限领,null过,0过,有但限制不过,不限制过
        int row = this.countRelevanceNum(couponId, customerId);
        if (coupon.getLimitQuantity() != null && coupon.getLimitQuantity() != 0 && coupon.getLimitQuantity() <= row) {
            throw new RRException("抱歉,该优惠券最多可领取" + coupon.getLimitQuantity() + "张,你已有" + row + "张该优惠券.");
        }

        CouponCustomerRelevance couponCustomerRelevance = new CouponCustomerRelevance();
        couponCustomerRelevance.setCustomerId(customerId);
        couponCustomerRelevance.setCouponId(couponId);
        couponCustomerRelevance.setState(1);
        couponCustomerRelevance.setReceiveTime(new Date());

        //冗字段
        couponCustomerRelevance.setCouponName(coupon.getName());
        couponCustomerRelevance.setDiscountAmount(coupon.getDiscountAmount());
        couponCustomerRelevance.setReachingAmount(coupon.getReachingAmount());
        couponCustomerRelevance.setDiscountPercent(coupon.getDiscountPercent());
        couponCustomerRelevance.setType(coupon.getType());
        couponCustomerRelevance.setSourceType(coupon.getSourceType());
        couponCustomerRelevance.setCondition1(coupon.getCondition1());
        couponCustomerRelevance.setPayAmount(coupon.getPayAmount());
        couponCustomerRelevance.setSourceType(coupon.getSourceType());
        couponCustomerRelevance.setSourceFrom("新人券");

        if (coupon.getValidType() == 1) {
            couponCustomerRelevance.setValidStartTime(coupon.getValidStartTime());//优惠券原有开始和结束时间
            couponCustomerRelevance.setValidEndTime(coupon.getValidEndTime());
        } else if (coupon.getValidType() == 2) {
            long today = new Date().getTime();
            long day7 = today + 86400000L * coupon.getValidDate();
            couponCustomerRelevance.setValidStartTime(new Date(today));//从现在开始
            couponCustomerRelevance.setValidEndTime(new Date(day7));//七天后过期
        }
        int id = this.insertSelective(coupon, couponCustomerRelevance);


        coupon.setValidStartTime(couponCustomerRelevance.getValidStartTime());
        coupon.setValidEndTime(couponCustomerRelevance.getValidEndTime());
        return coupon;
    }

    @Override
    public List<Coupon> getOrderExternalCoupon(Integer orderId) {
        OrderManage orderManage = orderManageMapper.selectById(orderId);
        List<OrderServe> orderServeList = orderServeMapper.selectList(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));
        return couponMapper
                .selectList(new LambdaQueryWrapper<Coupon>().eq(Coupon::getSourceType, 1))
                .stream()
                //过滤所有可用在服务上的
                .filter(coupon -> {
                    List<Integer> canUserServeIds = couponCustomerRelevanceMapper.queryCouponCustomerServeRelevance(coupon.getId());
                    canUserServeIds.retainAll(orderServeList.stream().map(OrderServe::getServeId).collect(Collectors.toList()));
                    return ListUtil.isNotEmpty(canUserServeIds);
                })
                //过滤达到可用金额
                .filter(coupon -> coupon.getReachingAmount().compareTo(orderManage.getTotalPrice()) < 0)
                .collect(Collectors.toList());
    }

    @Override
    public void techSubmitCoupons(Integer orderId, List<Integer> couponIds) {
        //查询订单
        OrderManage orderManage = orderManageMapper.selectById(orderId);

        SetUpOuterCouponCommand cmd = SetUpOuterCouponCommand.builder()
                .customerId(orderManage.getOrderUser())
                .orderId(orderId)
                .outerCouponIds(couponIds)
                .build();
        marketService.setUpOuterCoupon(cmd);
        marketService.setterOrder(orderId);
    }


    private void joinCouponCustomerRelevanceMapper(List<CouponCustomerRelevance> couponCustomerRelevanceList) {
        List<Integer> ids = couponCustomerRelevanceList.stream().map(CouponCustomerRelevance::getCouponId).collect(Collectors.toList());
        List<Coupon> couponList = couponMapper.selectList(
                new LambdaQueryWrapper<Coupon>().in(Coupon::getId, ids)
        );
        Map<Integer, List<Coupon>> couponMap = couponList.stream().collect(Collectors.groupingBy(Coupon::getId));
        couponCustomerRelevanceList.forEach(couponCustomerRelevance -> {
            if (ListUtil.isNotEmpty(couponMap.get(couponCustomerRelevance.getCouponId()))) {
                couponCustomerRelevance.setCoupon(couponMap.get(couponCustomerRelevance.getCouponId()).stream().findAny().orElse(null));
            }
        });
    }
}
