package com.gogirl.interfaces.store;

import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.gogirl.application.store.store.ShopManageService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.mapper.store.StoreManageMapper;
import com.gogirl.infrastructure.util.MapDistance;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@RestController
@Api(tags = {"version2.0: 店铺"}, value = "version2.0: 店铺")
@AllArgsConstructor
@Slf4j
public class ShopManageController {

    /**
     * domain service
     */
    private final ShopManageService shopManageService;
    private final GogirlTokenService gogirlTokenService;
    private final StoreManageMapper storeManageMapper;

    @ApiOperation(value = "查询店铺列表，longitude,latitude和页码都是可选参数")
    @GetMapping("/customer/shop/queryShopForPage")
    public JsonResult<PageInfo<StoreManage>> queryShopForPage(@RequestParam(required = false) String longitude,
                                                              @RequestParam(required = false) String latitude,
                                                              @RequestParam(required = false) Integer pageNum,
                                                              @RequestParam(required = false) Integer pageSize) {
        if (pageNum != null && pageSize != null) {
            PageHelper.startPage(pageNum, pageSize);
        }
        StoreManage pageReq = new StoreManage();
        pageReq.setLongitude(longitude);
        pageReq.setLatitude(latitude);
        List<StoreManage> lists = shopManageService.listShopManageForPage(pageReq);

        if (StringUtils.isNotBlank(longitude) && StringUtils.isNotBlank(latitude) && !longitude.equals("undefined") && !latitude.equals("undefined")) {

            for (StoreManage storeManage : lists) {
                Double distance = MapDistance.getDistance(longitude, latitude, storeManage.getLongitude(), storeManage.getLatitude());
                storeManage.setDistance(distance);
            }
            lists.sort(Comparator.comparing(StoreManage::getDistance));
        }
        return JsonResult.success(new PageInfo<>(lists));
    }

    @ApiOperation(value = "id查店铺详情")
    @GetMapping("/customer/shop/queryShopForDetail")
    public JsonResult<StoreManage> queryShopForDetail(@RequestParam(required = false) String longitude,
                                                      @RequestParam(required = false) String latitude,
                                                      @RequestParam(required = false) Integer id) {


        StoreManage storeManage = storeManageMapper.selectById(id);
        if (latitude != null && longitude != null && !latitude.equals("undefined") && !longitude.equals("undefined")) {
            Double arg0 = MapDistance.getDistance(longitude, latitude, storeManage.getLongitude(), storeManage.getLatitude());
            storeManage.setDistance(arg0);
        }
        return JsonResult.success(storeManage);
    }

    @ApiOperation(value = "查询店照，不传departmentId则查全部店铺的店照")
    @GetMapping("/customer/shop/queryShopEnvironmentPhotos")
    public JsonResult<List<String>> queryShopEnvironmentPhotos(@RequestParam(required = false) Integer departmentId,
                                                               @RequestParam(required = false) String token,
                                                               @RequestParam(required = false) String formId) {

        StoreManage storeManage = new StoreManage();
        storeManage.setId(departmentId);
        List<StoreManage> lists = shopManageService.listShopManageForPage(storeManage);

        List<String> resultList = lists.stream()
                .filter(Objects::nonNull)
                .map(StoreManage::getShopEnvironmentPhotos)
                .filter(Objects::nonNull)
                .map(phones -> Lists.newArrayList(phones.split(",")))
                .flatMap(List::stream)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        return JsonResult.success(resultList);
    }

}
