package com.gogirl.interfaces.order.mall;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.order.mall.IMallOrderTechnicianService;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.domain.order.mall.MallOrder;
import com.gogirl.domain.order.mall.MallOrderTechnician;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.infrastructure.util.lock.MallOrderIdLock;
import com.gogirl.shared.market.ListMallOrderRequest;
import com.gogirl.shared.market.SubmitMallOrderCommand;
import com.gogirl.shared.market.UpdateMallOrderTechnicianRequest;
import com.gogirl.shared.member.ImmediatePurchaseCommand;
import com.gogirl.shared.order.CancelMallOrderCommand;
import com.gogirl.shared.order.mall.QueryMallOrderAchievementRequest;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;

@Api(tags = "商城订单接口", value = "商城订单接口")
@RestController
@AllArgsConstructor
@Slf4j
public class MallOrderController {

    private final MallOrderService mallOrderService;
    private final IMallOrderTechnicianService mallOrderTechnicianService;


    @ApiOperation("提交订单")
    @PostMapping("/customer/mallOrder/submitMallOrder")
    public JsonResult<MallOrder> submitMallOrder(@RequestBody SubmitMallOrderCommand cmd) {

        log.info("token:{}", SessionUtils.getCustomerToken().toString());
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCustomerId(currentCustomerId);
        mallOrderService.submitMallOrder(cmd);
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(cmd.getExecutionResult());
        return JsonResult.success(mallOrder);
    }

    @ApiOperation("立即购买")
    @PostMapping("/customer/mallOrder/immediatePurchase")
    public JsonResult<MallOrder> immediatePurchase(@RequestBody ImmediatePurchaseCommand cmd) {
        log.info("token:{}", SessionUtils.getCustomerToken().toString());
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCustomerId(currentCustomerId);
        mallOrderService.immediatePurchase(cmd);
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(cmd.getExecutionResult());
        return JsonResult.success(mallOrder);
    }


    @ApiOperation("订单确认收货")
    @GetMapping("/customer/mallOrder/takeDeliveryOfMallOrder")
    public JsonResult<Integer> takeDeliveryOfMallOrder(@RequestHeader String token,
                                                       @RequestParam Long mallOrderId) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        mallOrderService.takeDeliveryOfMallOrder(currentCustomerId, mallOrderId);
        return JsonResult.success();
    }

    @ApiOperation("用户端查看我的订单列表")
    @PostMapping("/customer/mallOrder/listMallOrderAggregate")
    public JsonResult<IPage<MallOrder>> listMallOrderAggregate(@RequestBody ListMallOrderRequest request) {

        IPage<MallOrder> data = mallOrderService.listMallOrderAggregate(request);
        return JsonResult.success(data);
    }

    @ApiOperation("用户端查看查看订单详情")
    @GetMapping("/customer/mallOrder/getMallOrderAggregate/{mallOrderId}")
    public JsonResult<MallOrder> getMallOrderAggregate(@PathVariable Long mallOrderId) {
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(mallOrderId);
        return JsonResult.success(mallOrder);
    }

    @ApiOperation(("/用户取消订单"))
    @PostMapping("/customer/mallOrder/cancelMallOrder")
    public JsonResult<Void> cancelMallOrder(@RequestBody CancelMallOrderCommand cmd) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCurrentCustomerId(currentCustomerId);
        mallOrderService.cancelMallOrder(cmd);
        return JsonResult.success();
    }


    @ApiModelProperty("/查询运费")
    @GetMapping("/customer/mallOrder/queryFreight")
    public JsonResult<BigDecimal> queryFreight(@RequestParam BigDecimal totalPrice) {
        BigDecimal freight = totalPrice.compareTo(new BigDecimal(100)) > 0 ? new BigDecimal("0.00") : new BigDecimal("6.00");
        return JsonResult.success(freight);
    }


    @ApiOperation("美甲师端口查看订单列表")
    @PostMapping("/technician/mallOrder/listMallOrderAggregate")
    public JsonResult<IPage<MallOrder>> techListMallOrderAggregate(@RequestBody ListMallOrderRequest request) {
        IPage<MallOrder> data = mallOrderService.listMallOrderAggregate(request);
        return JsonResult.success(data);
    }

    @ApiOperation("美甲师端口查看查看订单详情")
    @GetMapping("/technician/mallOrder/getMallOrderAggregate/{mallOrderId}")
    public JsonResult<MallOrder> techGetMallOrderAggregate(@PathVariable Long mallOrderId) {
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(mallOrderId);
        return JsonResult.success(mallOrder);
    }

    @ApiOperation("美甲师确认收货")
    @GetMapping("/technician/mallOrder/confirmOrderShip")
    public JsonResult<Void> confirmOrderShip(@RequestParam Long mallOrderId) {
        MallOrderIdLock lock = MallOrderIdLock.getInstance();
        try {
            lock.lock(mallOrderId);
            mallOrderService.confirmOrderShip(mallOrderId);
            return JsonResult.success();
        } finally {
            lock.unlock(mallOrderId);
        }
    }

    @ApiOperation("更新商城推荐人")
    @PostMapping("/technician/mallOrder/updateMallOrderTechnician")
    public JsonResult<Void> updateMallOrderTechnician(@RequestBody UpdateMallOrderTechnicianRequest request) {
        mallOrderService.updateMallOrderTechnician(request);
        return JsonResult.success();
    }


    @ApiOperation("查询美甲师业绩列表")
    @PostMapping("/technician/mallOrder/queryMallOrderAchievement")
    public JsonResult<Page<MallOrderTechnician>> queryMallOrderAchievement(@RequestBody QueryMallOrderAchievementRequest request) {
        Page<MallOrderTechnician> page = mallOrderTechnicianService.queryMallOrderAchievement(request);
        return JsonResult.success(page);
    }

    @ApiOperation("查询美甲师总业绩")
    @GetMapping("/technician/mallOrder/queryTotalMallOrderAchievement")
    public JsonResult<BigDecimal> queryTotalMallOrderAchievement(@RequestParam(required = false) Integer technicianId,
                                                                 @RequestParam(required = false) Integer departmentId,
                                                                 @RequestParam String date) {
        BigDecimal totalMallOrderAchievement = mallOrderTechnicianService.queryTotalMallOrderAchievement(technicianId, departmentId, date);
        return JsonResult.success(totalMallOrderAchievement);
    }

}
