package com.gogirl.application.user.customer.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.domain.store.store.StoreUser;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.mapper.order.serve.OrderManageMapper;
import com.gogirl.infrastructure.mapper.store.store.UserManageMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.shared.user.command.ConsumerCommand;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
@AllArgsConstructor
@Slf4j
@Transactional
public class CustomerBalanceServiceImpl extends ServiceImpl<CustomerBalanceMapper, CustomerBalance> implements CustomerBalanceService {

    private final CustomerBalanceMapper customerBalanceMapper;
    private final CustomerBalanceRecordMapper balanceRecordDao;
    private final OrderManageMapper orderManageMapper;
    private final CustomerBalanceRecordMapper customerBalanceRecordMapper;

    @Override
    public CustomerBalance getCustomerBalance(int customerId) {
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>()
                .eq(CustomerBalance::getCustomerId, customerId));

        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(customerId);
            customerBalanceMapper.insert(customerBalance);
        }
        return customerBalance;
    }


    @Override
    public void consumer(ConsumerCommand cmd) {
        CustomerBalance customerBalance = customerBalanceMapper.selectOne(new LambdaQueryWrapper<CustomerBalance>()
                .eq(CustomerBalance::getCustomerId, cmd.getCustomerId()));

        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(cmd.getCustomerId());
            customerBalanceMapper.insert(customerBalance);
        }

        if (customerBalance.getBalance() < cmd.getAmount()) {
            throw new RRException(2002, "余额不足");
        }

        //余额支付
        CustomerBalanceRecord record = CustomerBalanceRecord
                .builder()
                .bestowAmount(0)
                .oldBalance(customerBalance.getBalance())
                .currentBalance(customerBalance.getBalance() - cmd.getAmount())
                .orderAmount(cmd.getAmount())
                .customerId(cmd.getCustomerId())
                .departmentId(cmd.getDepartmentId())
                .orderId(String.valueOf(cmd.getOrderId()))
                .discount(1.00)
                .orderState(cmd.getOrderStatus())
                //来源 会员卡收款
                .source(2)
                //类型 预约订单扣款
                .type(-1)
                .time(new Date())
                .build();

        customerBalanceRecordMapper.insert(record);
        customerBalance.setBalance(customerBalance.getBalance() - cmd.getAmount());
        customerBalance.setTotalExpenditure(customerBalance.getTotalExpenditure() + cmd.getAmount());
        customerBalanceMapper.updateById(customerBalance);
    }

    private final UserManageMapper userManageMapper;

    @Override
    public PageInfo<CustomerBalanceRecord> getBalanceRecordCard(Integer customerId, int pageNum, int pageSize) {
        CustomerBalanceRecord param = new CustomerBalanceRecord();
        param.setCustomerId(customerId);
        PageHelper.startPage(pageNum, pageSize);

        List<CustomerBalanceRecord> list = balanceRecordDao.getBalanceRecordCard(customerId);
        list.stream()
                .filter(balanceRecord -> balanceRecord.getType().equals(-1))
                .forEach(balanceRecord -> {
                    balanceRecord.setOrderManage(orderManageMapper.selectById(Integer.valueOf(balanceRecord.getOrderId())));
                    if (balanceRecord.getOrderAmount() != null && balanceRecord.getOrderAmount() != 0 && balanceRecord.getOrderManage() != null) {
                        balanceRecord.getOrderManage().setTotalPrice(new BigDecimal(balanceRecord.getOrderAmount()).divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP));
                    }
                });

        //设置推荐人名字
        List<StoreUser> allStoreUser = userManageMapper.selectList(new QueryWrapper<>());
        Map<String, String> idMapName = new HashMap<>();

        for (StoreUser storeUser : allStoreUser) {
            idMapName.put(String.valueOf(storeUser.getId()), storeUser.getName());
        }
        for (CustomerBalanceRecord customerBalanceRecord : list) {
            String refereeId = customerBalanceRecord.getRefereeId();
            if (refereeId != null && !refereeId.isEmpty()) {
                String[] refereeIdArr = refereeId.split(",");
                StringBuilder refereeName = new StringBuilder();
                for (String s : refereeIdArr) {
                    if (s != null && !s.isEmpty()) {
                        refereeName.append(",").append(idMapName.getOrDefault(s, s));
                    }
                }
                customerBalanceRecord.setRefereeId(refereeName.substring(1));
            }
        }


        PageInfo<CustomerBalanceRecord> pageInfo = new PageInfo<>(list);
        return pageInfo;
    }
}
