package com.gogirl.infrastructure.schedule;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.application.order.serve.ScheduleManageService;
import com.gogirl.application.product.serve.FeaturesMappingService;
import com.gogirl.application.product.serve.ProduceSalesService;
import com.gogirl.application.store.store.StoreManageService;
import com.gogirl.application.store.store.StoreTechnicianService;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.domain.order.mall.MallOrder;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.ScheduleManage;
import com.gogirl.domain.order.serve.ScheduleServe;
import com.gogirl.domain.product.mall.MallCategory;
import com.gogirl.domain.product.serve.BaseFeatures;
import com.gogirl.domain.product.serve.FeaturesMapping;
import com.gogirl.domain.product.serve.ProduceSales;
import com.gogirl.domain.store.career.Career;
import com.gogirl.domain.store.store.StoreData;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.domain.xcx.WeekConfig;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.config.property.GogirlProperties;
import com.gogirl.infrastructure.mapper.market.coupon.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.order.mall.MallOrderMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderManageMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.order.serve.ScheduleManageMapper;
import com.gogirl.infrastructure.mapper.product.mall.MallCategoryMapper;
import com.gogirl.infrastructure.mapper.product.mall.MallProductMapper;
import com.gogirl.infrastructure.mapper.product.serve.BaseFeaturesMapper;
import com.gogirl.infrastructure.mapper.product.serve.FeaturesMappingMapper;
import com.gogirl.infrastructure.mapper.product.serve.ProduceSalesMapper;
import com.gogirl.infrastructure.mapper.store.career.CareerMapper;
import com.gogirl.infrastructure.mapper.store.store.StoreDataMapper;
import com.gogirl.infrastructure.mapper.store.store.StoreManageMapper;
import com.gogirl.infrastructure.mapper.store.store.StoreTechnicianMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.mapper.xcx.WeekConfigMapper;
import com.gogirl.infrastructure.service.push.PushMsgService;
import com.gogirl.infrastructure.service.push.TechnicianPushService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@EnableScheduling
@AllArgsConstructor
@Slf4j
@Component
public class Schedule {

    /**
     * domain service
     */
    private final StoreTechnicianService storeTechnicianService;
    private final StoreManageService storeManageService;
    private final ScheduleManageService scheduleManageService;
    private final PushMsgService pushMsgService;
    private final CustomerService customerService;
    private final MallOrderService mallOrderService;
    private final ProduceSalesService produceSalesService;
    private final FeaturesMappingService featuresMappingService;

    /**
     * repository
     */
    private final CouponCustomerRelevanceMapper couponCustomerRelevanceDao;
    private final CustomerBalanceRecordMapper balanceRecordDao;
    private final ScheduleManageMapper scheduleManageMapper;
    private final MallOrderMapper mallOrderMapper;
    private final ProduceSalesMapper produceSalesMapper;
    private final BaseFeaturesMapper baseFeaturesMapper;
    private final FeaturesMappingMapper featuresMappingMapper;
    private final OrderServeMapper orderServeMapper;
    private final WeekConfigMapper weekConfigMapper;
    private final StoreTechnicianMapper storeTechnicianMapper;
    private final CareerMapper careerMapper;
    private final StoreManageMapper storeManageMapper;
    private final StoreDataMapper storeDataMapper;
    private OrderManageMapper orderManageMapper;
    private final MallProductMapper mallProductMapper;
    private final MallCategoryMapper mallCategoryMapper;

    /**
     * rpc
     */
    private final RestTemplate restTemplate;

    /**
     * config
     */
    private final GogirlProperties gogirlProperties;
    private final TechnicianPushService technicianPushService;

    /**
     * 每天3点判断优惠券是否过期。
     */
    @Scheduled(cron = "0 0 3 * * ?")
    public void setCouponExpireJob() {
        log.debug("每天定时任务,判断优惠券是否过期开始");
        couponCustomerRelevanceDao.setCouponExpire(new Date());
        log.debug("每天定时任务,判断优惠券是否过期结束");
    }

    /**
     * 每天23：50设置推荐人:当天服务的技师为推荐人。
     */
    @Scheduled(cron = "0 50 23 * * *")
    public void setChargeReferees() {
        log.debug("每天晚上11:50定时设置当天服务的技师为推荐人** 任务开始");

        SimpleDateFormat yyyyMMdd = new SimpleDateFormat("yyyy-MM-dd");

        String day = yyyyMMdd.format(new Date());

        log.debug("设置推荐人:当天服务的技师为推荐人");
        Map<String, String> mapTechnicianManage = new HashMap<>();
        List<StoreTechnician> listTechnicianManage = storeTechnicianService.listTechnicianForPage(new StoreTechnician());

        //查出所有的技师,找到技师id和user_id
        for (StoreTechnician item : listTechnicianManage) {
            mapTechnicianManage.put(String.valueOf(item.getTechnicianId()), String.valueOf(item.getUserId()));
        }

        //找到当天的订单的推荐人
        List<CustomerBalanceRecord> list = balanceRecordDao.getXcxChargeRecord(day);

        //遍历设置推荐人
        for (CustomerBalanceRecord customerBalanceRecord : list) {

            //推荐人列表
            Map<String, String> map = new HashMap<>();
            //推荐人id
            StringBuilder referee_ids = new StringBuilder();
            //充值记录
            CustomerBalanceRecord cbrt = balanceRecordDao.selectById(customerBalanceRecord.getId());
            //每次都修改推荐人,且每晚都会设置推荐人
            if (customerBalanceRecord.getRefereeId() != null) {
                String[] arr = customerBalanceRecord.getRefereeId().split(",");
                //
                for (String s : arr) {
                    if (!map.containsKey(s)) {
                        map.put(s, s);
                        referee_ids.append(",").append(mapTechnicianManage.get(s));
                    }
                }
                //去掉首尾,
                if (referee_ids.toString().startsWith(",")) {
                    referee_ids = new StringBuilder(referee_ids.substring(1));
                }
                customerBalanceRecord.setRefereeId(referee_ids.toString());
            }
            //如果店铺id不为空
            if (cbrt != null && cbrt.getDepartmentId() != null) {
                customerBalanceRecord.setDepartmentId(cbrt.getDepartmentId());
            }

            if (customerBalanceRecord.getId() != null
                    && (customerBalanceRecord.getRefereeId() != null || customerBalanceRecord.getDepartmentId() != null)) {
                balanceRecordDao.updateById(customerBalanceRecord);
            }
        }
        log.debug("每天晚上11:50定时设置当天服务的技师为推荐人** 任务结束");
    }

    /**
     * 每15分钟提前一小时通知客户到店服务
     */
    @Scheduled(cron = "0 0/15 * * * *")
    public void notifyCustomerToShopService() {
        log.debug("提前一个小时通知客户到店服务** 任务开始");

        List<ScheduleManage> scheduleManageList = scheduleManageMapper.selectArriveTimeBeforeOneHour();

        scheduleManageList.forEach(scheduleManage -> {

            //如果没有发送过
            if (scheduleManage.getIsSend() == null || !scheduleManage.getIsSend().equals(1)) {
                Customer customer = customerService.getById(scheduleManage.getScheduledUser());
                StoreManage storeManage = storeManageService.getById(scheduleManage.getDepartmentId());

                /*用户超时提醒*/
                List<ScheduleServe> scheduleServeList = scheduleManageMapper.selectByScheduleId(scheduleManage.getId());
                pushMsgService.sendScheduleMsg(customer.getOpenid1(), scheduleServeList.get(0).getServeName(), scheduleManage.getArriveTime(), storeManage.getName(), storeManage.getAddress());

                /*美甲师超时预约提醒*/

                scheduleManage.setIsSend(1);
                scheduleManageMapper.updateById(scheduleManage);

            }
        });

        log.debug("提前一个小时通知客户到店服务** 任务结束");
    }

    /**
     * 每两分钟定时检查有无过期的预约
     */
    @Scheduled(cron = "0 0/2 * * * *")
    public void cancelPassSchedule() {

        log.debug("定时检查有无过期的预约** 任务开始");
        Date passDate = new Date(new Date().getTime() - new Long("900000"));

        List<ScheduleManage> scheduleManageList = scheduleManageMapper.selectList(new LambdaQueryWrapper<ScheduleManage>()
                .lt(ScheduleManage::getArriveTime, passDate)
                .eq(ScheduleManage::getStatus, 1));

        //批量更新为 失约预约
        if (ListUtil.isNotEmpty(scheduleManageList)) {

            scheduleManageList.forEach(scheduleManage -> {

                List<ScheduleServe> scheduleServeList = scheduleManageMapper.selectByScheduleId(scheduleManage.getId());

                if (!scheduleServeList.stream().map(ScheduleServe::getStatus)
                        .map(status -> status.equals(1))
                        .collect(Collectors.toList())
                        .contains(false)) {

                    scheduleManage.setStatus(3);
                    scheduleManage.setLastUpdateTime(new Date());
                    Customer customer = customerService.getById(scheduleManage.getScheduledUser());
                    StoreManage storeManage = storeManageService.getById(scheduleManage.getDepartmentId());
                    if (customer != null) {
                        try {
                            pushMsgService.sendOverScheduleMsg(customer.getOpenid1(), scheduleServeList.get(0).getServeName(), scheduleManage.getArriveTime(), storeManage.getName(), storeManage.getAddress());
                            technicianPushService.ScheduledOverTimeMsg(scheduleManage.getId());
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
            scheduleManageService.updateBatchById(scheduleManageList);
        }

        log.debug("定时检查有无过期的预约** 任务结束");
    }

    /**
     * 每两分钟定时检查有无过期的商城订单
     */
    @Scheduled(cron = "0 0/2 * * * *")
    public void cancelPassMallOrder() {

        log.debug("定时检查有无过期的商城订单** 任务开始");
        Date passDate = new Date(new Date().getTime() - new Long("1800000"));

        List<MallOrder> mallOrderList = mallOrderMapper.selectList(new LambdaQueryWrapper<MallOrder>()
                .lt(MallOrder::getCreateTime, passDate)
                //状态待付款
                .eq(MallOrder::getStatus, 1));

        //批量更新为 失约预约
        if (ListUtil.isNotEmpty(mallOrderList)) {
            mallOrderList.forEach(mallOrder -> {
                //交易关闭
                mallOrder.setStatus(5);
                //超时关闭
                mallOrder.setCloseReason(1);
            });
        }

        if (ListUtil.isNotEmpty(mallOrderList)) {
            mallOrderService.updateBatchById(mallOrderList);
        }
        log.debug("定时检查有无过期的商城订单** 任务结束");
    }

    /**
     * 每天3点同步商品销量
     */
    @Scheduled(cron = "0 0 3 * * ?")
    public void syncProduceSales() {
        log.debug("同步商品销量** 任务开始");
        BaseFeatures baseFeatures = baseFeaturesMapper.selectOne(new LambdaQueryWrapper<BaseFeatures>().eq(BaseFeatures::getName, "热销款"));
        if (baseFeatures != null) {
            List<ProduceSales> produceSalesList = produceSalesMapper.countAll();

            featuresMappingMapper.delete(new LambdaQueryWrapper<FeaturesMapping>()
                    .eq(FeaturesMapping::getFeaturesId, baseFeatures.getId())
                    .eq(FeaturesMapping::getDataType, 2));

            featuresMappingService.saveBatch(produceSalesList.stream()
                    .map(produceSales -> {
                        FeaturesMapping featuresMapping = new FeaturesMapping();
                        featuresMapping.setDataType(2);
                        featuresMapping.setFeaturesId(baseFeatures.getId());
                        featuresMapping.setDataId(produceSales.getProduceId());
                        return featuresMapping;
                    })
                    .collect(Collectors.toList()));

            produceSalesService.remove(new LambdaQueryWrapper<>());
            produceSalesService.saveBatch(produceSalesList);

        }
        log.debug("同步商品销量** 任务结束");
    }

    /**
     * 每分钟同步订单实际时间
     */
    @Scheduled(cron = "0 0/2 * * * *")
    @Deprecated
    public void syncOrderTimes() {
        orderServeMapper.syncOrderServeTimes();
    }

    /**
     * 每天0点定时同步闲时预约周信息
     */
    @Scheduled(cron = "0 0 0 * * *")
    public void weekTest() {
        log.debug("同步闲时折扣周");
        weekConfigMapper.delete(new LambdaQueryWrapper<>());

        LocalDate localDate = LocalDate.now();
        for (int i = 0; i < 7; i++) {
            WeekConfig weekConfig = new WeekConfig();
            weekConfig.setWeek(localDate.getDayOfWeek().getValue());
            weekConfig.setWeekStr(localDate.getDayOfWeek().toString());
            weekConfig.setWeekDate(localDate);
            weekConfigMapper.insert(weekConfig);
            localDate = localDate.plusDays(1);
        }
        log.debug("同步闲时折扣周 结束");
    }

    /**
     * 每天0点定时同步美甲师成长历程
     */
    @Scheduled(cron = "0 0 0 * * *")
    public void syncTechnicianCareer() {

        log.info("定时同步美甲师成长历程** 任务开始");
        List<StoreTechnician> storeTechnicianList = storeTechnicianMapper.selectList(new LambdaQueryWrapper<StoreTechnician>()
                .eq(StoreTechnician::getStatus, 1)
                .eq(StoreTechnician::getIsAdmin, false));

        storeTechnicianList.forEach(storeTechnician -> {

//            ParameterizedTypeReference<AdminResult<AdminPage<AdminTechnicianScore>>> responseBodyType = new ParameterizedTypeReference<AdminResult<AdminPage<AdminTechnicianScore>>>() {
//            };
//            GetTechnicianPageRequest allRequest = GetTechnicianPageRequest.builder()
//                    .page(1)
//                    .rows(30)
//                    .sidx("grade")
//                    .sord("asc")
//                    //在职
//                    .status(1)
//                    .technician_id(storeTechnician.getId())
//                    .build();
//            HttpHeaders headers = new HttpHeaders();
//            headers.setContentType(MediaType.APPLICATION_JSON);
//            HttpEntity<GetTechnicianPageRequest> allHttpEntity = new HttpEntity<>(allRequest, headers);

//            AdminTechnicianScore allAdminResult = restTemplate.exchange(gogirlProperties.getAdminBackendUrl(), HttpMethod.POST, allHttpEntity, responseBodyType)
//                    .getBody()
//                    .getData()
//                    .getItems()
//                    .stream()
//                    .findAny()
//                    .orElse(new AdminTechnicianScore());

//            GetTechnicianPageRequest monthRequest = GetTechnicianPageRequest.builder()
//                    .page(1)
//                    .rows(30)
//                    .sidx("grade")
//                    .sord("asc")
//                    //在职
//                    .status(1)
//                    .date_utc(Lists.newArrayList(LocalDateTime.of(LocalDate.of(2020, 4, 1), LocalTime.of(0, 0)).toString(), LocalDateTime.now().toString()))
//                    .technician_id(storeTechnician.getId())
//                    .build();

//
//            HttpEntity<GetTechnicianPageRequest> monthHttpEntity = new HttpEntity<>(monthRequest, headers);
//            log.debug(JsonUtilByJackson.writeValueAsString(monthRequest));
//            AdminTechnicianScore monthAdminResult = restTemplate.exchange(gogirlProperties.getAdminBackendUrl(), HttpMethod.POST, monthHttpEntity, responseBodyType)
//                    .getBody()
//                    .getData()
//                    .getItems()
//                    .stream()
//                    .findAny()
//                    .orElse(new AdminTechnicianScore());


            Career career = careerMapper.selectOne(new LambdaQueryWrapper<Career>().eq(Career::getTechnicianId, storeTechnician.getId()));
            if (career == null) {
                career = new Career();
                career.setTechnicianId(storeTechnician.getId());
            }

            career.setJoinDate(storeTechnician.getCreateTime());

            if (storeTechnician.getCreateTime() != null) {
                long pastMills = System.currentTimeMillis() - storeTechnician.getCreateTime().getTime();
                int day = (int) (pastMills / 1000 / 60 / 60 / 24);
                career.setOnJobTime(day);
            }

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM");
            String month = simpleDateFormat.format(new Date());


            /*当月*/

            //当月开卡数量
//            career.setMonthBalanceCardNum(monthAdminResult.getCard_count());

            //当月考勤准时
            career.setMonthAttendancePercent(null);

            //当月信息录入完整度
//            career.setMonthInfoPercent(monthAdminResult.getAvg_data_integrity());

            //当月订单总数
            Integer monthOrderNum = careerMapper.queryMonthOrderNum(storeTechnician.getId(), month);
            career.setMonthOrderNum(monthOrderNum);

            //当月客户好评率
//            career.setMonthPraisePercent(monthAdminResult.getPraise_proportion());

            //当月复购率
//            career.setMonthRepurchasePercent(monthAdminResult.getRepeat_purchase_client_roportion());

            //当月服务客户数
//            career.setMonthServeCustomerNum(monthAdminResult.getTotal_client_count());

            //当月服务次数
//            career.setMonthServeNum(monthAdminResult.getTotal_service_count());

            //当月信息录入完整度
//            BigDecimal montInfoPercent = monthAdminResult.getAvg_data_integrity();
//            if (montInfoPercent != null) {
//                career.setMonthInfoPercent(montInfoPercent.divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP));
//            }

            /*总计*/
            //总考勤准时率
            career.setTotalAttendancePercent(null);

            //总开卡数
//            career.setTotalBalanceCardNum(allAdminResult.getCard_count());

            //总信息录入完整度
//            BigDecimal totalInfoPercent = allAdminResult.getAvg_data_integrity();
//            if (totalInfoPercent != null) {
//                career.setTotalInfoPercent(totalInfoPercent.divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP));
//            }
            //总订单数
            Integer totalOrderNum = careerMapper.queryTotalOrderNum(storeTechnician.getId(), month);
            career.setTotalOrderNum(totalOrderNum);

            //总好评率
//            career.setTotalPraisePercent(allAdminResult.getPraise_proportion());

            //总考勤准时比率
            career.setTotalAttendancePercent(null);

            //总复购率
//            career.setTotalRepurchasePercent(allAdminResult.getRepeat_purchase_client_roportion());

            //服务客户总数
//            career.setTotalServeCustomerNum(allAdminResult.getTotal_client_count());

            //服务总次数
//            career.setTotalServeNum(allAdminResult.getTotal_service_count());

            /*美甲师信息*/
            //美甲师名字
            career.setTechnicianName(storeTechnician.getName());
            career.setTechnicianId(storeTechnician.getId());
            if (career.getId() != null) {
                careerMapper.updateById(career);
            } else {
                careerMapper.insert(career);
            }
        });

        log.info("定时同步美甲师成长历程** 任务结束");
    }

    /**
     * 每天0点定时同步门店数据
     */
    @Scheduled(cron = "0 0 0 * * *")
    public void syncStoreData() {

        log.info("定时同步门店数据** 任务开始");
        List<StoreManage> storeManageList = storeManageMapper.selectList(new LambdaQueryWrapper<StoreManage>()
                .eq(StoreManage::getOperatingStatus, 1));


        storeManageList.forEach(store -> {

            StoreData storeData = storeDataMapper.selectOne(new LambdaQueryWrapper<StoreData>().eq(StoreData::getDepartmentId, store.getId()));
            if (storeData == null) {
                storeData = new StoreData();
                storeData.setDepartmentId(store.getId());
                storeData.setDepartmentName(store.getName());
            }

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM");
            String month = simpleDateFormat.format(new Date());

            //当月开卡数量
            Integer MonthBalanceCardNum = storeDataMapper.queryMonthBalanceCardNum(store.getId(), month);
            storeData.setMonthBalanceCardNum(MonthBalanceCardNum);

            //考勤准时
            storeData.setMonthAttendancePercent(null);
            //信息录入完整度
            storeData.setMonthInfoPercent(null);

            //当月订单总数
            Integer monthOrderNum = storeDataMapper.queryMonthOrderNum(store.getId(), month);
            storeData.setMonthOrderNum(monthOrderNum);

            //客户好评率
            storeData.setMonthPraisePercent(null);

            //当月复购率
            //复购率
            Integer monthRepurchaseNum = storeDataMapper.queryMonthAgainOrderNum(store.getId(), month);
            if (monthOrderNum != null && monthOrderNum != 0) {
                storeData.setMonthRepurchasePercent((double) monthRepurchaseNum / monthOrderNum);
            }

            //当月服务客户总数
            Integer monthServeCustomerNum = storeDataMapper.queryMonthServeCustomerNum(store.getId(), month);
            storeData.setMonthServeCustomerNum(monthServeCustomerNum);

            //当月服务总次数
            storeData.setMonthServeNum(monthServeCustomerNum);

            //总考勤准时率
            storeData.setTotalAttendancePercent(null);


            //开卡总数
            Integer totalBalanceCardNum = storeDataMapper.queryTotalBalanceCardNum(store.getId(), month);
            storeData.setTotalBalanceCardNum(totalBalanceCardNum);

            //信息录入完整度
            storeData.setTotalInfoPercent(null);

            //订单总数
            Integer totalOrderNum = storeDataMapper.queryTotalOrderNum(store.getId(), month);
            storeData.setTotalOrderNum(totalOrderNum);

            //客户好评率
            storeData.setTotalPraisePercent(null);

            //复购率
            Integer totalRepurchaseNum = storeDataMapper.queryTotalAgainOrderNum(store.getId(), month);
            if (totalOrderNum != null && totalOrderNum != 0) {
                storeData.setTotalRepurchasePercent((double) totalRepurchaseNum / totalOrderNum);
            }

            //服务客户总数
            Integer totalServeCustomerNum = storeDataMapper.queryTotalServeCustomerNum(store.getId(), month);
            storeData.setTotalServeCustomerNum(totalServeCustomerNum);

            //服务总次数
            storeData.setTotalServeNum(totalServeCustomerNum);
//            storeData.setTechnicianId(storeTechnician.getId());

            if (storeData.getId() != null) {
                storeDataMapper.updateById(storeData);
            } else {
                storeDataMapper.insert(storeData);
            }
        });

        log.info("定时同步门店数据** 任务结束");
    }

    /**
     * 每30分钟一次评价提醒图推送
     */
    @Scheduled(cron = "0 0/30 * * * *")
    public void testPaper() {

        log.info("评价提醒推送定时任务开始");
        Date date = new Date(System.currentTimeMillis() - gogirlProperties.getTestPaperTime() * 60000L);

        List<OrderManage> orderManageList = orderManageMapper.selectList(
                new LambdaQueryWrapper<OrderManage>()
                        .ne(OrderManage::getIsSendCommentMsg, 1)
                        .eq(OrderManage::getStatus, OrderManage.STATUS_UN_COMMENT)
                        .le(OrderManage::getCreateTime, date));

        orderManageList.forEach(orderManage -> {

            try {
                Customer customer = customerService.getById(orderManage.getOrderUser());
                pushMsgService.sendCommentMsg(customer.getOpenid1(), orderManage.getId());

                orderManage.setIsSendCommentMsg(1);
                orderManageMapper.updateById(orderManage);
            } catch (Exception e) {
                log.error("推送失败 订单id：{}", orderManage.getId());
            }
        });
        log.info("评价提醒推送定时任务结束");
    }

    /**
     * 每分钟同步余额为0的会员为普通会员
     */
    @Scheduled(cron = "0 0/1 * * * *")
    public void syncBalanceLevel() {
        balanceRecordDao.syncBalanceLevel();
    }

    /**
     * 每天三点处理色胶的首页标题
     */
    @Scheduled(cron = "0 0 3 * * ?")
    public void syncMallProductIndexTitle() {

        MallCategory mallCategory = mallCategoryMapper.selectOne(new LambdaQueryWrapper<MallCategory>().eq(MallCategory::getCategoryName, "日式基础色胶"));

        mallProductMapper.syncMallProductIndexTitle(mallCategory.getId());
    }
}
