package com.gogirl.interfaces.xcx;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.github.binarywang.wxpay.bean.order.WxPayMpOrderResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.gogirl.application.market.DiscountConfigService;
import com.gogirl.application.order.member.IChargeOrderService;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.order.serve.PayService;
import com.gogirl.application.store.store.CustomerDepartmentRelevanceService;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.market.discount.DiscountConfig;
import com.gogirl.domain.store.store.StoreUser;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.domain.user.customer.CustomerBalance;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.domain.user.customer.CustomerMessage;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.JsonUtilByFsJson;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.config.GogirlProperties;
import com.gogirl.infrastructure.feign.wx.WxPayControllerFeign;
import com.gogirl.infrastructure.mapper.store.store.UserManageMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerMessageMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.shared.member.order.command.BalanceWxPayQuery;
import com.gogirl.shared.member.order.query.UnifiedOrderQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.net.UnknownHostException;
import java.util.*;


@RestController
@Api(tags = {"version3.0: 预约订单支付接口"}, value = "支付")
@AllArgsConstructor
@Slf4j
public class PayController {

    /**
     * rpc service
     */
    private final WxPayControllerFeign wxPayControllerFeign;
    /**
     * domain service
     */
    private final GogirlTokenService gogirlTokenService;
    private final DiscountConfigService discountConfigService;
    private final CustomerBalanceService balanceService;
    private final CustomerService customerService;

    /**
     * mapper
     */
    private final UserManageMapper userManageMapper;
    private final CustomerMessageMapper customerMessageMapper;

    /**
     * cqrs service
     */
    private final PayService payService;

    /**
     * config
     */
    private final GogirlProperties gogirlProperties;
    private final OrderManageService orderManageService;
    private final CustomerDepartmentRelevanceService customerDepartmentRelevanceService;
    private final CustomerBalanceMapper customerBalanceMapper;
    private final CustomerBalanceRecordMapper customerBalanceRecordMapper;

    @ApiOperation(value = "微信支付预约")
    @PostMapping("/customer/pay/unifiedOrder")
    public JsonResult<WxPayMpOrderResult> unifiedOrder(@RequestHeader String token,
                                                       @RequestParam @NotNull Integer orderId) throws WxPayException, UnknownHostException {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        log.info("**微信支付订单**,会员:{}，订单id:{} ", currentCustomerId, orderId);

        String openid = gogirlTokenService.getByToken(token).getOpenid();
        UnifiedOrderQuery qry = new UnifiedOrderQuery(currentCustomerId, orderId, openid);
        WxPayMpOrderResult result = payService.unifiedOrder(qry);
        return JsonResult.success(result);
    }

    @ApiOperation(value = "会员卡支付预约订单")
    @PostMapping("/customer/pay/consumeBalance")
    public JsonResult<Void> consumeBalance(@RequestHeader String token,
                                           @RequestParam Integer orderId) {
        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        log.info("**会员卡支付订单**,会员:{},订单id:{}", currentCustomerId, orderId);
        payService.consumeBalance(currentCustomerId, orderId);
        return JsonResult.success();
    }

    @ApiOperation(value = "申请支付预约订单，待店员确认是否完成订单。5：刷pos机7：大众点评团购")
    @PostMapping("/customer/pay/chosePayType")
    public JsonResult<Void> chosePayType(@RequestHeader String token,
                                         @RequestParam @NotNull Integer orderId,
                                         @RequestParam @NotNull Integer payType) {

        log.info("*****用户选择支付方式,支付类型{}", payType);
        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        payService.chosePayType(orderId, payType);
        return JsonResult.success();
    }

    @ApiOperation(value = "余额微信支付")
    @GetMapping("/customer/pay/balanceWxPay")
    public JsonResult<WxPayMpOrderResult> balanceWxPay(@RequestHeader String token,
                                                       @RequestParam @NotNull Integer orderId) throws WxPayException, UnknownHostException {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));
        log.info("**余额微信支付订单**,会员:{}，订单id:{} ", currentCustomerId, orderId);

        String openid = gogirlTokenService.getByToken(token).getOpenid();
        BalanceWxPayQuery qry = new BalanceWxPayQuery(currentCustomerId, orderId, openid);
        WxPayMpOrderResult result = payService.balanceWxPay(qry);
        return JsonResult.success(result);
    }

    @ApiOperation(value = "预约订单会员余额和pos支付")
    @GetMapping("/customer/pay/balancePosPay")
    public JsonResult<Void> balancePosPay(@RequestHeader String token,
                                          @RequestParam @NotNull Integer orderId) throws WxPayException, UnknownHostException {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));
        log.info("**余额pos支付订单**,会员:{}，订单id:{} ", currentCustomerId, orderId);
        payService.balancePosPay(currentCustomerId, orderId);
        return JsonResult.success();
    }

    private final IChargeOrderService chargeOrderService;

    @ApiOperation(value = "用户端，充值会员卡")
    @PostMapping("/customer/pay/chargeBalance")
    public JsonResult<?> customerChargeBalance(@RequestHeader String token,
                                               @RequestParam Integer chargeId) throws UnknownHostException, WxPayException {
        log.info("获取微信支付签名token:" + token + "；chargeId:" + chargeId);
        WxPayMpOrderResult result = chargeOrderService.customerChargeBalance(chargeId);
        return JsonResult.success(result);
    }

    @ApiOperation("朋友代付")
    @GetMapping("/customer/pay/friendPayOrder/{orderId}")
    public JsonResult<Integer> friendPayOrder(@RequestHeader String token,
                                              @PathVariable Integer orderId,
                                              @ApiParam("朋友手机号") @RequestParam String phone) {
        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        Customer customer = customerService.selectByPhone(phone);
        if (customer == null) {
            throw new RRException(500, "用户不存在");
        }
        Map<String, Object> params = new HashMap<>();
        params.put(CustomerMessage.PARAM_FRIEND_PAY_ORDER_ID, orderId);
        params.put(CustomerMessage.PARAM_FRIEND_PAY_CUSTOMER_ID, currentCustomerId);
        String paramsJson = JsonUtilByFsJson.beanToJson(params);

        List<CustomerMessage> list = customerMessageMapper.selectList(
                new LambdaQueryWrapper<CustomerMessage>().eq(CustomerMessage::getParamsJson, paramsJson)
        );

        //避免重复发送
        if (ListUtil.isEmpty(list)) {
            CustomerMessage customerMessage = CustomerMessage.builder()
                    .customerId(customer.getId())
                    .header("代付订单提醒")
                    .paramsJson(paramsJson)
                    .status(CustomerMessage.STATUS_UN_READ)
                    .type(CustomerMessage.TYPE_FRIEND_SCHEDULE_PAY)
                    .build();

            //保存消息
            customerMessageMapper.insert(customerMessage);

            return JsonResult.success();
        }
        return JsonResult.success();
    }

    @ApiOperation(value = "查看消费记录")
    @GetMapping("/customer/pay/getBalanceRecordCard")
    public JsonResult<PageInfo<CustomerBalanceRecord>> getBalanceRecordCard(Integer customerId,
                                                                            int pageNum,
                                                                            int pageSize) {
        CustomerBalanceRecord param = new CustomerBalanceRecord();
        param.setCustomerId(customerId);
        PageHelper.startPage(pageNum, pageSize);
        List<CustomerBalanceRecord> list = balanceService.getBalanceRecordCard(param);

        //设置推荐人名字
        List<StoreUser> allStoreUser = userManageMapper.selectList(new QueryWrapper<>());
        Map<String, String> idMapName = new HashMap<>();

        for (StoreUser storeUser : allStoreUser) {
            idMapName.put(String.valueOf(storeUser.getId()), storeUser.getName());
        }
        for (CustomerBalanceRecord customerBalanceRecord : list) {
            String refereeId = customerBalanceRecord.getRefereeId();
            if (refereeId != null && !refereeId.isEmpty()) {
                String[] refereeIdArr = refereeId.split(",");
                StringBuilder refereeName = new StringBuilder();
                for (String s : refereeIdArr) {
                    if (s != null && !s.isEmpty()) {
                        refereeName.append(",").append(idMapName.getOrDefault(s, s));
                    }
                }
                customerBalanceRecord.setRefereeId(refereeName.substring(1));
            }
        }


        PageInfo<CustomerBalanceRecord> pageInfo = new PageInfo<>(list);
        return JsonResult.success(pageInfo);
    }

    /**
     * 5美甲师端收款。发券
     */
    @ApiOperation(value = "美甲师端确认收款。2：会员卡收款；5：刷pos机；7：大众点评团购，8.微信扫二维码支付")
    @PostMapping("/technician/pay/receiveMoney")
    public JsonResult<Void> receiveMoney(@RequestHeader String token,
                                         @NotNull Integer orderId,
                                         @NotNull Integer payType) {

        GogirlToken gt = gogirlTokenService.getByToken(token);
        Integer technicianId = SessionUtils.getTechnicianId();
        orderManageService.confirmOrder(orderId, payType, technicianId);

        return JsonResult.success();
    }

    /**
     * 美甲师端，充值会员卡
     */
    @Transactional
    @ApiOperation(value = "美甲师端，充值会员卡")
    @PostMapping("/technician/pay/chargeCustomerCard")
    public JsonResult<Void> chargeCustomerCard(@RequestHeader String token,
                                               String name,
                                               String phone,
                                               Double amount,
                                               Integer source,
                                               String refereeId,
                                               String remark) {

        Customer customer = customerService.selectByPhone(phone);

        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);
        Integer departmentId = gogirlToken.getDepartmentId();

        log.info("{} 充值给 {}", gogirlToken.getTechnicianId(), phone);

        //新建会员
        if (customer == null) {
            customer = new Customer();
            customer.setStoreRecordRealName(name);
            customer.setPhone(phone);
            customerService.save(customer);
        }

        //更新会员的店员记录名字
        else if (name != null && !name.isEmpty()) {
            customer.setStoreRecordRealName(name);
            customerService.updateById(customer);
        }

        //店铺会员关联
        customerDepartmentRelevanceService.insertDepartmentRelevanceIfNotExist(customer.getId(), departmentId, 4, new Date());

        //折扣信息
        DiscountConfig discountConfig = discountConfigService.selectByCharge((int) (amount * 100));
        //会员卡
        CustomerBalance customerBalance = customerBalanceMapper.selectByCustomerId(customer.getId());
        //充值类型
        Integer type = customerBalance == null ? 2 : 1;

        //如果会员卡不存在就新增会员卡
        if (customerBalance == null) {
            customerBalance = CustomerBalance.getInstance(customer.getId());
            customerBalanceMapper.insert(customerBalance);
        }
        int chargeAmount = (int) (amount * 100);

        //充值订单id
        String orderId = IdWorker.getIdStr();

        //更新余额
        customerBalance.setBalance(customerBalance.getBalance() + chargeAmount);
        customerBalance.setTotalCharge(customerBalance.getTotalCharge() + chargeAmount);
        customerBalance.setUpdateTime(new Date());

        //会员等级
        customerBalance.setLevel(discountConfig.getLevel());
        customerBalanceMapper.updateById(customerBalance);


        //充值流水
        CustomerBalanceRecord customerBalanceRecord = CustomerBalanceRecord.builder()
                .currentBalance(customerBalance.getBalance())
                .customerId(customer.getId())
                //充送金额由定时任务计算
                .bestowAmount(0)
                .departmentId(gogirlToken.getDepartmentId())
                .departmentName(gogirlToken.getDepartmentName())
                .discount(1d)
                .orderAmount(chargeAmount)
                .orderId(orderId)
                //支付方式
                .source(source)
                .refereeId(refereeId)
                .remark(remark)
                //订单类型
                .type(type)
                .time(new Date())
                .build();
        customerBalanceRecordMapper.insert(customerBalanceRecord);

        return JsonResult.success();
    }
}
