package com.gogirl.application.order.mall.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gogirl.application.order.mall.MallCustomerAddressService;
import com.gogirl.application.order.mall.MallOrderDetailService;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.application.order.mall.MallShoppingCartService;
import com.gogirl.domain.order.mall.*;
import com.gogirl.domain.product.mall.MallProduct;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.SpringBeanUtils;
import com.gogirl.infrastructure.mapper.order.mall.MallOrderAddressMapper;
import com.gogirl.infrastructure.mapper.order.mall.MallOrderMapper;
import com.gogirl.infrastructure.mapper.product.mall.MallProductMapper;
import com.gogirl.shared.market.SubmitMallOrderCommand;
import com.gogirl.shared.member.ImmediatePurchaseCommand;
import com.gogirl.shared.order.CancelMallOrderCommand;
import com.gogirl.shared.order.MallOrderDTO;
import com.gogirl.shared.order.MallOrderDetailDTO;
import com.gogirl.shared.order.SettleMallOrderItemQuery;
import com.gogirl.shared.order.SettleMallOrderQuery;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
@Slf4j
@Transactional
public class MallOrderServiceImpl extends ServiceImpl<MallOrderMapper, MallOrder> implements MallOrderService {

    private final MallOrderDetailService mallOrderDetailService;

    private final MallOrderMapper mallOrderMapper;

    private final MallOrderAddressMapper mallOrderAddressMapper;


    /**
     * domain service
     */
    private final MallShoppingCartService mallShoppingCartService;
    private final MallCustomerAddressService mallCustomerAddressService;
    private final MallProductMapper mallProductMapper;

    @Override
    public MallOrder saveAggregate(MallOrder mallOrder) {
        //订单
        this.save(mallOrder);

        //订单详情
        mallOrder.getMallOrderDetailList().forEach(mallOrderDetail -> mallOrderDetail.setOrderId(mallOrder.getId()));
        mallOrderDetailService.saveBatch(mallOrder.getMallOrderDetailList());

        //订单地址（自提没有订单地址）
        if (mallOrder.getMallOrderAddress() != null) {
            mallOrder.getMallOrderAddress().setOrderId(mallOrder.getId());
            mallOrderAddressMapper.insert(mallOrder.getMallOrderAddress());
        }
        return mallOrder;
    }

    public MallOrderDTO querySettleMalOrder(SettleMallOrderQuery qry) {

        List<SettleMallOrderItemQuery> settleMallOrderItemQryList = qry.getSettleMallOrderItemQueryList();

        List<MallProduct> productList = mallProductMapper
                .selectBatchIds(settleMallOrderItemQryList.stream()
                        .map(SettleMallOrderItemQuery::getProductId)
                        .collect(Collectors.toList()));

        List<MallOrderDetailDTO> mallOrderDTODetailList = productList.stream()
                .map(product -> {

                    //商品数量
                    Integer num = settleMallOrderItemQryList.stream().filter(var -> var.getProductId().intValue() == product.getId()).findAny().orElseThrow(NullPointerException::new).getNum();
                    MallOrderDetailDTO mallOrderDetailDTO = MallOrderDetailDTO.builder()
                            .num(num)
                            .coverImages(product.getCoverImages())
                            .productName(product.getName())
                            .price(product.getPrice())
                            .productSku(product.getProductSku())
                            .productId(product.getId())
                            .specification(product.getSpecification())
                            .unit(product.getUnit())
                            .build();
                    if (product.getIsBargainPrice() == 1) {
                        mallOrderDetailDTO.setPrice(product.getBargainPrice());
                    }

                    mallOrderDetailDTO.setAmount(product.getPrice().multiply(new BigDecimal(num)));
                    //todo 优惠计算
                    mallOrderDetailDTO.setDiscountAmount(BigDecimal.ZERO);
                    mallOrderDetailDTO.setPayAmount(mallOrderDetailDTO.getAmount().subtract(mallOrderDetailDTO.getDiscountAmount()));
                    return mallOrderDetailDTO;
                })
                .collect(Collectors.toList());

        //计算订单总金额
        MallOrderDTO mallOrderDTO = new MallOrderDTO();

        BigDecimal totalAmount = BigDecimal.ZERO;
        BigDecimal totalDisCountAmount = BigDecimal.ZERO;
        BigDecimal totalPayAmount = BigDecimal.ZERO;
        for (MallOrderDetailDTO mallOrderDetail : mallOrderDTODetailList) {
            totalAmount = totalAmount.add(mallOrderDetail.getAmount());
            totalDisCountAmount = totalDisCountAmount.add(mallOrderDetail.getDiscountAmount());
            totalPayAmount = totalPayAmount.add(mallOrderDetail.getPayAmount());
        }


        mallOrderDTO.setTotalAmount(totalAmount);
        mallOrderDTO.setTotalDiscountAmount(totalDisCountAmount);
        mallOrderDTO.setTotalPayAmount(totalPayAmount);
        mallOrderDTO.setMallOrderDetailDTOList(mallOrderDTODetailList);
        return mallOrderDTO;

    }

    @Override
    public void submitMallOrder(SubmitMallOrderCommand cmd) {
        long time = System.currentTimeMillis();
        log.info("用户{}下单，时间{}", cmd.getCustomerId(), time);


        List<MallShoppingCart> mallShoppingCartList = mallShoppingCartService.list(new LambdaQueryWrapper<MallShoppingCart>().in(MallShoppingCart::getId, cmd.getMallShoppingCartIdList()));

        if (mallShoppingCartList.size() != cmd.getMallShoppingCartIdList().size()) {
            throw new RRException(500, "请重新检查购物车");
        }

        //清除购物车项
        mallShoppingCartService.remove(new LambdaQueryWrapper<MallShoppingCart>().in(MallShoppingCart::getId, cmd.getMallShoppingCartIdList()));


        //产品服务获取结算订单
        List<SettleMallOrderItemQuery> settleMallOrderItemQryList = mallShoppingCartList.stream()
                .map(mallShoppingCart -> {
                    SettleMallOrderItemQuery settleMallOrderItemQry = new SettleMallOrderItemQuery();
                    settleMallOrderItemQry.setNum(mallShoppingCart.getNum());
                    settleMallOrderItemQry.setProductId(mallShoppingCart.getProductId());
                    return settleMallOrderItemQry;
                })
                .collect(Collectors.toList());

        SettleMallOrderQuery settleMallOrderQry = new SettleMallOrderQuery(cmd.getCustomerId(), settleMallOrderItemQryList);
        log.info("获取产品服务结算订单开始 param:{}", settleMallOrderQry);
        MallOrderDTO mallOrderDTO = this.querySettleMalOrder(settleMallOrderQry);
        log.info("获取产品服务结算订单结束 mallOrderDTO:{}", mallOrderDTO);

        MallOrder mallOrder = new MallOrder();

        //订单参数
        SpringBeanUtils.copyPropertiesIgnoreNull(cmd, mallOrder);
        //结算订单参数
        SpringBeanUtils.copyPropertiesIgnoreNull(mallOrderDTO, mallOrder);

        //结算订单详情参数
        mallOrder.setMallOrderDetailList(mallOrderDTO
                .getMallOrderDetailDTOList()
                .stream()
                .map(mallOrderDetailDTO -> {
                    MallOrderDetail mallOrderDetail = new MallOrderDetail();
                    BeanUtils.copyProperties(mallOrderDetailDTO, mallOrderDetail);
                    return mallOrderDetail;
                })
                .collect(Collectors.toList()));

        mallOrder.setCustomerId(cmd.getCustomerId());

        //订单状态
        mallOrder.setStatus(MallOrder.STATUS_UN_PAY);

        //时间
        mallOrder.setCreateTime(System.currentTimeMillis());
        //下单时间
        mallOrder.setOrderTime(System.currentTimeMillis());

        //交易方式 线上交易
        mallOrder.setTransactionType(MallOrder.TRANSACTION_TYPE_ONLINE);

        if (mallOrder.getTransportMode().equals(1)) {
            //设置运费
            mallOrder.setFreight(mallOrder.getTotalAmount().compareTo(new BigDecimal(100)) > 0 ? new BigDecimal(0.00) : new BigDecimal(6.00));
        } else {
            mallOrder.setFreight(BigDecimal.ZERO);
        }

        mallOrder.setTotalPayAmount(mallOrder.getTotalAmount().add(mallOrder.getFreight()));
        //收货地址
        MallCustomerAddress mallCustomerAddress = mallCustomerAddressService.getById(cmd.getMallCustomerAddressId());
        MallOrderAddress mallOrderAddress = new MallOrderAddress();
        if (mallCustomerAddress != null) {
            BeanUtils.copyProperties(mallCustomerAddress, mallOrderAddress, "id");
            mallOrderAddress.setName(mallCustomerAddress.getCustomerName());
            mallOrderAddress.setPhone(mallCustomerAddress.getPhone());
            mallOrder.setMallOrderAddress(mallOrderAddress);
        }

        //保存订单聚合
        this.saveAggregate(mallOrder);

        //设置命令结果
        cmd.setExecutionResult(mallOrder.getId());
        log.info("用户{}下单结束，时间{},耗时{}", cmd.getCustomerId(), System.currentTimeMillis(), System.currentTimeMillis() - time);

    }

    @Override
    public void cancelMallOrder(CancelMallOrderCommand cmd) {
        MallOrder mallOrder = mallOrderMapper.selectById(cmd.getOrderId());
        if (!mallOrder.getStatus().equals(MallOrder.STATUS_UN_PAY)) {
            throw new RRException(500, "只能取消待付款的订单");
        }
        mallOrder.setStatus(MallOrder.STATUS_TRADE_CLOSE);
        mallOrder.setCloseReason(MallOrder.CLOSE_CUSTOMER_CANCEL);
        mallOrder.setRemark(mallOrder.getRemark() + "_用户取消订单");
        mallOrderMapper.updateById(mallOrder);
    }

    @Override
    public void immediatePurchase(ImmediatePurchaseCommand cmd) {

        long time = System.currentTimeMillis();
        log.info("用户{}下单，时间{}", cmd.getCustomerId(), time);

        //产品服务获取结算订单
        List<SettleMallOrderItemQuery> settleMallOrderItemQryList = Lists.newArrayList(
                new SettleMallOrderItemQuery(cmd.getProductId(), cmd.getNum())
        );

        SettleMallOrderQuery settleMallOrderQry = new SettleMallOrderQuery(cmd.getCustomerId(), settleMallOrderItemQryList);

        log.info("获取产品服务结算订单开始 param:{}", settleMallOrderQry);
        MallOrderDTO mallOrderDTO = this.querySettleMalOrder(settleMallOrderQry);
        log.info("获取产品服务结算订单结束 mallOrderDTO:{}", mallOrderDTO);

        MallOrder mallOrder = new MallOrder();

        //订单参数
        SpringBeanUtils.copyPropertiesIgnoreNull(cmd, mallOrder);
        //结算订单参数
        SpringBeanUtils.copyPropertiesIgnoreNull(mallOrderDTO, mallOrder);

        //结算订单详情参数
        mallOrder.setMallOrderDetailList(mallOrderDTO
                .getMallOrderDetailDTOList()
                .stream()
                .map(mallOrderDetailDTO -> {
                    MallOrderDetail mallOrderDetail = new MallOrderDetail();
                    BeanUtils.copyProperties(mallOrderDetailDTO, mallOrderDetail);
                    return mallOrderDetail;
                })
                .collect(Collectors.toList()));

        mallOrder.setCustomerId(cmd.getCustomerId());

        //订单状态
        mallOrder.setStatus(MallOrder.STATUS_UN_PAY);

        //时间
        mallOrder.setCreateTime(System.currentTimeMillis());
        mallOrder.setOrderTime(System.currentTimeMillis());

        //交易方式 线上交易
        mallOrder.setTransactionType(MallOrder.TRANSACTION_TYPE_ONLINE);

        //收货地址
        MallCustomerAddress mallCustomerAddress = mallCustomerAddressService.getById(cmd.getMallCustomerAddressId());
        MallOrderAddress mallOrderAddress = new MallOrderAddress();
        if (mallCustomerAddress != null) {
            BeanUtils.copyProperties(mallCustomerAddress, mallOrderAddress, "id");
            mallOrderAddress.setName(mallCustomerAddress.getCustomerName());
            mallOrderAddress.setPhone(mallCustomerAddress.getPhone());
            mallOrder.setMallOrderAddress(mallOrderAddress);
        }

        //保存订单聚合
        this.saveAggregate(mallOrder);

        //设置命令结果
        cmd.setExecutionResult(mallOrder.getId());
        log.info("用户{}下单结束，时间{},耗时{}", cmd.getCustomerId(), System.currentTimeMillis(), System.currentTimeMillis() - time);


    }

    @Override
    public IPage<MallOrder> listMallOrderAggregate(Page<MallOrder> page, MallOrder mallOrder) {

        LambdaQueryWrapper<MallOrder> queryWrapper = new LambdaQueryWrapper<>();
        if (mallOrder.getStatus() != null) {
            queryWrapper.eq(MallOrder::getStatus, mallOrder.getStatus());
        }
        queryWrapper.eq(MallOrder::getCustomerId, mallOrder.getCustomerId());
        return mallOrderMapper.listMallOrderAggregate(page, mallOrder);
    }

    @Override
    public MallOrder getMallOrderAggregate(Long mallOrderId) {
        MallOrder mallOrder = mallOrderMapper.selectById(mallOrderId);

        MallOrderAddress mallOrderAddress = mallOrderAddressMapper.selectOne(new LambdaQueryWrapper<MallOrderAddress>().eq(MallOrderAddress::getOrderId, mallOrderId));
        List<MallOrderDetail> orderDetailList = mallOrderDetailService.list(new LambdaQueryWrapper<MallOrderDetail>().eq(MallOrderDetail::getOrderId, mallOrderId));

        mallOrder.setMallOrderDetailList(orderDetailList);
        mallOrder.setMallOrderAddress(mallOrderAddress);

        return mallOrder;
    }

    @Override
    public void takeDeliveryOfMallOrder(Integer currentCustomerId, Long mallOrderId) {
        MallOrder mallOrder = this.getById(mallOrderId);
        if (!mallOrder.getStatus().equals(MallOrder.STATUS_SHIPPED)) {
            throw new RRException(500, "订单状态异常");
        }
        mallOrder.setStatus(MallOrder.STATUS_SIGNED);
    }

}
