package com.gogirl.interfaces.order.serve;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.market.discount.IVipServeService;
import com.gogirl.application.order.serve.ScheduleManageService;
import com.gogirl.application.user.customer.CustomerBalanceService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.assembler.PurchaseSkuDTOAssembler;
import com.gogirl.assembler.ScheduleManageDTOAssembler;
import com.gogirl.domain.market.discount.VipServe;
import com.gogirl.domain.order.serve.ScheduleManage;
import com.gogirl.domain.product.serve.BaseProduce;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.dto.LeisureScheduleServe;
import com.gogirl.dto.LeisureScheduleServeQuery;
import com.gogirl.dto.QueryLeisureTechnicianReq;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.product.purchase.PurchaseSkuMapper;
import com.gogirl.infrastructure.mapper.product.serve.BaseProduceMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.infrastructure.util.lock.CustomerIdLock;
import com.gogirl.infrastructure.util.lock.ScheduleServeIdLock;
import com.gogirl.infrastructure.util.lock.ScheduledLock;
import com.gogirl.shared.order.CancelScheduleCommand;
import com.gogirl.shared.order.SubmitScheduleCommand;
import com.gogirl.shared.order.UpdateScheduleCommand;
import com.gogirl.shared.order.IdleTimeDTO;
import com.gogirl.shared.order.ScheduleManageDTO;
import com.gogirl.shared.order.IdleTimeQuery;
import com.gogirl.shared.order.ScheduleManagePageQuery;
import com.gogirl.shared.product.PurchaseSkuDTO;
import com.gogirl.shared.user.VipServeQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.text.ParseException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

@Api(tags = "version3.0:预约接口", value = "version3.0:预约接口")
@RestController
@AllArgsConstructor
@Slf4j
public class ScheduleManageController {

    private final ScheduleManageService scheduleManageService;
    private final GogirlTokenService gogirlTokenService;
    private final CustomerBalanceService customerBalanceService;


    private final ScheduleManageDTOAssembler scheduleManageDTOAssembler;
    private final IVipServeService vipServeService;

    @ApiModelProperty("查询可预约时间列表")
    @PostMapping("/customer/schedule/queryIdleTime")
    public JsonResult<List<IdleTimeDTO>> queryIdleTime(@RequestBody IdleTimeQuery qry) throws ParseException, ExecutionException, InterruptedException {
        List<IdleTimeDTO> idleTimeDTOList = scheduleManageService.queryIdleTime(qry);
        return JsonResult.success(idleTimeDTOList);
    }

    @ApiOperation(value = "查询显示预约时间列表")
    @GetMapping("/customer/schedule/queryLeisureTime")

    public JsonResult<List<IdleTimeDTO>> queryLeisureTime(@RequestHeader String token,
                                                          @RequestParam Integer departmentId,
                                                          @RequestParam String scheduleDate) throws ParseException {
        List<IdleTimeDTO> idleTimeDTOList = scheduleManageService.queryLeisureTime(departmentId, scheduleDate);
        return JsonResult.success(idleTimeDTOList);
    }

    @ApiOperation("提交预约")
    @PostMapping("/customer/schedule/submitSchedule")
    public JsonResult<Void> submitSchedule(@RequestHeader String token,
                                           @RequestBody SubmitScheduleCommand cmd) throws ParseException, ExecutionException, InterruptedException {
        Integer customerId = SessionUtils.getCustomerId();
        CustomerIdLock lock = CustomerIdLock.getInsatance();
        try {
            lock.lock(customerId);
            cmd.getScheduleManageDTO().setScheduledUser(customerId);
            scheduleManageService.submitSchedule(cmd);
        } finally {
            lock.unlock(customerId);
        }
        return JsonResult.success();
    }

    @ApiOperation(value = "用户取消预约")
    @PostMapping("/customer/schedule/cancelOrder")
    public JsonResult<Void> cancelSchedule(@RequestBody CancelScheduleCommand cmd) {
        ScheduledLock scheduledLock = ScheduledLock.getInsatance();
        try {
            log.info("用户取消预约:{}", cmd);
            scheduledLock.lock(cmd.getScheduleId());
            scheduleManageService.cancelSchedule(cmd);
        } finally {
            scheduledLock.unlock(cmd.getScheduleId());
        }
        return JsonResult.success();
    }

    @ApiOperation("用户更新预约")
    @PostMapping("/customer/schedule/updateSchedule")
    public JsonResult<Void> updateSchedule(@RequestHeader String token,
                                           @RequestBody SubmitScheduleCommand cmd) throws ParseException, ExecutionException, InterruptedException {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.getScheduleManageDTO().setScheduledUser(currentCustomerId);

        ScheduledLock scheduledLock = ScheduledLock.getInsatance();
        try {
            log.info("用户更新预约:{}", cmd);
            scheduledLock.lock(cmd.getScheduleManageDTO().getId());
            scheduleManageService.updateSchedule(cmd);
        } finally {
            scheduledLock.unlock(cmd.getScheduleManageDTO().getId());
        }
        return JsonResult.success();
    }

    @ApiOperation(value = "根据id查询预约详情")
    @GetMapping("/customer/schedule/queryScheduleManage")
    public JsonResult<ScheduleManageDTO> queryScheduleManage(@RequestParam Integer id) {
        ScheduleManage scheduleManage = scheduleManageService.queryScheduleManage(id);
        return JsonResult.success(scheduleManageDTOAssembler.apply(scheduleManage));
    }

    @PostMapping("/customer/schedule/queryPageScheduleManage")
    @ApiOperation("/分页查询预约")
    public JsonResult<Page<ScheduleManageDTO>> queryPageScheduleManage(@RequestHeader String token,
                                                                       @RequestBody ScheduleManagePageQuery qry) {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));
        qry.setCustomerId(currentCustomerId);

        IPage<ScheduleManage> page = scheduleManageService.queryPageScheduleManage(qry);
        return JsonResult.success((Page<ScheduleManageDTO>) page.convert(scheduleManageDTOAssembler));
    }

    @ApiOperation("是否能够免费添加该服务")
    @PostMapping("/customer/schedule/checkVipServe")
    public JsonResult<Void> checkVipServe(@RequestHeader String token,
                                          @RequestParam Integer serveId) {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        VipServeQuery qry = new VipServeQuery();
        qry.setServeId(serveId);
        qry.setVipLevel(customerBalanceService.getCustomerBalance(currentCustomerId).getLevel());


        List<VipServe> vipServeList = vipServeService.list(new LambdaQueryWrapper<VipServe>()
                .eq(VipServe::getServeId, serveId)
                .eq(VipServe::getVipLevel, customerBalanceService.getCustomerBalance(currentCustomerId).getLevel()));

        if (ListUtil.isEmpty(vipServeList)) {
            throw new RRException(1002, "会员才能享受服务");
        }

        return JsonResult.success();
    }

    @ApiOperation(value = "店员查询可预约时间")
    @GetMapping("/technician/schedule/queryReservableTime")
    public JsonResult<List<Map<String, Object>>> queryReservableTime(String startDate,
                                                                     Integer lengthTime,
                                                                     Integer departmentId,
                                                                     Integer orderId,
                                                                     Boolean needRemoveOldServe) {
        List<Map<String, Object>> result = scheduleManageService.queryReservableTime(startDate, lengthTime, departmentId, orderId, needRemoveOldServe);
        return JsonResult.success(result);
    }

    @PostMapping("/technician/schedule/queryPageScheduleManage")
    @ApiOperation("/店员分页查询预约")
    public JsonResult<Page<ScheduleManageDTO>> technicianQueryPageScheduleManage(@RequestHeader String token,
                                                                                 @RequestBody ScheduleManagePageQuery qry) {
        GogirlToken gogirlToken = SessionUtils.getTechnicianToken();
        qry.setDepartmentId(gogirlToken.getDepartmentId());
        IPage<ScheduleManage> page = scheduleManageService.queryPageScheduleManage(qry);
        return JsonResult.success((Page<ScheduleManageDTO>) page.convert(scheduleManageDTOAssembler));
    }

    private final BaseProduceMapper baseProduceMapper;
    private final PurchaseSkuMapper purchaseSkuMapper;
    private final PurchaseSkuDTOAssembler purchaseSkuDTOAssembler;

    @ApiOperation(value = "根据id查询预约详情")
    @GetMapping("/technician/schedule/queryScheduleManage")
    public JsonResult<ScheduleManageDTO> technicianQueryScheduleManage(@RequestParam Integer id) {
        ScheduleManage scheduleManage = scheduleManageService.queryScheduleManage(id);
        ScheduleManageDTO scheduleManageDTO = scheduleManageDTOAssembler.apply(scheduleManage);
        scheduleManageDTO.getScheduleServeDTOList().forEach(scheduleServeDTO -> {
            //如果预约服务有款式 就查询款式使用色号
            if (scheduleServeDTO.getProduceId() != null) {
                List<PurchaseSkuDTO> purchaseSkuDTOList = purchaseSkuMapper.selectByProduceId(scheduleServeDTO.getProduceId()).stream().map(purchaseSkuDTOAssembler).collect(Collectors.toList());
                scheduleServeDTO.setPurchaseSkuDTOList(purchaseSkuDTOList);

                BaseProduce baseProduce = baseProduceMapper.selectById(scheduleServeDTO.getProduceId());
                scheduleServeDTO.setProducePractice(baseProduce.getPractice());
            }
        });
        return JsonResult.success(scheduleManageDTO);
    }

    @ApiOperation("更新预约")
    @PostMapping("/technician/schedule/updateSchedule")
    public JsonResult<Void> updateSchedule(@RequestBody SubmitScheduleCommand cmd) throws ParseException, ExecutionException, InterruptedException {
        cmd.getScheduleManageDTO().setDepartmentId(SessionUtils.getTechnicianToken().getDepartmentId());
        scheduleManageService.updateSchedule(cmd);
        return JsonResult.success();
    }

    @ApiOperation("更新前检查预约")
    @PostMapping("/technician/schedule/checkBeforeUpdate")
    public JsonResult<Void> checkBeforeUpdate(@RequestBody UpdateScheduleCommand cmd) {

        scheduleManageService.checkBeforeUpdate(cmd);
        return JsonResult.success();
    }


    @ApiOperation("更新预约服务状态")
    @GetMapping("/technician/schedule/updateScheduledServeStatus")
    public JsonResult<Void> updateScheduledServeStatus(@RequestHeader String token,
                                                       @RequestParam Integer scheduleServeId,
                                                       @RequestParam Integer status,
                                                       @RequestParam Integer forceLeisureConfig) {
        ScheduleServeIdLock lock = ScheduleServeIdLock.getInsatance();
        try {
            lock.lock(scheduleServeId);
            scheduleManageService.updateScheduledServeStatus(scheduleServeId, status, forceLeisureConfig);
        } finally {
            lock.unlock(scheduleServeId);
        }
        return JsonResult.success();
    }


    @ApiOperation("当前服务是否在闲时间范围内 1-是 2-否")
    @GetMapping("/technician/schedule/getScheduledServeLeisure")
    public JsonResult<Integer> getScheduledServeLeisure(@RequestHeader String token,
                                                        @RequestParam Integer scheduleServeId) {
        Integer result = scheduleManageService.getScheduledServeLeisure(scheduleServeId);
        return JsonResult.success(result);
    }


    @ApiOperation("分页查询闲时服务")
    @PostMapping("/customer/queryPageLeisureScheduleServe")
    public JsonResult<IPage<LeisureScheduleServe>> queryPageLeisureScheduleServe(@RequestBody LeisureScheduleServeQuery query) {
        IPage<LeisureScheduleServe> page = scheduleManageService.queryPageLeisureScheduleServe(query);
        return JsonResult.success(page);
    }

    @Deprecated
    @ApiOperation("查询闲时美甲师")
    @PostMapping("/customer/queryLeisureTechnician")
    public JsonResult<StoreTechnician> queryLeisureTechnician(@RequestBody QueryLeisureTechnicianReq req) throws ParseException {
        scheduleManageService.queryLeisureTechnician(req.getDepartmentId(), req.getServeId(), req.getStartTime(), req.getEndTime());
        return JsonResult.success();
    }
}
