package com.gogirl.interfaces.user;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.github.binarywang.wxpay.bean.order.WxPayMpOrderResult;
import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.gogirl.application.market.TimesCardService;
import com.gogirl.application.order.serve.OrderServeService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.domain.market.timescard.TimesCardCustomerRelevance;
import com.gogirl.domain.market.timescard.TimesCardType;
import com.gogirl.domain.market.timescard.TimesCardTypeContent;
import com.gogirl.domain.market.timescard.TimesCardUsedRecord;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.annotation.GogirlMember;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.config.property.GogirlProperties;
import com.gogirl.infrastructure.feign.wx.WxPayControllerFeign;
import com.gogirl.infrastructure.mapper.market.discount.DiscountConfigMapper;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardTypeContentMapper;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardTypeMapper;
import com.gogirl.infrastructure.mapper.market.timescard.TimesCardUsedRecordMapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@RestController
@Slf4j
@AllArgsConstructor
@Api(tags = {"version2.0: 次卡"}, value = "version2.0: 次卡")
public class TimesCardController {

    private final TimesCardService timesCardService;
    private final GogirlProperties gogirlProperties;
    private final GogirlTokenService gogirlTokenService;

    /**
     * rpc service
     */

    private final WxPayControllerFeign wxPayControllerFeign;

    private final TimesCardCustomerRelevanceMapper timesCardCustomerRelevanceMapper;
    private final TimesCardTypeMapper timesCardTypeMapper;
    private final TimesCardUsedRecordMapper timesCardUsedRecordMapper;
    private final TimesCardTypeContentMapper timesCardTypeContentMapper;
    private final OrderServeService orderServeService;

    @ApiOperation(value = "查询次卡种类列表", notes = "查询次卡种类列表")
    @GetMapping("/customer/timescard/getTimesCardTypeList")
    public JsonResult<List<TimesCardType>> getTimesCardTypeList() {
        return JsonResult.success(timesCardService.getTimesCardTypeList());
    }

    @ApiOperation(value = "查询次卡种类详情")
    @GetMapping("/customer/timescard/getTimesCardTypeDetail")
    @GogirlMember
    public JsonResult<TimesCardType> getTimesCardTypeDetail(@RequestParam Integer id) {
        return JsonResult.success(timesCardService.getAggregate(id));
    }

    @ApiOperation(value = "用户购买次卡")
    @PostMapping("/customer/timescard/buyTimesCard")
    public JsonResult<WxPayMpOrderResult> buyTimesCard(@RequestHeader String token,
                                                       @RequestParam Integer timesCardId) throws UnknownHostException, WxPayException {
        log.info("********用户购买次卡");
        log.info("获取微信支付签名token:" + token + "orderId:" + timesCardId);
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);

        TimesCardType tct = timesCardService.getTimesCardTypeDetail(timesCardId);

        //计算次卡的价钱
        BigDecimal sumAmount = tct.getPayAmount().multiply(new BigDecimal(tct.getSumTimes()));
        String outTradeNo = IdWorker.getIdStr();
        String attach = timesCardId + "," + gogirlToken.getCustomerId();

        //发起统一支付
        SimpleDateFormat yyyyMMddHHmmss = new SimpleDateFormat("yyyyMMddHHmmss");

        WxPayUnifiedOrderRequest request = WxPayUnifiedOrderRequest.newBuilder()
                .deviceInfo("设备号")
                .body("gogirl美甲美睫")
                .detail("是否显示具体做了哪些服务")
                .attach(attach)
                .outTradeNo(outTradeNo)
                .totalFee(sumAmount.multiply(new BigDecimal("100")).intValue())
                .spbillCreateIp(InetAddress.getLocalHost().getHostAddress())
                .timeStart(yyyyMMddHHmmss.format(new Date()))
                .notifyUrl(gogirlProperties.getNotifyTimesCard())
                .tradeType("JSAPI")
                .openid(gogirlToken.getOpenid())
                .build();

        WxPayMpOrderResult result = wxPayControllerFeign.createJsapiOrder(request);
        return JsonResult.success(result);
    }

    @ApiOperation(value = "查询我的次卡列表")
    @GetMapping("/customer/timescard/getMyTimesCardList")
    public JsonResult<List<TimesCardCustomerRelevance>> getMyTimesCardList(@RequestHeader String token) {
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);

        List<TimesCardCustomerRelevance> myTimesCardList = timesCardCustomerRelevanceMapper.selectList(
                new LambdaQueryWrapper<TimesCardCustomerRelevance>().eq(TimesCardCustomerRelevance::getCustomerId, gogirlToken.getCustomerId())
        );

        //todo
        myTimesCardList.forEach(timesCardCustomerRelevance -> {
            TimesCardType timesCardType = timesCardTypeMapper.selectOne(new LambdaQueryWrapper<TimesCardType>().eq(TimesCardType::getId, timesCardCustomerRelevance.getCardTypeId()));
            timesCardCustomerRelevance.setTimesCardType(timesCardType);
        });
        return JsonResult.success(myTimesCardList);
    }

    @ApiOperation(value = "查询用户次卡详情")
    @GetMapping("/customer/timescard/getTimesCardCustomerRelevanceDetail")
    public JsonResult<TimesCardCustomerRelevance> getTimesCardCustomerRelevanceDetail(@RequestParam Integer id) {

        TimesCardCustomerRelevance timesCardCustomerRelevance = timesCardCustomerRelevanceMapper.selectById(id);

        //次卡类型
        TimesCardType timesCardType = timesCardTypeMapper.selectOne(new LambdaQueryWrapper<TimesCardType>().eq(TimesCardType::getId, timesCardCustomerRelevance.getCardTypeId()));
        timesCardCustomerRelevance.setTimesCardType(timesCardType);

        List<TimesCardTypeContent> cardTypeContentList = timesCardTypeContentMapper.selectList(new LambdaQueryWrapper<TimesCardTypeContent>()
                .eq(TimesCardTypeContent::getCardTypeId, timesCardType.getId()));
        timesCardType.setCardTypeCententList(cardTypeContentList);

        List<TimesCardUsedRecord> usedRecordList = timesCardUsedRecordMapper.selectList(new LambdaQueryWrapper<TimesCardUsedRecord>()
                .eq(TimesCardUsedRecord::getStatus, 2)
                .eq(TimesCardUsedRecord::getCardRelevanceCustomerId, timesCardCustomerRelevance.getId()));

        //次卡使用记录
        timesCardCustomerRelevance.setTimesCardUsedRecordList(usedRecordList);
        return JsonResult.success(timesCardCustomerRelevance);
    }

    private final DiscountConfigMapper discountConfigMapper;

    @ApiOperation(value = "根据订单号查询我的可用的次卡")
    @GetMapping("/customer/timescard/getMyTimesCardByOrderId")
    public JsonResult<List<TimesCardCustomerRelevance>> getMyTimesCardByOrderId(@RequestHeader String token,
                                                                                @RequestParam Integer orderId) {
        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        List<OrderServe> orderServeList = orderServeService.list(new LambdaQueryWrapper<OrderServe>().eq(OrderServe::getOrderId, orderId));

        List<TimesCardCustomerRelevance> timesCardCustomerRelevanceList = timesCardCustomerRelevanceMapper
                .selectList(new LambdaQueryWrapper<TimesCardCustomerRelevance>()
                        .eq(TimesCardCustomerRelevance::getStatus, TimesCardCustomerRelevance.STATUS_NORMAL)
                        .eq(TimesCardCustomerRelevance::getCustomerId, currentCustomerId))
                .stream()
                .filter(TimesCardCustomerRelevance::isValid)
                .filter(timesCardCustomerRelevance -> {
                    //次卡可以作用的服务
                    List<Integer> canUserServeIds = timesCardCustomerRelevanceMapper.queryTimesCardServeIds(timesCardCustomerRelevance.getCardTypeId());
                    //可用服务与当前服务交集
                    canUserServeIds.retainAll(orderServeList.stream()
                            .map(OrderServe::getServeId)
                            .collect(Collectors.toList()));
                    //不为空
                    return ListUtil.isNotEmpty(canUserServeIds);
                })
                .peek(timesCardCustomerRelevance -> {
                    List<Integer> canUserServeIds = timesCardCustomerRelevanceMapper.queryTimesCardServeIds(timesCardCustomerRelevance.getCardTypeId());

                    canUserServeIds.retainAll(orderServeList.stream()
                            .filter(orderServe -> orderServe.getBindOuterCoupon() == 2)
                            .filter(orderServe -> orderServe.getBindInnerCoupon() == 2)
                            .filter(orderServe -> !orderServe.isActivity())
                            .map(OrderServe::getServeId)
                            .collect(Collectors.toList()));
                    timesCardCustomerRelevance.setCanBeUse(ListUtil.isNotEmpty(canUserServeIds));
                })
                .collect(Collectors.toList());
        return JsonResult.success(timesCardCustomerRelevanceList);
    }
}
