package com.gogirl.application.order.mall.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.binarywang.wxpay.bean.request.WxPayRefundRequest;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.gogirl.application.order.mall.MallOrderDetailService;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.application.order.mall.MallRefundOrderDetailService;
import com.gogirl.application.order.mall.MallRefundOrderService;
import com.gogirl.domain.order.mall.MallOrder;
import com.gogirl.domain.order.mall.MallOrderDetail;
import com.gogirl.domain.order.mall.MallRefundOrder;
import com.gogirl.domain.order.mall.MallRefundOrderDetail;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.config.property.GogirlProperties;
import com.gogirl.infrastructure.config.property.WxProperties;
import com.gogirl.infrastructure.mapper.order.mall.MallRefundOrderMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerMapper;
import com.gogirl.infrastructure.service.wx.WxPayService;
import com.gogirl.shared.market.RefundOrderByDetailIdReq;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.stream.Collectors;

@Service
@AllArgsConstructor
@Slf4j
@Transactional
public class MallRefundOrderServiceImpl extends ServiceImpl<MallRefundOrderMapper, MallRefundOrder> implements MallRefundOrderService {

    private final MallOrderService mallOrderService;
    private final MallRefundOrderDetailService mallRefundOrderDetailService;

    private final MallOrderDetailService mallOrderDetailService;

    private final GogirlProperties gogirlProperties;

    private final CustomerMapper customerMapper;

    private final WxProperties wxProperties;

    private final WxPayService wxPayService;

    /**
     * 退款订单参数检查
     *
     * @param mallOrder       需要退款的订单
     * @param mallRefundOrder 前端退款订单参数对象
     */
    private void refundCheck(MallOrder mallOrder, MallRefundOrder mallRefundOrder) {

        if (this.getOne(new LambdaQueryWrapper<MallRefundOrder>()
                .eq(MallRefundOrder::getMallOrderId, mallRefundOrder.getMallOrderId())) != null) {
            throw new RRException(500, "请勿重复提交");
        }
        if (!mallOrder.getCustomerId().equals(mallRefundOrder.getCustomerId())) {
            throw new RRException(500, "不能退款不属于自己的订单");
        }
        if (!mallOrder.getStatus().equals(MallOrder.STATUS_UN_SHIP)) {
            throw new RRException(500, "只能退款等待发货订单");
        }
    }


    /**
     * 根据订单构建全额退款订单对象
     *
     * @param mallOrder       需要退款的订单
     * @param mallRefundOrder 前端退款订单参数对象
     */
    private void buildMallRefundOrderAggregate(MallOrder mallOrder, MallRefundOrder mallRefundOrder) {

        //todo 扩展部分退款 数量和订单详情的支付金额来计算
        mallRefundOrder.setMallOrderDetailList(mallOrder.getMallOrderDetailList()
                .stream()
                .map(mallOrderDetail -> {
                    MallRefundOrderDetail mallRefundOrderDetail = new MallRefundOrderDetail();
                    //退款详情退款金额
                    mallRefundOrderDetail.setAmount(mallOrderDetail.getPayAmount());
                    //退款详情关联的订单详情id
                    mallRefundOrderDetail.setMallOrderDetailId(mallOrderDetail.getId());
                    //退款详情关联的订单详情id
                    mallRefundOrderDetail.setRefundNum(mallOrderDetail.getNum());
                    return mallRefundOrderDetail;
                })
                .collect(Collectors.toList()));


        //设置退款时间
        mallRefundOrder.setRefundTime(System.currentTimeMillis());
        //设置退款时间
        mallRefundOrder.setRefundTotalAmount(mallOrder.getTotalPayAmount());
        //设置退款金额
        mallRefundOrder.setMallOrderId(mallOrder.getId());
        //退款单状态 -退款中
        mallRefundOrder.setStatus(MallRefundOrder.STATUS_REFUNDING);

    }

    @Transactional(timeout = 10000)
    @Override
    public void refundOrder(MallRefundOrder mallRefundOrder) {


        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(mallRefundOrder.getMallOrderId());
        this.refundCheck(mallOrder, mallRefundOrder);

        //商城订单状态-退款中
        mallOrder.setStatus(MallOrder.STATUS_REFUNDING);
        mallOrderService.updateById(mallOrder);


        //根据订单构建退款订单对象
        mallRefundOrder.buildMallRefundOrderAggregate(mallOrder);

        //退款订单状态-退款中
        mallRefundOrder.setStatus(MallRefundOrder.STATUS_REFUNDING);

        this.saveAggregate(mallRefundOrder);
    }

    @Override
    public void saveAggregate(MallRefundOrder mallRefundOrder) {
        this.save(mallRefundOrder);
        mallRefundOrderDetailService.saveBatch(mallRefundOrder.getMallOrderDetailList());
    }

    @Override
    public void refund(Long orderDetailId) throws WxPayException {
        //订单详情
        MallOrderDetail mallOrderDetail = mallOrderDetailService.getById(orderDetailId);


        //订单
        MallOrder mallOrder = mallOrderService.getById(mallOrderDetail.getOrderId());

        //用户
        Customer customer = customerMapper.selectById(mallOrder.getCustomerId());

        //退款单
        MallRefundOrder mallRefundOrder;

        //退款单详情
        MallRefundOrderDetail mallRefundOrderDetail;

        //退款中
        if (mallOrderDetail.getRefundStatus().equals(MallOrderDetail.REFUND_STATUS_REFUNDING)) {
            mallRefundOrderDetail = mallRefundOrderDetailService.getOne(new LambdaQueryWrapper<MallRefundOrderDetail>().eq(MallRefundOrderDetail::getMallOrderDetailId, mallOrderDetail.getId()));
            mallRefundOrder = this.getById(mallRefundOrderDetail.getMallRefundOrderId());
        }
        //正常
        else if (mallOrderDetail.getRefundStatus().equals(MallOrderDetail.REFUND_STATUS_NORMAL)) {
            mallOrderDetail.setRefundStatus(MallOrderDetail.REFUND_STATUS_REFUNDED);
            mallOrderDetailService.updateById(mallOrderDetail);

            mallRefundOrder = MallRefundOrder.builder()
                    //订单id
                    .mallOrderId(mallOrder.getId())
                    //退款联系人人姓名
                    .refundContactName(customer.getRealName())
                    //退款联系人手机号
                    .refundContactPhone(customer.getPhone())
                    //退款理由
                    .refundReason("后台退款")
                    //退款时间
                    .refundTime(System.currentTimeMillis())
                    //退款总金额
                    .refundTotalAmount(mallOrderDetail.getPayAmount())
                    //用户id
                    .customerId(customer.getId())
                    ///状态已退款
                    .status(MallRefundOrder.STATUS_REFUNDING)
                    .build();
            this.save(mallRefundOrder);

            //退款单
            mallRefundOrderDetail = MallRefundOrderDetail.builder()
                    .mallOrderDetailId(mallOrderDetail.getId())
                    .mallRefundOrderId(mallRefundOrder.getId())
                    //退款数量
                    .refundNum(mallOrderDetail.getNum())
                    //退款金额
                    .amount(mallOrderDetail.getAmount())
                    .build();
            mallRefundOrderDetailService.save(mallRefundOrderDetail);
        }
        //invalid
        else {
            throw new RRException("invalid status");
        }

        WxPayRefundRequest refundRequest = WxPayRefundRequest.newBuilder()
                .notifyUrl(gogirlProperties.getNotifyRefundMallOrder())
                .outTradeNo(String.valueOf(mallRefundOrder.getId()))
                .outRefundNo(String.valueOf(mallRefundOrder.getId()))
                .totalFee(mallRefundOrder.getRefundTotalAmount().intValue() * 100)
                .refundFee(mallRefundOrder.getRefundTotalAmount().intValue() * 100)
                .build();

        refundRequest.setAppid(wxProperties.getCustomerMiniAppId());
        wxPayService.refund(refundRequest);
    }

    @Override
    public void refundOrderByDetailId(RefundOrderByDetailIdReq request) {
        //订单详情
        MallOrderDetail mallOrderDetail = mallOrderDetailService.getById(request.getMallOrderDetailId());
        //退款中
        mallOrderDetail.setRefundStatus(MallOrderDetail.REFUND_STATUS_REFUNDING);
        mallOrderDetailService.updateById(mallOrderDetail);

        MallRefundOrder mallRefundOrder = MallRefundOrder.builder()
                //订单id
                .mallOrderId(request.getMallOrderId())
                //退款联系人人姓名
                .refundContactName(request.getRefundContactName())
                //退款联系人手机号
                .refundContactPhone(request.getRefundContactPhone())
                //退款理由
                .refundReason(request.getRefundReason())
                //退款时间
                .refundTime(System.currentTimeMillis())
                //退款总金额
                .refundTotalAmount(mallOrderDetail.getPayAmount())
                //用户id
                .customerId(request.getCustomerId())
                ///状态退款中
                .status(MallRefundOrder.STATUS_REFUNDING)
                .build();
        this.save(mallRefundOrder);

        //退款单
        MallRefundOrderDetail mallRefundOrderDetail = MallRefundOrderDetail.builder()
                .mallOrderDetailId(mallOrderDetail.getId())
                .mallRefundOrderId(mallRefundOrder.getId())
                //退款数量
                .refundNum(mallOrderDetail.getNum())
                //退款金额
                .amount(mallOrderDetail.getAmount())
                .build();
        mallRefundOrderDetailService.save(mallRefundOrderDetail);
    }


}
