package com.gogirl.application.store.store.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.TakeLeaveEventCmdService;
import com.gogirl.assembler.ApplyTakeLeaveCommandAssembler;
import com.gogirl.assembler.OverTimeLogCommandAssembler;
import com.gogirl.domain.store.oa.OverTimeRecordLog;
import com.gogirl.domain.store.oa.OvertimeRecord;
import com.gogirl.domain.store.oa.TakeLeaveEvent;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.store.*;
import com.gogirl.shared.store.command.ApplyTakeLeaveCommand;
import com.gogirl.shared.store.command.ApprovalTakeLeaveCommand;
import com.gogirl.shared.store.query.qry.PageApplyQuery;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
@Transactional
public class TakeLeaveEventCmdServiceImpl implements TakeLeaveEventCmdService {

    /**
     * assembler
     */
    private final ApplyTakeLeaveCommandAssembler applyTakeLeaveCommandAssembler;
    private final OverTimeLogCommandAssembler overTimeLogCommandAssembler;

    /**
     * repository
     */
    private final StoreTechnicianMapper storeTechnicianRepository;
    private final TakeLeaveEventMapper takeLeaveEventRepository;
    private final OvertimeRecordMapper overtimeRecordRepository;
    private final OverTimeRecordLogMapper overTimeRecordLogRepository;


    @Override
    public void applyTakeLeave(ApplyTakeLeaveCommand cmd) {

        TakeLeaveEvent takeLeaveEvent = applyTakeLeaveCommandAssembler.apply(cmd);
        takeLeaveEvent.setStatus(TakeLeaveEvent.STATUS_UN_APPROVAL);

        takeLeaveEvent.setTimeLength(cmd.getTimeLength());

        //处理人
        StoreTechnician storeTechnician = storeTechnicianRepository.selectById(cmd.getDealingTechnicianId());
        takeLeaveEvent.setDealingTechnicianName(storeTechnician.getName());
        takeLeaveEvent.setCreateTime(System.currentTimeMillis());

        //
        takeLeaveEventRepository.insert(takeLeaveEvent);

        //调休
        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_ADJUST_REST)) {

            List<OverTimeRecordLog> overTimeRecordLogList = cmd.getOverTimeLogCommandList()
                    .stream().map(overTimeLogCommandAssembler)
                    .peek(overTimeRecordLog -> overTimeRecordLog.setTakeLeaveEventId(takeLeaveEvent.getId()))
                    .collect(Collectors.toList());

            //检查总调休时间
            if (!takeLeaveEvent.getTimeLength().toString()
                    .equals(overTimeRecordLogList.stream().map(OverTimeRecordLog::getLengthTime).reduce(BigDecimal.ZERO, BigDecimal::add).toString())) {
                throw new RRException(500, "调休时间有误");
            }

            overTimeRecordLogList.forEach(overTimeRecordLog -> {
                //加班记录
                OvertimeRecord overtimeRecord = overtimeRecordRepository.selectById(overTimeRecordLog.getOverTimeRecordId());
                //扣除这一次申请的时间
                overtimeRecord.setLeftTimeLength(overtimeRecord.getLeftTimeLength().subtract(overTimeRecordLog.getLengthTime()));
                //检查调休时间和加班时间匹配
                if (overtimeRecord.getLeftTimeLength().compareTo(BigDecimal.ZERO) < 0) {
                    throw new RRException(500, "超出加班时间");
                }
                if (overtimeRecord.getLeftTimeLength().compareTo(BigDecimal.ZERO) == 0) {
                    //不能调休
                    overtimeRecord.setStatus(2);
                }
                overtimeRecord.setLastUpdateTime(System.currentTimeMillis());
                //更新加班使用情况
                overtimeRecordRepository.updateById(overtimeRecord);
            });

            //写入加班使用情况日志
            overTimeRecordLogList.forEach(overTimeRecordLogRepository::insert);
        }
    }

    @Transactional
    @Override
    public void approvalTakeLeave(ApprovalTakeLeaveCommand cmd) {

        TakeLeaveEvent takeLeaveEvent = takeLeaveEventRepository.selectById(cmd.getTakeLeaveEventId());

        if (!takeLeaveEvent.getStatus().equals(TakeLeaveEvent.STATUS_UN_APPROVAL)) {
            throw new RRException(500, "审批状态异常");
        }

        //加班申请通过 添加加班记录
        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_OVER_TIME)
                && cmd.getApproval().equals(TakeLeaveEvent.STATUS_APPROVAL_PASS)) {

            OvertimeRecord overtimeRecord = OvertimeRecord.builder()
                    .createTime(System.currentTimeMillis())
                    .startTime(takeLeaveEvent.getStartTime())
                    .endTime(takeLeaveEvent.getEndTime())
                    .lastUpdateTime(System.currentTimeMillis())
                    .leftTimeLength(takeLeaveEvent.getTimeLength())
                    .status(1)
                    .totalTimeLength(takeLeaveEvent.getTimeLength())
                    .technicianId(takeLeaveEvent.getApplyTechnicianId())
                    .build();
            overtimeRecordRepository.insert(overtimeRecord);
        }


        //调休申请拒绝
        if (takeLeaveEvent.getType().equals(TakeLeaveEvent.TYPE_ADJUST_REST)
                && cmd.getApproval().equals(TakeLeaveEvent.STATUS_APPROVAL_REFUSE)) {
            Long takeLeaveEventId = cmd.getTakeLeaveEventId();
            List<OverTimeRecordLog> overTimeRecordLogList = overTimeRecordLogRepository
                    .selectList(new LambdaQueryWrapper<OverTimeRecordLog>().eq(OverTimeRecordLog::getTakeLeaveEventId, takeLeaveEventId));

            overTimeRecordLogList.stream().collect(Collectors.groupingBy(OverTimeRecordLog::getOverTimeRecordId))
                    .forEach((overTimeRecordId, list) -> {
                        OvertimeRecord overtimeRecord = overtimeRecordRepository.selectById(overTimeRecordId);
                        BigDecimal timeLength = list.stream().map(OverTimeRecordLog::getLengthTime).reduce(BigDecimal.ZERO, BigDecimal::add);
                        overtimeRecord.setLeftTimeLength(overtimeRecord.getLeftTimeLength().add(timeLength));
                        overtimeRecordRepository.insert(overtimeRecord);
                    });

            overTimeRecordLogRepository.deleteBatchIds(overTimeRecordLogList.stream().map(OverTimeRecordLog::getId).collect(Collectors.toList()));
        }

        takeLeaveEvent.setStatus(cmd.getApproval());
        takeLeaveEvent.setRejectReason(cmd.getRejectReason());
        takeLeaveEventRepository.updateById(takeLeaveEvent);
    }

    @Override
    public TakeLeaveEvent queryTakeLeave(Long id) {
        TakeLeaveEvent takeLeaveEvent = takeLeaveEventRepository.selectById(id);
        StoreTechnician storeTechnician = storeTechnicianRepository.selectById(takeLeaveEvent.getApplyTechnicianId());
        takeLeaveEvent.setApplyStoreTechnicianDTO(storeTechnician);
        return takeLeaveEvent;
    }

    @Override
    public IPage<TakeLeaveEvent> queryPageTakeLeave(PageApplyQuery qry) {

        LambdaQueryWrapper<TakeLeaveEvent> wrapper = new LambdaQueryWrapper<>();
        if (qry.getApplyTechnicianId() != null) {
            wrapper.eq(TakeLeaveEvent::getApplyTechnicianId, qry.getApplyTechnicianId());
        }
        if (qry.getDealingTechnicianId() != null) {
            wrapper.eq(TakeLeaveEvent::getDealingTechnicianId, qry.getDealingTechnicianId());
        }
        if (qry.getStatus() != null) {
            wrapper.eq(TakeLeaveEvent::getStatus, qry.getStatus());
        }
        wrapper.orderByDesc(TakeLeaveEvent::getCreateTime);
        IPage<TakeLeaveEvent> page = new Page<>(qry.getPageNum(), qry.getPageSize());

        page = takeLeaveEventRepository.selectPage(page, wrapper);

        if (ListUtil.isEmpty(page.getRecords())) {
            return page;
        }
        List<Integer> ids = page.getRecords().stream().map(TakeLeaveEvent::getApplyTechnicianId).collect(Collectors.toList());
        List<StoreTechnician> storeTechnicianList = storeTechnicianRepository.selectBatchIds(ids);
        Map<Integer, List<StoreTechnician>> map = storeTechnicianList.stream().collect(Collectors.groupingBy(StoreTechnician::getId));

        page.getRecords().forEach(takeLeaveEvent -> {
            if (ListUtil.isNotEmpty(map.get(takeLeaveEvent.getApplyTechnicianId()))) {
                StoreTechnician storeTechnician = map.get(takeLeaveEvent.getApplyTechnicianId()).stream().findAny().orElse(null);
                takeLeaveEvent.setApplyStoreTechnicianDTO(storeTechnician);
            }
        });
        return page;
    }

    private final StoreManageMapper storeManageMapper;

    @Override
    public List<StoreTechnician> queryCheckTechnicianList(Integer technicianId, Integer departmentId) {
        StoreManage storeManage = storeManageMapper.selectById(departmentId);
        StoreTechnician storeTechnician;
        if (storeManage.getMasterUserId().equals(technicianId))
            storeTechnician = storeTechnicianRepository.selectById(storeManage.getOperateUserId());
        else
            storeTechnician = storeTechnicianRepository.selectById((storeManage.getMasterUserId()));
        return Lists.newArrayList(storeTechnician);
    }

}
