package com.gogirl.interfaces.order.mall;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.domain.order.mall.MallOrder;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.shared.market.ListMallOrderRequest;
import com.gogirl.shared.market.SubmitMallOrderCommand;
import com.gogirl.shared.member.ImmediatePurchaseCommand;
import com.gogirl.shared.order.CancelMallOrderCommand;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;

@Api(tags = "商城订单接口", value = "商城订单接口")
@RestController
@AllArgsConstructor
@Slf4j
public class MallOrderController {

    private final MallOrderService mallOrderService;


    @ApiOperation("提交订单")
    @PostMapping("/customer/mallOrder/submitMallOrder")
    public JsonResult<MallOrder> submitMallOrder(@RequestHeader String token,
                                                 @RequestBody SubmitMallOrderCommand cmd) {

        log.info("token:{}", SessionUtils.getCustomerToken().toString());
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCustomerId(currentCustomerId);
        mallOrderService.submitMallOrder(cmd);
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(cmd.getExecutionResult());
        return JsonResult.success(mallOrder);
    }

    @ApiOperation("立即购买")
    @PostMapping("/customer/mallOrder/immediatePurchase")
    public JsonResult<MallOrder> immediatePurchase(@RequestHeader String token,
                                                   @RequestBody ImmediatePurchaseCommand cmd) {
        log.info("token：{}", token);
        log.info("token:{}", SessionUtils.getCustomerToken().toString());
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCustomerId(currentCustomerId);
        mallOrderService.immediatePurchase(cmd);
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(cmd.getExecutionResult());
        return JsonResult.success(mallOrder);
    }


    @ApiOperation("订单确认收货")
    @GetMapping("/customer/mallOrder/takeDeliveryOfMallOrder")
    public JsonResult<Integer> takeDeliveryOfMallOrder(@RequestHeader String token,
                                                       @PathVariable Long mallOrderId) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        mallOrderService.takeDeliveryOfMallOrder(currentCustomerId, mallOrderId);
        return JsonResult.success();
    }

    @ApiOperation("我的订单列表")
    @PostMapping("/customer/mallOrder/listMallOrderAggregate")
    public JsonResult<IPage<MallOrder>> listMallOrderAggregate(@RequestHeader String token,
                                                               @RequestBody ListMallOrderRequest reqDTO) {

        Integer currentCustomerId = SessionUtils.getCustomerId();
        Page<MallOrder> page = new Page<>();
        page.setCurrent(reqDTO.getPageIndex());
        page.setSize(reqDTO.getPageSize());

        MallOrder mallOrder = ListMallOrderRequest.convertToMallOrder(reqDTO);
        mallOrder.setCustomerId(currentCustomerId);
        IPage<MallOrder> data = mallOrderService.listMallOrderAggregate(page, mallOrder);
        return JsonResult.success(data);
    }

    @ApiOperation("查看订单详情")
    @GetMapping("/customer/mallOrder/getMallOrderAggregate/{mallOrderId}")
    public JsonResult<MallOrder> getMallOrderAggregate(@RequestHeader String token,
                                                       @PathVariable Long mallOrderId) {
        MallOrder mallOrder = mallOrderService.getMallOrderAggregate(mallOrderId);
        return JsonResult.success(mallOrder);
    }

    @ApiOperation(("/用户取消订单"))
    @PostMapping("/customer/mallOrder/cancelMallOrder")
    public JsonResult<Void> cancelMallOrder(@RequestHeader String token, @RequestBody CancelMallOrderCommand cmd) {
        Integer currentCustomerId = SessionUtils.getCustomerId();
        cmd.setCurrentCustomerId(currentCustomerId);
        mallOrderService.cancelMallOrder(cmd);
        return JsonResult.success();
    }


    @ApiModelProperty("/查询运费")
    @GetMapping("/customer/mallOrder/queryFreight")
    public JsonResult<BigDecimal> queryFreight(@RequestParam BigDecimal totalPrice) {
        BigDecimal freight = totalPrice.compareTo(new BigDecimal(100)) > 0 ? new BigDecimal("0.00") : new BigDecimal("6.00");
        return JsonResult.success(freight);
    }

}
