package com.gogirl.application.store.store.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.CareerCmdService;
import com.gogirl.application.store.store.WorksImagesService;
import com.gogirl.domain.product.purchase.PurchaseSku;
import com.gogirl.domain.product.serve.BaseProduce;
import com.gogirl.domain.store.career.Career;
import com.gogirl.domain.store.career.CareerWorks;
import com.gogirl.domain.store.career.CareerWorksImages;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.common.util.SessionUtils;
import com.gogirl.infrastructure.mapper.product.purchase.PurchaseSkuMapper;
import com.gogirl.infrastructure.mapper.product.serve.BaseProduceMapper;
import com.gogirl.infrastructure.mapper.store.career.CareerMapper;
import com.gogirl.infrastructure.mapper.store.career.CareerWorksImagesMapper;
import com.gogirl.infrastructure.mapper.store.career.WorksMapper;
import com.gogirl.shared.store.SubmitWorksCommand;
import com.gogirl.shared.store.WorksQuery;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@Slf4j
@AllArgsConstructor
@Transactional
public class CareerCmdServiceImpl implements CareerCmdService {


    private final WorksMapper worksMapper;
    private final WorksImagesService worksImagesService;

    private final PurchaseSkuMapper purchaseSkuMapper;
    private final BaseProduceMapper baseProduceMapper;

    @Override
    public void submitWorks(SubmitWorksCommand cmd) {

        CareerWorks submittedWork = worksMapper.selectOne(new LambdaQueryWrapper<CareerWorks>()
                .eq(CareerWorks::getTechnicianId, cmd.getTechnicianId())
                .eq(CareerWorks::getProduceId, cmd.getProduceId())
        );
        if (submittedWork != null) {
            throw new RRException("您已经上传过美甲作品");
        }

        CareerWorks works = new CareerWorks();
        works.setCreateTime(System.currentTimeMillis());
        works.setRemarks(cmd.getRemarks());
        works.setTechnicianId(cmd.getTechnicianId());
        works.setProduceId(cmd.getProduceId());

        BaseProduce baseProduce = baseProduceMapper.selectById(works.getProduceId());
        works.setProduceName(baseProduce.getName());
        worksMapper.insert(works);

        List<CareerWorksImages> worksImagesList = cmd.getImageUrlList()
                .stream()
                .map(imageUrl -> {
                    CareerWorksImages worksImages = new CareerWorksImages();
                    worksImages.setImageUrl(imageUrl);
                    worksImages.setWorksId(works.getId());
                    return worksImages;
                })
                .collect(Collectors.toList());

        worksImagesService.saveBatch(worksImagesList);
    }


    @Override
    public IPage<CareerWorks> queryWorks(WorksQuery qry) {

        IPage<CareerWorks> page = new Page<>(qry.getPageNum(), qry.getPageSize());

        Calendar start = Calendar.getInstance();
        start.set(Calendar.YEAR, qry.getYear());
        start.set(Calendar.MONTH, qry.getMonth() - 1);
        start.set(Calendar.DAY_OF_MONTH, start.getActualMinimum(Calendar.DAY_OF_MONTH));

        Calendar end = Calendar.getInstance();
        end.set(Calendar.YEAR, qry.getYear());
        end.set(Calendar.MONTH, qry.getMonth() - 1);
        end.set(Calendar.DAY_OF_MONTH, end.getActualMaximum(Calendar.DAY_OF_MONTH));

        long startTime = start.getTimeInMillis();
        long entTime = end.getTimeInMillis();

        page = worksMapper.selectPage(page, new LambdaQueryWrapper<CareerWorks>()
                .gt(CareerWorks::getCreateTime, startTime)
                .lt(CareerWorks::getCreateTime, entTime)
                .eq(CareerWorks::getTechnicianId, SessionUtils.getTechnicianId())
                .orderByDesc(CareerWorks::getCreateTime));

        if (ListUtil.isEmpty(page.getRecords())) {
            return page;
        }

        List<Long> ids = page.getRecords().stream().map(CareerWorks::getId).collect(Collectors.toList());
        List<CareerWorksImages> worksImages = worksImagesMapper.selectList(new LambdaQueryWrapper<CareerWorksImages>().in(CareerWorksImages::getWorksId, ids));
        Map<Long, List<CareerWorksImages>> map = worksImages.stream().collect(Collectors.groupingBy(CareerWorksImages::getWorksId));
        page.getRecords().forEach(works -> {
            List<CareerWorksImages> worksImagesList = map.get(works.getId());
            if (ListUtil.isNotEmpty(worksImagesList)) {
                works.setWorksImagesList(new HashSet<>(worksImagesList));
            }
        });
        return page;
    }

    private final CareerWorksImagesMapper worksImagesMapper;


    private final CareerMapper careerMapper;

    @Override
    public Career career(Integer technicianId) {
        Career career = careerMapper.selectOne(new LambdaQueryWrapper<Career>().eq(Career::getTechnicianId, technicianId));
        if (career == null) {
            career = new Career();
            career.setTechnicianId(technicianId);
        }
        return career;
    }

    @Override
    public Page<BaseProduce> queryWorkTpl(Integer technicianId, Integer pageNum, Integer pageSize, String searchKeyWord) {
        Page<BaseProduce> baseProducePage = new Page<>(pageNum, pageSize);

        baseProducePage = careerMapper.queryWorkTpl(baseProducePage, technicianId, searchKeyWord);

        List<PurchaseSku> purchaseSkuList = purchaseSkuMapper.listByProduceIds(baseProducePage.getRecords().stream().map(BaseProduce::getId).collect(Collectors.toList()));

        Map<Integer, List<PurchaseSku>> map = purchaseSkuList.stream().collect(Collectors.groupingBy(PurchaseSku::getProduceId));

        baseProducePage.getRecords().forEach(baseProduce -> {
            List<PurchaseSku> list = map.get(baseProduce.getId());
            if (ListUtil.isNotEmpty(list)) {
                baseProduce.setPurchaseSkuList(list);
            }
        });

        return baseProducePage;
    }

}
