package com.gogirl.interfaces.customer;

import com.gogirl.application.common.xcx.GogirlTokenService;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.infrastructure.common.annotation.AuthIgnore;
import com.gogirl.infrastructure.common.annotation.LoginIgnore;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.shared.member.CustomerOrderDetail;
import com.gogirl.shared.user.CustomerQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.security.InvalidAlgorithmParameterException;
import java.util.Date;
import java.util.List;

@Api(tags = {"用户接口"})
@RestController
@Slf4j
public class CustomerController {

    @Resource
    private CustomerService customerService;
    @Resource
    private GogirlTokenService gogirlTokenService;

    @ApiOperation("用户授权接街道地址信息")
    @GetMapping("/customer/xcx/bindAddressInfo")
    public JsonResult<String> bindAddressInfo(@RequestParam Integer customerId,
                                              @RequestParam String province,
                                              @RequestParam String city,
                                              @RequestParam String area,
                                              @RequestParam String street) {
        customerService.bindAddressInfo(customerId, province, city, area, street);
        return JsonResult.success();
    }

    @LoginIgnore
    @AuthIgnore
    @ApiOperation(value = "小程序用户登录")
    @GetMapping("/customer/xcx/login")
    public JsonResult<String> login(@RequestParam String code) {
        String token = customerService.login(code);
        return JsonResult.success(token);
    }

    @LoginIgnore
    @AuthIgnore
    @ApiOperation(value = "客户授权后绑定个人信息")
    @RequestMapping(method = {RequestMethod.GET, RequestMethod.POST}, value = "/customer/xcx/authorized1")
    public JsonResult<Customer> authorized1(@RequestParam(name = "token", required = false) String token,
                                            @RequestParam(name = "encryptedData", required = false) String encryptedData,
                                            @RequestParam(name = "iv", required = false) String iv) throws InvalidAlgorithmParameterException {
        Customer customer = customerService.authorized1(token, encryptedData, iv);
        return JsonResult.success(customer);
    }

    @ApiOperation(value = "h5用户登录")
    @GetMapping("/customer/no_h5Login")
    public JsonResult<String> h5Login(@RequestParam Integer orderId) {
        String token = customerService.h5Login(orderId);
        return JsonResult.success(token);
    }


    @ApiOperation("查询会员")
    @PostMapping("/customer/xcx/query")
    public JsonResult<Customer> query(@RequestBody CustomerQuery qry) {
        Customer customer = customerService.getById(qry.getCustomerId());
        return JsonResult.success(customer);
    }


    @AuthIgnore
    @LoginIgnore
    @ApiOperation(value = "客户授权手机号码")
    @PostMapping("/customer/xcx/authorizedPhone")
    public JsonResult<Customer> authorizedPhone(@RequestHeader String token,
                                                @RequestParam String encryptedData,
                                                @RequestParam String iv) throws InvalidAlgorithmParameterException {
        Customer customer = customerService.authorizedPhone(token, encryptedData, iv);
        return JsonResult.success(customer);
    }

    @AuthIgnore
    @ApiOperation(value = "客户根据token获取用户信息")
    @GetMapping("/customer/xcx/getUserInfo")
    public JsonResult<Customer> getUserInfo(@RequestParam String token) throws Exception {
        Customer customer = customerService.getUserInfo(token);
        return JsonResult.success(customer);
    }

    @ApiOperation(value = "获取验证码，用于绑定手机号码")
    @PostMapping("/customer/phone/getBindCode")
    public JsonResult<String> getBindCode(@RequestParam String phone) {
        String code = gogirlTokenService.getBindCode(phone);
        return JsonResult.success(code);
    }

    @ApiOperation(value = "小程序用户根据验证码绑定手机号码和称呼 生日的格式是:yyyy-MM-dd")
    @PostMapping("/customer/phone/bindPhoneAndName")
    public JsonResult<Customer> bindPhoneAndCode(@RequestParam(required = false) String phone,
                                                 @RequestParam(required = false) String code,
                                                 @RequestParam(required = false) String realName,
                                                 @RequestParam(required = false)
                                                 @DateTimeFormat(pattern = "yyyy-MM-dd") Date birthday,
                                                 @RequestParam(required = false) String sex) throws Exception {
        Customer customer = gogirlTokenService.bindPhoneAndCode(phone, code, realName, birthday, sex);
        return JsonResult.success(customer);

    }

    @ApiOperation(value = "查询用户可选年龄段")
    @PostMapping("/technician/getAgeGroups")
    public JsonResult<List<String>> getAgeGroups() {
        List<String> ageGroups = customerService.getAgeGroups();
        return JsonResult.success(ageGroups);
    }

    @ApiOperation(value = "用户详情数据查看")
    @PostMapping("/technician/queryCustomerDetail")
    public JsonResult<CustomerOrderDetail> queryCustomerDetail(@RequestParam Integer customerId,
                                                               @RequestParam(required = false) Integer orderId) {
        CustomerOrderDetail customerOrderDetail = customerService.queryCustomerDetail(customerId, orderId);
        return JsonResult.success(customerOrderDetail);
    }

    @ApiOperation(value = "美甲师修改用户详情")
    @PostMapping("/technician/updateCustomerDetail")
    public JsonResult<Void> updateCustomerDetail(@RequestParam(required = false) Integer birthdayMonth,
                                                 @RequestParam(required = false) Integer birthdayDay,
                                                 @RequestParam(required = false) String ageGroup,
                                                 @RequestParam Integer customerId,
                                                 @RequestParam(required = false) Integer orderId,
                                                 @RequestParam(required = false) Integer orderServeId,
                                                 @RequestParam(required = false) String storeRecordRealName,
                                                 @RequestParam(required = false) String sex,
                                                 @RequestParam(required = false) Integer age,
                                                 @RequestParam(required = false) String job,
                                                 @RequestParam(required = false) String preference,
                                                 @RequestParam(required = false) String character,
                                                 @RequestParam(required = false) Integer customerSource) {
        customerService.updateCustomerDetail(birthdayMonth, birthdayDay, ageGroup, customerId, orderId, orderServeId, storeRecordRealName, sex, age, job, preference, character, customerSource);
        return JsonResult.success();
    }

    @LoginIgnore
    @AuthIgnore
    @ApiOperation(value = "获取验证码，用于绑定手机号码")
    @PostMapping("/technician//phone/getBindCode")
    public JsonResult<String> techGetBindCode(@RequestParam String phone) {
        String code = gogirlTokenService.getBindCode(phone);
        return JsonResult.success(code);
    }


    @LoginIgnore
    @AuthIgnore
    @ApiOperation(value = "美甲师根据验证码绑定手机号码")
    @PostMapping("/technician/phone/bindPhone_t")
    public JsonResult<Void> bindPhone_t(@RequestParam String phone,
                                        @RequestParam String code,
                                        @RequestParam String messageCode) {
        gogirlTokenService.bindTechnicianIdPhone(phone, code, messageCode);
        return JsonResult.success();
    }

}
