package com.gogirl.domain.market.coupon;

import com.baomidou.mybatisplus.annotation.*;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.infrastructure.common.annotation.DictParam;
import com.gogirl.infrastructure.common.util.SessionUtils;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Date;

@ApiModel("优惠券领取记录表")
@Builder
@AllArgsConstructor
@NoArgsConstructor
@TableName("coupon_customer_relevance")
@Data
public class CouponCustomerRelevance implements Serializable {

    public static Integer STATE_CHOSE = 1;
    public static Integer STATE_USED = 2;

    public static Integer SOURCE_TYPE_INNER = 0;
    public static Integer SOURCE_TYPE_OUTER = 1;

    public static Integer TYPE_DISCOUNT = 5;

    @TableId(type = IdType.AUTO)
    private Integer id;

    @ApiModelProperty("顾客Id")
    private Integer customerId;

    @ApiModelProperty("卡券id")
    private Integer couponId;

    @DictParam(dictCode = "COUPON_CUSTOMER_RELEVANCE_STATE", dictValueParam = "stateStr")
    @ApiModelProperty("该优惠券使用状态1.正常,2.已使用,3.已过期")
    private Integer state;

    @TableField(exist = false)
    @ApiModelProperty("该优惠券使用状态1.正常,2.已使用,3.已过期")
    private String stateStr;

    @ApiModelProperty("领奖码")
    private String code;

    @ApiModelProperty("领券时间")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    private Date receiveTime;

    @ApiModelProperty("可用开始日期")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    private Date validStartTime;

    @ApiModelProperty("可用截止日期")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    private Date validEndTime;

    @ApiModelProperty("使用日期")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    private Date useDate;

    //冗余coupon字段
    @ApiModelProperty("卡券配置备注")
    private String remark;

    @ApiModelProperty("能否使用")
    @TableField(exist = false)
    private Boolean canBeUse;

    @TableField(exist = false)
    private Coupon coupon;

    @TableField(exist = false)
    private Customer customer;

    @ApiModelProperty("卡券名称")
    private String couponName;

    @ApiModelProperty("优惠金额")
    private BigDecimal discountAmount;

    @ApiModelProperty("满这个金额才能减")
    private BigDecimal reachingAmount;

    @ApiModelProperty("优惠折扣")
    private BigDecimal discountPercent;

    @DictParam(dictCode = "COUPON_TYPE", dictValueParam = "typeStr")
    @ApiModelProperty("优惠券类型:1.现金抵扣券;2.免单券；3满减券；4卸甲券；5 = 折扣券")
    private Integer type;

    @TableField(exist = false)
    private String typeStr;
    @ApiModelProperty("0 = 内部券 1 = 外部券")
    @DictParam(dictCode = "COUPON_SOURCE_TYPE", dictValueParam = "sourceTypeStr")
    private Integer sourceType;

    @TableField(exist = false)
    private String sourceTypeStr;

    @ApiModelProperty("使用条件:1.重叠使用")
    private String condition1;

    @ApiModelProperty("客户购买该券时，实际支付的金额(用该金额算业绩)")
    private BigDecimal payAmount;

    @ApiModelProperty("卡券来源")
    private String sourceFrom;

    @ApiModelProperty("卡券来源id")
    private String sourceFromId;

    @TableField(strategy = FieldStrategy.IGNORED)
    private Integer orderId;

    @ApiModelProperty("1-计算 2-不计算")
    private Integer isCalcAchievement = 1;

    @ApiModelProperty("是否免单券 1-是 2-否")
    private Integer isFreeVoucher;

    /**
     * 根据优惠券配置构造一张优惠券
     * <p>
     * 只能 被{@link com.gogirl.application.market.coupon.impl.CouponServiceImpl#sendCoupon(Integer, Integer)} 使用
     *
     * @param coupon 优惠券配置
     * @return
     */
    public static CouponCustomerRelevance getInstance(Coupon coupon) {

        CouponCustomerRelevance customerRelevance = CouponCustomerRelevance.builder()
                //使用条件
                .condition1(coupon.getCondition1())
                //卡券配置id
                .couponId(coupon.getId())
                //卡券名字
                .couponName(coupon.getName())
                //卡券折扣金额
                .discountAmount(coupon.getDiscountAmount())
                //卡券折扣
                .discountPercent(coupon.getDiscountRate())
                //卡券支付基恩
                .payAmount(coupon.getPayAmount())
                //达到多少金额可用
                .reachingAmount(coupon.getReachingAmount())
                //获取时间
                .receiveTime(new Date())
                //卡券来源
                .sourceFrom(SessionUtils.getSourceFrom() + "小程序发券")
                //卡券内部券or外部券
                .sourceType(coupon.getSourceType())
                //卡券类型
                .type(coupon.getType())
                //卡券状态 1
                .state(CouponCustomerRelevance.STATE_CHOSE)
                .build();
        if (coupon.getValidType().equals(1)) {
            customerRelevance.setValidStartTime(coupon.getValidStartTime());
            customerRelevance.setValidEndTime(coupon.getValidEndTime());
        } else {
            long startMills = System.currentTimeMillis();
            //必须加L否则有精度错误
            long endMills = startMills + coupon.getValidDate() * 24 * 60 * 60 * 1000L;
            Date startTime = new Date(startMills);
            Date endTime = new Date(endMills);
            customerRelevance.setValidStartTime(startTime);
            customerRelevance.setValidEndTime(endTime);
        }
        return customerRelevance;
    }


    public boolean isValid() {
        return validStartTime.getTime() <= System.currentTimeMillis()
                && validEndTime.getTime() >= System.currentTimeMillis();
    }


}
