package com.gogirl.interfaces.store.career;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.CareerCmdService;
import com.gogirl.application.store.store.StoreDataService;
import com.gogirl.assembler.TrainingStartClassDTOAssembler;
import com.gogirl.assembler.WorksAssembler;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.product.serve.BaseProduce;
import com.gogirl.domain.store.career.Career;
import com.gogirl.domain.store.career.Works;
import com.gogirl.domain.store.store.StoreData;
import com.gogirl.domain.store.store.TechnicianAchieve;
import com.gogirl.domain.store.training.SkillSku;
import com.gogirl.domain.store.training.TechnicianScore;
import com.gogirl.domain.store.training.TrainingStartClass;
import com.gogirl.domain.store.training.TrainingStartClassLogTechnician;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.store.store.TechnicianScoreMapper;
import com.gogirl.infrastructure.mapper.store.training.SkillSkuMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassLogTechnicianMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.infrastructure.util.lock.WorkLock;
import com.gogirl.shared.store.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 美甲师职业生涯接口
 */
@Api(tags = "美甲师职业生涯接口", value = "美甲师职业生涯接口")
@RestController
@AllArgsConstructor
public class CareerController {


    private final TechnicianScoreMapper technicianScoreMapper;
    private final TrainingStartClassLogTechnicianMapper trainingStartClassLogTechnicianMapper;
    private final SkillSkuMapper skillSkuMapper;
    private final TrainingStartClassMapper trainingStartClassMapper;


    private final CareerCmdService careerCmdService;
    private final OrderServeMapper orderServeMapper;
    private final CustomerBalanceRecordMapper customerBalanceRecordMapper;

    @ApiOperation("提交美甲作品")
    @PostMapping("/technician/career/submitWorks")
    public JsonResult<Void> submitWorks(@RequestHeader String token,
                                        @RequestBody @Valid SubmitWorksCommand cmd) {
        Integer technicianId = SessionUtils.getTechnicianId();
        WorkLock workLock = WorkLock.getInstance();
        try {
            workLock.lock(technicianId);
            cmd.setTechnicianId(technicianId);
            careerCmdService.submitWorks(cmd);
        } finally {
            workLock.unlock(technicianId);

        }
        return JsonResult.success();
    }


    @ApiOperation("获取美甲师需要打板的款式")
    @GetMapping("/technician/career/queryWorkTpl")
    public JsonResult<Page<BaseProduce>> queryWorkTpl(@RequestHeader String token,
                                                      @RequestParam Integer pageNum,
                                                      @RequestParam Integer pageSize,
                                                      @RequestParam(required = false) String searchKeyWord) {
        Integer technicianId = SessionUtils.getTechnicianId();
        Page<BaseProduce> baseProduceList = careerCmdService.queryWorkTpl(technicianId, pageNum, pageSize, searchKeyWord);
        return JsonResult.success(baseProduceList);
    }

    @ApiOperation(("查询美甲师作品"))
    @PostMapping("/technician/career/queryWorks")
    public JsonResult<Page<Works>> queryWorks(@RequestHeader String token,
                                              @RequestBody WorksQuery qry) {
        Integer technicianId = SessionUtils.getTechnicianId();
        qry.setTechnicianId(technicianId);
        IPage<Works> worksIPage = careerCmdService.queryWorks(qry);
        return JsonResult.success((Page<Works>) worksIPage);
    }

    private final WorksAssembler worksAssembler;

    @ApiOperation("报名技能培训")
    @PostMapping("/technician/career/signUpTraining")
    public JsonResult<Void> signUpTraining(@RequestBody SignUpTrainingCommand cmd) {
        careerCmdService.signUpTraining(cmd);
        return JsonResult.success();
    }

    @ApiOperation("填写技能培训反馈")
    @PostMapping("/technician/career/submitFeedBack")
    public JsonResult<Void> submitFeedBack(@RequestBody SubmitFeedbackCommand cmd) {
        careerCmdService.submitFeedBack(cmd);
        return JsonResult.success();
    }

    private final TrainingStartClassDTOAssembler trainingStartClassDTOAssembler;

    @ApiOperation("课程列表")
    @PostMapping("/technician/career/queryPageClass")
    public JsonResult<Page<TrainingStartClassDTO>> queryPageClass(@RequestBody TrainingStartClassQuery qry) {
        IPage<TrainingStartClass> page = careerCmdService.queryPageClass(qry);
        return JsonResult.success((Page<TrainingStartClassDTO>) page.convert(trainingStartClassDTOAssembler));
    }

    @ApiOperation("培训记录")
    @GetMapping("/technician/career/queryPageStartClassLog")
    public JsonResult<IPage<TrainingStartClassLogTechnician>> queryPageStartClassLog(@RequestParam Integer pageNum,
                                                                                     @RequestParam Integer pageSize,
                                                                                     @RequestHeader String token) {


        Integer technicianId = SessionUtils.getTechnicianId();
        IPage<TrainingStartClassLogTechnician> pageRequest = new Page<>(pageNum, pageSize);
        IPage<TrainingStartClassLogTechnician> page = trainingStartClassLogTechnicianMapper.selectPage(pageRequest, new LambdaQueryWrapper<TrainingStartClassLogTechnician>().eq(TrainingStartClassLogTechnician::getTechnicianId, technicianId));

        List<Integer> ids = page.getRecords().stream().map(TrainingStartClassLogTechnician::getTrainingStartClassId).collect(Collectors.toList());
        if (ListUtil.isNotEmpty(ids)) {
            List<TrainingStartClass> trainingStartClassesList = trainingStartClassMapper.selectBatchIds(ids);
            page.getRecords().forEach(trainingStartClassLogTechnician -> {
                trainingStartClassLogTechnician.setTrainingStartClass(
                        trainingStartClassesList.stream().filter(trainingStartClass -> trainingStartClass.getId().equals(trainingStartClassLogTechnician.getTrainingStartClassId())).findAny().orElse(null));
            });
        }
        return JsonResult.success(page);
    }

    @ApiOperation("成长历程")
    @GetMapping("/technician/career/career")
    public JsonResult<Career> career(@RequestHeader String token) {
        Career career = careerCmdService.career(SessionUtils.getTechnicianId());
        return JsonResult.success(career);
    }

    private final StoreDataService storeDataService;

    @ApiOperation("门店数据")
    @GetMapping("/technician/career/storeData")
    public JsonResult<StoreData> storeData(@RequestHeader String token,
                                           @RequestParam String month) {
        StoreData storeData = storeDataService.storeData(month);
        return JsonResult.success(storeData);
    }

    @ApiOperation("分页查询订单业绩")
    @GetMapping("/technician/career/queryPageAchieve")
    public JsonResult<Page<OrderServe>> queryPageAchieve(@RequestHeader String token,
                                                         @RequestParam String type,
                                                         @RequestParam Integer pageNum,
                                                         @RequestParam Integer pageSize,
                                                         @RequestParam(required = false) String month) {

        Page<OrderServe> page = new Page<>(pageNum, pageSize);
        Integer technicianId = SessionUtils.getTechnicianId();
        Page<OrderServe> result = orderServeMapper.queryPageAchieve(page, technicianId, type, month);
        return JsonResult.success(result);
    }

    @ApiOperation("分页查询订单业绩")
    @GetMapping("/technician/career/queryPageTechnicianAchieve")
    public JsonResult<Page<TechnicianAchieve>> queryPageTechnicianAchieve(@RequestHeader String token,
                                                                          @RequestParam String type,
                                                                          @RequestParam Integer pageNum,
                                                                          @RequestParam Integer pageSize,
                                                                          @RequestParam(required = false) String month,
                                                                          @RequestParam(required = false) Integer technicianId) {

        Page<OrderServe> page = new Page<>(pageNum, pageSize);

        Integer departmentId = SessionUtils.getTechnicianToken().getDepartmentId();
        if (type.equals("开卡")) {
            Page<TechnicianAchieve> result = orderServeMapper.queryPageChargeTechnicianAchieve(page, technicianId, month, departmentId);
            return JsonResult.success(result);
        } else {
            Page<TechnicianAchieve> result = orderServeMapper.queryPageOrderTechnicianAchieve(page, technicianId, type, month, departmentId);
            return JsonResult.success(result);
        }
    }

    @ApiOperation("分页查询开卡业绩")
    @GetMapping("/technician/career/queryPageChargeAchieve")
    public JsonResult<Page<CustomerBalanceRecord>> queryPageChargeAchieve(@RequestHeader String token,
                                                                          @RequestParam Integer pageNum,
                                                                          @RequestParam Integer pageSize,
                                                                          @RequestParam(required = false) String month) {

        Page<CustomerBalanceRecord> page = new Page<>(pageNum, pageSize);
        Integer technicianId = SessionUtils.getTechnicianId();
        Page<CustomerBalanceRecord> result = customerBalanceRecordMapper.queryPageAchieve(page, technicianId, month);
        return JsonResult.success(result);
    }

    @ApiOperation("查询美甲师评分")
    @GetMapping("/technician/career/queryPageTechScore")
    public JsonResult<List<TechnicianScore>> queryPageTechScore(@RequestHeader String token) {

        Integer technicianId = SessionUtils.getTechnicianId();

        List<TechnicianScore> list = technicianScoreMapper.selectList(
                new LambdaQueryWrapper<TechnicianScore>().eq(TechnicianScore::getTechnicianId, technicianId)
        );
        if (ListUtil.isNotEmpty(list)) {
            List<SkillSku> skillSkuList = skillSkuMapper.selectBatchIds(list.stream().map(TechnicianScore::getSkillSkuId).collect(Collectors.toList()));
            list.forEach(technicianScore -> technicianScore.setSkillSku(skillSkuList.stream().filter(skillSku -> skillSku.getId().equals(technicianScore.getSkillSkuId())).findAny().orElse(null)));
        }
        return JsonResult.success(list);
    }

}
