package com.gogirl.infrastructure.util;

import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.exception.RRException;

/**
 * 应用会话管理工具类
 */
public class SessionUtils {

    /*美甲师会话线程变量*/
    private static final ThreadLocal<GogirlToken> technicianTokenThreadLocal = new ThreadLocal<>();
    /*用户会员线程变量*/
    private static final ThreadLocal<GogirlToken> customerTokenThreadLocal = new ThreadLocal<>();
    /*请求来源线程变量*/
    private static final ThreadLocal<String> sourceFromThreadLocal = new ThreadLocal<>();

    /**
     * 保存美甲师登录信息到本地线程变量
     *
     * @param gogirlToken 美甲师登录信息
     */
    public static void putTechnicianToken(GogirlToken gogirlToken) {
        technicianTokenThreadLocal.set(gogirlToken);
    }

    /**
     * 获取美甲师登录信息
     *
     * @return 美甲师登录信息
     */
    public static GogirlToken getTechnicianToken() {
        return technicianTokenThreadLocal.get();
    }

    /**
     * 清除美甲师登录信息
     */
    public static void removeTechnicianToken() {
        technicianTokenThreadLocal.remove();
    }

    /**
     * 保存用户登录信息到本地线程变量
     *
     * @param gogirlToken 用户登录信息
     */
    public static void putCustomerToken(GogirlToken gogirlToken) {
        customerTokenThreadLocal.set(gogirlToken);
    }

    /**
     * 获取用户登录信息
     *
     * @return 用户登录信息
     */
    public static GogirlToken getCustomerToken() {
        return customerTokenThreadLocal.get();
    }

    /**
     * 清除用户登录信息
     */
    public static void removeCustomerToken() {
        customerTokenThreadLocal.remove();
    }


    /**
     * 获取当前登录美甲师id
     *
     * @return 美甲师id
     */
    public static Integer getTechnicianId() {
        GogirlToken gogirlToken = technicianTokenThreadLocal.get();
        if (gogirlToken == null) {
            throw new RRException(2000, "token已经失效，请重新登陆");
        }
        return gogirlToken.getTechnicianId();
    }

    /**
     * 获取当前登录用户id
     *
     * @return 客户id
     */
    public static Integer getCustomerId() {
        GogirlToken gogirlToken = customerTokenThreadLocal.get();
        if (gogirlToken == null) {
            throw new RRException(2000, "token已经失效，请重新登陆");
        }
        return gogirlToken.getCustomerId();
    }

    /**
     * 设置当前请求来源到线程变量
     *
     * @param sourceFrom 来源：客户：customer ；美甲师:technician ；H5：h5。
     */
    public static void putSourceFrom(String sourceFrom) {
        sourceFromThreadLocal.set(sourceFrom);
    }

    /**
     * 获取当前请求来源
     *
     * @return 请求来源
     */
    public static String getSourceFrom() {
        return sourceFromThreadLocal.get();
    }

    /**
     * 清空请求来源
     */
    public static void removeSourceFrom() {
        sourceFromThreadLocal.remove();
    }

    /**
     * 当前请求是否来自客户
     *
     * @return bool
     */
    public static boolean isSourceFromCustomer() {
        return sourceFromThreadLocal.get().equals("customer");
    }

    /**
     * 当前请求是否来自美甲师
     *
     * @return bool
     */
    public static boolean isSourceFromTechnician() {
        return sourceFromThreadLocal.get().equals("technician");
    }
}
