package com.gogirl.interfaces.store.shop;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.StoreManageService;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.util.MapDistanceUtils;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@RestController
@Api(tags = {"店铺接口"})
@AllArgsConstructor
@Slf4j
public class ShopManageController {


    private final StoreManageService storeManageService;


    @ApiOperation(value = "获取美甲师绑定的店铺")
    @GetMapping("/technician/xcx/getTechnicianShopList")
    public JsonResult<List<StoreManage>> getTechnicianShopList(@RequestHeader String token) {
        List<StoreManage> storeManageList = storeManageService.getTechnicianShopList();
        return JsonResult.success(storeManageList);
    }


    @ApiOperation(value = "查询店铺列表，longitude,latitude和页码都是可选参数")
    @GetMapping("/customer/shop/queryShopForPage")
    public JsonResult<IPage<StoreManage>> queryShopForPage(@RequestParam(required = false) String longitude,
                                                           @RequestParam(required = false) String latitude,
                                                           @RequestParam(required = false, defaultValue = "1") Integer pageNum,
                                                           @RequestParam(required = false, defaultValue = "20") Integer pageSize) {
        IPage<StoreManage> page = new Page<>(pageNum, pageSize);

        StoreManage pageReq = new StoreManage();
        pageReq.setLongitude(longitude);
        pageReq.setLatitude(latitude);
        page = storeManageService.listShopManageForPage(page, pageReq);

        if (StringUtils.isNotBlank(longitude) &&
                StringUtils.isNotBlank(latitude) &&
                !longitude.equals("undefined") && !latitude.equals("undefined")) {

            for (StoreManage storeManage : page.getRecords()) {
                Double distance = MapDistanceUtils.getDistance(longitude, latitude, storeManage.getLongitude(), storeManage.getLatitude());
                storeManage.setDistance(distance);
            }
            page.getRecords().sort(Comparator.comparing(StoreManage::getDistance));
        }
        return JsonResult.success(page);
    }

    @ApiOperation(value = "id查店铺详情")
    @GetMapping("/customer/shop/queryShopForDetail")
    public JsonResult<StoreManage> queryShopForDetail(@RequestParam(required = false) String longitude,
                                                      @RequestParam(required = false) String latitude,
                                                      @RequestParam(required = false) Integer id) {
        StoreManage storeManage = storeManageService.getShopManageForDetail(id, longitude, latitude);


        return JsonResult.success(storeManage);
    }

    @ApiOperation(value = "查询店照，不传departmentId则查全部店铺的店照")
    @GetMapping("/customer/shop/queryShopEnvironmentPhotos")
    public JsonResult<List<String>> queryShopEnvironmentPhotos(@RequestParam(required = false) Integer departmentId,
                                                               @RequestParam(required = false) String token,
                                                               @RequestParam(required = false) String formId) {

        StoreManage storeManage = new StoreManage();
        storeManage.setId(departmentId);
        List<StoreManage> lists = storeManageService.listShopManageForPage(new Page<>(1, 10), storeManage).getRecords();

        List<String> resultList = lists.stream()
                .filter(Objects::nonNull)
                .map(StoreManage::getShopEnvironmentPhotos)
                .filter(Objects::nonNull)
                .map(phones -> Lists.newArrayList(phones.split(",")))
                .flatMap(List::stream)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        return JsonResult.success(resultList);
    }

}
